﻿Imports System
Imports System.Text
Imports System.Runtime.InteropServices
Imports System.Threading

Imports Peak.Can.Basic
Imports Peak.Can.IsoTp
Imports Peak.Can.Uds


Module Module1

    Const BUFFER_SIZE As Integer = 256
    Const ISOTP_PING_MSG As String = "PING"
    Const ISOTP_PING_MSG_SIZE As UInt32 = 4
    Const CAN_ID_START_RANGE As UInt32 = 1
    Const CAN_ID_STOP_RANGE As UInt32 = 10
    Const PING_TIMING_MS As Integer = 500


    Function OK_KO(test As Boolean) As String
        If test Then Return "OK"
        Return "KO"
    End Function

    Function UDS_STATUS_OK_KO(test As uds_status) As String
        Return OK_KO(UDSApi.StatusIsOk_2013(test))
    End Function

    Function ISOTP_STATUS_OK_KO(test As cantp_status) As String
        Return OK_KO(CanTpApi.StatusIsOk_2016(test))
    End Function



    ''' <summary>Structure passed as thread parameters</summary>
    Public Structure task_params
        ''' <summary>Server channel handle</summary>
        Public server_handle As cantp_handle
        ''' <summary>Server address</summary>
        Public server_address As UInt32
        ''' <summary>Determine if the thread should end or not</summary>
        Public stop_task As Boolean
    End Structure

    ''' <summary>ISOTP server task: periodically send "PING" message using different CAN identifiers</summary>
    ''' <param name="parameters">pointer on task_params structures</param>
    Sub isotp_server_task(ByRef parameters As task_params)
        Dim status As cantp_status
        Dim tx_msg As cantp_msg
        Dim can_id As UInt32

        ' Init variables
        tx_msg = New cantp_msg()
        can_id = CAN_ID_START_RANGE

        ' Send loop
        Do
            ' Wait before sending the next message
            Thread.Sleep(PING_TIMING_MS)

            ' Initialize ISOTP Tx message containing "PING"
            status = CanTpApi.MsgDataAlloc_2016(tx_msg, cantp_msgtype.PCANTP_MSGTYPE_CAN)
            Console.WriteLine("[ISOTP] Allocate ISOTP tx message: {0}", ISOTP_STATUS_OK_KO(status))
            Dim bytes_array As Byte() = Encoding.ASCII.GetBytes(ISOTP_PING_MSG)
            status = CanTpApi.MsgDataInit_2016(tx_msg, can_id, cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD, ISOTP_PING_MSG_SIZE, bytes_array)
            Console.WriteLine("[ISOTP] Initialize ISOTP tx message: {0}", ISOTP_STATUS_OK_KO(status))

            ' Send "PING" message
            status = CanTpApi.Write_2016(parameters.server_handle, tx_msg)
            Console.WriteLine("[ISOTP] Send ISOTP ""PING"" message (can id=0x{0:x}): {1}", can_id, ISOTP_STATUS_OK_KO(status))

            ' Free messages
            status = CanTpApi.MsgDataFree_2016(tx_msg)
            Console.WriteLine("[ISOTP] Free ISOTP TX message: {0}", ISOTP_STATUS_OK_KO(status))

            ' Update can id for next message
            can_id += 1
            If (can_id > CAN_ID_STOP_RANGE) Then
                can_id = CAN_ID_START_RANGE
            End If
        Loop While parameters.stop_task = False
    End Sub

    ''' <summary>UDS server task: respond TesterPresent request</summary>
    ''' <param name="parameters">pointer on task_params structures</param>
    Sub uds_server_task(ByRef parameters As task_params)
        Dim config_physical As uds_msgconfig
        Dim wait_result As Boolean
        Dim status As uds_status
        Dim read_status As uds_status
        Dim request_msg As uds_msg
        Dim response_msg As uds_msg

        ' Init variables
        config_physical = New uds_msgconfig()
        request_msg = New uds_msg()
        response_msg = New uds_msg()

        ' Set server address parameter
        status = UDSApi.SetValue_2013(parameters.server_handle, uds_parameter.PUDS_PARAMETER_SERVER_ADDRESS, parameters.server_address, CType(Marshal.SizeOf(parameters.server_address), UInt32))
        Console.WriteLine("[UDS] Set UDS server address: {0}", UDS_STATUS_OK_KO(status))

        ' Set a receive event for UDS
        Dim receive_event As System.Threading.AutoResetEvent = New System.Threading.AutoResetEvent(False)
        If IntPtr.Size = 4 Then
            Dim tmp_buffer As UInt32 = Convert.ToUInt32(receive_event.SafeWaitHandle.DangerousGetHandle().ToInt32())
            status = UDSApi.SetValue_2013(parameters.server_handle, uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, tmp_buffer, CType(Marshal.SizeOf(tmp_buffer), UInt32))
        ElseIf IntPtr.Size = 8 Then
            Dim tmp_buffer As Int64 = receive_event.SafeWaitHandle.DangerousGetHandle().ToInt64()
            Dim byte_array As Byte() = BitConverter.GetBytes(tmp_buffer)
            status = UDSApi.SetValue_2013(parameters.server_handle, uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, byte_array, CType(Marshal.SizeOf(tmp_buffer), UInt32))
        End If
        Console.WriteLine("[UDS] Set UDS receive event parameter: {0}", UDS_STATUS_OK_KO(status))

        ' Initialize a physical configuration
        config_physical.can_id = &HFFFFFFFFUI
        config_physical.can_msgtype = cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD
        config_physical.nai.protocol = uds_msgprotocol.PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL
        config_physical.nai.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL
        config_physical.type = uds_msgtype.PUDS_MSGTYPE_USDT
        config_physical.nai.extension_addr = 0

        Do

            ' Wait a receive event on receiver
            wait_result = receive_event.WaitOne(1000)

            ' If we get a receive event
            If (wait_result) Then
                Do

                    ' Read first available message (no filtering based on message's type is set):
                    read_status = UDSApi.Read_2013(parameters.server_handle, request_msg)
                    Console.WriteLine("[UDS] Try to read a message: {0}", UDS_STATUS_OK_KO(read_status))
                    If (UDSApi.StatusIsOk_2013(read_status)) Then

                        ' We receive a request, check its length and if it is not a loopback message and if it is a USDT message
                        If (request_msg.type = uds_msgtype.PUDS_MSGTYPE_USDT And request_msg.msg.Msgdata_any_Copy.length >= 1 And (request_msg.msg.Msgdata_any_Copy.flags And cantp_msgflag.PCANTP_MSGFLAG_LOOPBACK) = 0) Then

                            ' This is a valid request, switch services
                            Dim service_id As Byte
                            Dim get_sid As Boolean = UDSApi.GetDataServiceId_2013(request_msg, service_id)
                            If (get_sid) Then
                                Select Case CType(service_id, uds_service)
                                    Case uds_service.PUDS_SERVICE_SI_TesterPresent
                                        ' Allocate response message
                                        status = UDSApi.MsgAlloc_2013(response_msg, config_physical, 2)
                                        Console.WriteLine("[UDS] Prepare response message for TesterPresent service: {0}", UDS_STATUS_OK_KO(status))

                                        If (UDSApi.StatusIsOk_2013(status)) Then
                                            ' Fill parameters
                                            Dim nadr As cantp_netaddrinfo = request_msg.msg.Msgdata_isotp_Copy.netaddrinfo
                                            nadr.target_addr = nadr.source_addr
                                            nadr.source_addr = parameters.server_address
                                            CanTpApi.setNetaddrinfo_2016(response_msg.msg, nadr)
                                            UDSApi.SetDataServiceId_2013(response_msg, uds_service.PUDS_SERVICE_SI_TesterPresent + UDSApi.PUDS_SI_POSITIVE_RESPONSE)
                                            UDSApi.SetDataParameter_2013(response_msg, 0, 0)

                                            ' Write response message
                                            status = UDSApi.Write_2013(parameters.server_handle, response_msg)
                                            Console.WriteLine("[UDS] Write response message for TesterPresent service: {0}", UDS_STATUS_OK_KO(status))
                                        End If

                                        ' Free response message (and clean memory in order to reallocate later)
                                        status = UDSApi.MsgFree_2013(response_msg)
                                        Console.WriteLine("[UDS] Free response message: {0}", UDS_STATUS_OK_KO(status))
                                    Case Else
                                        Console.WriteLine("[UDS] Unknown service (0x{0:x2})", service_id)
                                End Select
                            End If
                        End If
                    End If

                    ' Free request message (and clean memory in order to reallocate later)
                    status = UDSApi.MsgFree_2013(request_msg)
                    Console.WriteLine("[UDS] Free request message: {0}", UDS_STATUS_OK_KO(status))
                Loop While Not UDSApi.StatusIsOk_2013(read_status, uds_status.PUDS_STATUS_NO_MESSAGE, False)
            End If
        Loop While parameters.stop_task = False

        ' Close receive event
        If IntPtr.Size = 4 Then
            Dim tmp_buffer As UInt32 = 0
            status = UDSApi.SetValue_2013(parameters.server_handle, uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, tmp_buffer, CType(Marshal.SizeOf(tmp_buffer), UInt32))
        ElseIf IntPtr.Size = 8 Then
            Dim tmp_buffer As Int64 = 0
            Dim byte_array As Byte() = BitConverter.GetBytes(tmp_buffer)
            status = UDSApi.SetValue_2013(parameters.server_handle, uds_parameter.PUDS_PARAMETER_RECEIVE_EVENT, byte_array, CType(Marshal.SizeOf(tmp_buffer), UInt32))
        End If
        Console.WriteLine("[UDS] Stop UDS receive event: {0}", UDS_STATUS_OK_KO(status))
        receive_event.Close()
        Console.WriteLine("[UDS] Close UDS receive event: {0}", UDS_STATUS_OK_KO(status))
    End Sub

    ''' <summary>Entry point of the program, start a small server which handle UDS testerpresent request and periodically send isotp messages</summary>
    ''' <returns>By convention, return success.</returns>
    Sub Main(ByVal args As String())
        Dim status As uds_status
        Dim t_params As task_params
        Dim buffer As StringBuilder = New StringBuilder(BUFFER_SIZE)
        Dim uds_server As Thread
        Dim isotp_server As Thread

        ' Initialize variables
        t_params.server_handle = cantp_handle.PCANTP_HANDLE_USBBUS2 ' TODO: modify the value according to your available PCAN devices.
        t_params.server_address = uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_1
        t_params.stop_task = False

        ' Print version informations
        status = UDSApi.GetValue_2013(cantp_handle.PCANTP_HANDLE_NONEBUS, uds_parameter.PUDS_PARAMETER_API_VERSION, buffer, BUFFER_SIZE)
        Console.WriteLine("PCAN-UDS API Version - {0}: {1}", buffer, UDS_STATUS_OK_KO(status))

        ' Initialize server
        status = UDSApi.Initialize_2013(t_params.server_handle, cantp_baudrate.PCANTP_BAUDRATE_500K)
        Console.WriteLine("Initialize channel: {0}", UDS_STATUS_OK_KO(status))

        ' Start uds and isotp servers
        uds_server = New Thread(Sub() uds_server_task(t_params))
        uds_server.Start()
        isotp_server = New Thread(Sub() isotp_server_task(t_params))
        isotp_server.Start()

        ' Read while user do not press Q
        Console.WriteLine("Start listening, press Q to quit.")
        t_params.stop_task = False
        Do
            ' Quit when user press Q
            If Console.KeyAvailable Then
                Dim keyboard_res As Char = Console.ReadKey().KeyChar

                If keyboard_res = "Q"c OrElse keyboard_res = "q"c Then
                    t_params.stop_task = True
                End If
            End If
        Loop While t_params.stop_task = False

        ' Close threads
        isotp_server.Join()
        uds_server.Join()

        ' Close channel
        status = UDSApi.Uninitialize_2013(t_params.server_handle)
        Console.WriteLine("Uninitialize channel: {0}", UDS_STATUS_OK_KO(status))

        ' Exit
        Console.WriteLine("Press any key to continue...")
        Console.In.Read()
    End Sub
End Module
