﻿Imports System
Imports System.Runtime.InteropServices
Imports System.Threading
Imports System.Text

Imports Peak.Can.Basic
Imports Peak.Can.IsoTp
Imports Peak.Can.Uds


Module Module1

    Const BUFFER_SIZE As Integer = 256
    Const UDS_REQUEST_TIMING_MS As Integer = 500
    Const CLIENT_EXECUTION_TIME_MS As Integer = 10000

    Function OK_KO(test As Boolean) As String
        If test Then Return "OK"
        Return "KO"
    End Function

    Function UDS_STATUS_OK_KO(test As uds_status) As String
        Return OK_KO(UDSApi.StatusIsOk_2013(test))
    End Function

    Function ISOTP_STATUS_OK_KO(test As cantp_status) As String
        Return OK_KO(CanTpApi.StatusIsOk_2016(test))
    End Function

    ''' <summary>Structure passed as thread parameters</summary>
    Public Structure task_params
        ''' <summary>Client channel handle</summary>
        Public client_handle As cantp_handle
        ''' <summary>Determine if the thread should end or not</summary>
        Public stop_task As Boolean
    End Structure

    ''' <summary>UDS client task: request TesterPresent service several time</summary>
    ''' <param name="parameters">pointer on task_params structures</param>
    Sub uds_client_task(ByRef parameters As task_params)
        Dim status As uds_status
        Dim config_physical As uds_msgconfig
        Dim msg_request As uds_msg
        Dim request_confirmation As uds_msg
        Dim response As uds_msg

        ' Initialize variables
        config_physical = New uds_msgconfig()
        msg_request = New uds_msg()
        request_confirmation = New uds_msg()
        response = New uds_msg()

        ' Initialize a physical configuration
        config_physical.can_id = uds_can_id.PUDS_CAN_ID_ISO_15765_4_PHYSICAL_REQUEST_1
        config_physical.can_msgtype = cantp_can_msgtype.PCANTP_CAN_MSGTYPE_STANDARD
        config_physical.nai.protocol = uds_msgprotocol.PUDS_MSGPROTOCOL_ISO_15765_2_11B_NORMAL
        config_physical.nai.target_type = cantp_isotp_addressing.PCANTP_ISOTP_ADDRESSING_PHYSICAL
        config_physical.type = uds_msgtype.PUDS_MSGTYPE_USDT
        config_physical.nai.source_addr = uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_TEST_EQUIPMENT
        config_physical.nai.target_addr = uds_address.PUDS_ADDRESS_ISO_15765_4_ADDR_ECU_1
        config_physical.nai.extension_addr = 0

        ' Execute TesterPresent and wait response
        Do
            ' Wait before transmitting the next message
            Thread.Sleep(UDS_REQUEST_TIMING_MS)

            status = UDSApi.SvcTesterPresent_2013(parameters.client_handle, config_physical, msg_request, UDSApi.uds_svc_param_tp.PUDS_SVC_PARAM_TP_ZSUBF)
            Console.WriteLine("[UDS] Execute TesterPresent service: {0}", UDS_STATUS_OK_KO(status))
            status = UDSApi.WaitForService_2013(parameters.client_handle, msg_request, response, request_confirmation)
            Console.WriteLine("[UDS] Wait for service: {0}", UDS_STATUS_OK_KO(status))
            status = UDSApi.MsgFree_2013(msg_request)
            Console.WriteLine("[UDS] Free request message: {0}", UDS_STATUS_OK_KO(status))
            status = UDSApi.MsgFree_2013(request_confirmation)
            Console.WriteLine("[UDS] Free request confirmation message: {0}", UDS_STATUS_OK_KO(status))
            status = UDSApi.MsgFree_2013(response)
            Console.WriteLine("[UDS] Free response message: {0}", UDS_STATUS_OK_KO(status))
        Loop While parameters.stop_task = False
    End Sub

    ''' <summary>ISOTP client task: read and print isotp messages</summary>
    ''' <param name="parameters">pointer on task_params structures</param>
    Sub isotp_client_task(ByRef parameters As task_params)
        ' Initialize variables
        Dim status As cantp_status
        Dim rx_msg As cantp_msg = New cantp_msg()
        Dim wait_result As Boolean
        Dim isotp_param As UInt32

        ' Configure isotp to get any messages (disable can identifier filtering)
        isotp_param = 0
        status = CanTpApi.SetValue_2016(parameters.client_handle, cantp_parameter.PCANTP_PARAMETER_FILTER_CAN_ID, isotp_param, CType(Marshal.SizeOf(isotp_param), UInt32))
        Console.WriteLine("[ISOTP] Disable can identifier filtering in isotp: {0}", ISOTP_STATUS_OK_KO(status))

        ' Configure isotp to get a copy of UDS messages
        isotp_param = 1
        status = CanTpApi.SetValue_2016(parameters.client_handle, cantp_parameter.PCANTP_PARAMETER_KEEP_HIGHER_LAYER_MESSAGES, isotp_param, CType(Marshal.SizeOf(isotp_param), UInt32))
        Console.WriteLine("[ISOTP] Activate higher layer messages in isotp: {0}", ISOTP_STATUS_OK_KO(status))

        ' Create a isotp receive event
        Dim receive_event As System.Threading.AutoResetEvent = New System.Threading.AutoResetEvent(False)
        If IntPtr.Size = 4 Then
            Dim tmp_buffer As UInt32 = Convert.ToUInt32(receive_event.SafeWaitHandle.DangerousGetHandle().ToInt32())
            status = CanTpApi.SetValue_2016(parameters.client_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, tmp_buffer, CType(Marshal.SizeOf(tmp_buffer), UInt32))
        ElseIf IntPtr.Size = 8 Then
            Dim tmp_buffer As Int64 = receive_event.SafeWaitHandle.DangerousGetHandle().ToInt64()
            Dim byte_array As Byte() = BitConverter.GetBytes(tmp_buffer)
            status = CanTpApi.SetValue_2016(parameters.client_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, byte_array, CType(Marshal.SizeOf(tmp_buffer), UInt32))
        End If
        Console.WriteLine("[ISOTP] Set isotp receive event parameter: {0}", ISOTP_STATUS_OK_KO(status))

        ' Read and print ISOTP messages
        Do
            wait_result = receive_event.WaitOne(1000)
            Console.WriteLine("[ISOTP] Wait a receive event from isotp: {0}", OK_KO(wait_result))

            ' Read ISOTP messages
            Do
                status = CanTpApi.Read_2016(parameters.client_handle, rx_msg)

                ' Check if we received a message
                If Not CanTpApi.StatusIsOk_2016(status, cantp_status.PCANTP_STATUS_NO_MESSAGE, False) Then
                    Console.WriteLine("[ISOTP] Read ISOTP message: {0}", ISOTP_STATUS_OK_KO(status))
                    Dim data_0 As Byte
                    Dim get_data_0 = CanTpApi.getData_2016(rx_msg, 0, data_0)
                    If (get_data_0 And (data_0 = uds_service.PUDS_SERVICE_SI_TesterPresent Or data_0 = uds_service.PUDS_SERVICE_SI_TesterPresent + UDSApi.PUDS_SI_POSITIVE_RESPONSE)) Then
                        ' This message is a TesterPresent message, PCANTP_PARAMETER_KEEP_HIGHER_LAYER_MESSAGES
                        ' option must be activated to get these messages.
                        Console.WriteLine("[ISOTP] Message is a TesterPresent service message.")
                    ElseIf CanTpApi.StatusIsOk_2016(status) And rx_msg.Msgdata_any_Copy.length < BUFFER_SIZE Then
                        ' This message is a CAN message received by ISOTP
                        Dim msg_data As StringBuilder = New StringBuilder()
                        For i = 0 To rx_msg.Msgdata_any_Copy.length - 1
                            Dim data As Byte
                            If (CanTpApi.getData_2016(rx_msg, i, data)) Then
                                msg_data.Append(Convert.ToChar(data))
                            End If
                        Next
                        Console.WriteLine("[ISOTP] Message from 0x{0:x} can identifier contains ""{1}""", rx_msg.can_info.can_id, msg_data)
                    End If

                    status = CanTpApi.MsgDataFree_2016(rx_msg)
                    Console.WriteLine("[ISOTP] Free RX message: {0}", ISOTP_STATUS_OK_KO(status))
                End If
            Loop While Not CanTpApi.StatusIsOk_2016(status, cantp_status.PCANTP_STATUS_NO_MESSAGE, False)
        Loop While parameters.stop_task = False

        ' Close receive event
        If IntPtr.Size = 4 Then
            Dim tmp_buffer As UInt32 = 0
            status = CanTpApi.SetValue_2016(parameters.client_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, tmp_buffer, CType(Marshal.SizeOf(tmp_buffer), UInt32))
        ElseIf IntPtr.Size = 8 Then
            Dim tmp_buffer As Int64 = 0
            Dim byte_array As Byte() = BitConverter.GetBytes(tmp_buffer)
            status = CanTpApi.SetValue_2016(parameters.client_handle, cantp_parameter.PCANTP_PARAMETER_RECEIVE_EVENT, byte_array, CType(Marshal.SizeOf(tmp_buffer), UInt32))
        End If
        Console.WriteLine("[ISOTP] Stop ISOTP receive event: {0}", ISOTP_STATUS_OK_KO(status))
        receive_event.Close()
        Console.WriteLine("[ISOTP] Close ISOTP receive event: {0}", ISOTP_STATUS_OK_KO(status))
    End Sub

    ''' <summary>Entry point of the program, start a UDS channel, ask in the same time UDS TesterPresent and isotp request</summary>
    ''' <returns>By convention, return success.</returns>
    Sub Main(ByVal args As String())
        Dim buffer As StringBuilder = New StringBuilder(BUFFER_SIZE)
        Dim status As uds_status
        Dim client_handle As cantp_handle
        Dim uds_client As Thread
        Dim isotp_client As Thread
        Dim t_params As task_params

        ' Initialize variables
        client_handle = cantp_handle.PCANTP_HANDLE_USBBUS1 ' TODO: modify the value according to your available PCAN devices.
        t_params.client_handle = client_handle
        t_params.stop_task = False

        ' Print version informations
        status = UDSApi.GetValue_2013(cantp_handle.PCANTP_HANDLE_NONEBUS, uds_parameter.PUDS_PARAMETER_API_VERSION, buffer, BUFFER_SIZE)
        Console.WriteLine("PCAN-UDS API Version - {0}: {1}", buffer, UDS_STATUS_OK_KO(status))

        ' Initialize client channel
        status = UDSApi.Initialize_2013(client_handle, cantp_baudrate.PCANTP_BAUDRATE_500K)
        Console.WriteLine("Initialize channel: {0}", UDS_STATUS_OK_KO(status))

        ' Start uds and isotp clients
        uds_client = New Thread(Sub() uds_client_task(t_params))
        uds_client.Start()

        isotp_client = New Thread(Sub() isotp_client_task(t_params))
        isotp_client.Start()

        Thread.Sleep(CLIENT_EXECUTION_TIME_MS)
        t_params.stop_task = True

        isotp_client.Join()
        uds_client.Join()

        ' Close channel
        status = UDSApi.Uninitialize_2013(client_handle)
        Console.WriteLine("Uninitialize channel: {0}", UDS_STATUS_OK_KO(status))

        ' Exit
        Console.WriteLine("Press any key to continue...")
        Console.In.Read()
    End Sub
End Module
