﻿Imports System.Text
Imports System.Runtime.InteropServices
Imports PcanUdsSampleVisualBasic.Peak.Can.Uds
Imports PcanUdsSampleVisualBasic.Peak.Can.Uds.UDSApi
Imports TPUDSCANHandle = System.UInt16

Public Class ApiUser
    Private _IsCommunicating As Boolean, _Response As TPUDSMsg, _FunctionalResponses As TPUDSMsg(), _ReceivedFunctionalResponses As UInteger, _RequestIsFunctional As Boolean, _UUDTResponse As TPUDSMsg

    <DllImport("PCAN-ISO-TP.dll", EntryPoint:="CANTP_SetValue")>
    Private Shared Function IsoTp_SetValue(
    <MarshalAs(UnmanagedType.U2)> ByVal CanChannel As TPUDSCANHandle,
    <MarshalAs(UnmanagedType.U1)> ByVal Parameter As Byte, ByRef NumericBuffer As UInteger, ByVal BufferLength As UInteger) As UInteger
    End Function

    Public ReadOnly Property IsConnected As Boolean
        Get
            Return mConnectedChannelHandle <> 0
        End Get
    End Property

    Public Property IsCommunicating As Boolean
        Get
            Return _IsCommunicating
        End Get
        Private Set(ByVal value As Boolean)
            _IsCommunicating = value
        End Set
    End Property

    Public ReadOnly Property LocalAddressIsExtended As Boolean
        Get
            Return mLocalNodeAddress > &HFF
        End Get
    End Property

    Public ReadOnly Property LocalNodeAddress As UInteger
        Get
            If LocalAddressIsExtended Then Return mLocalNodeAddress Xor PUDS_SERVER_ADDR_FLAG_ENHANCED_ISO_15765_3
            Return mLocalNodeAddress
        End Get
    End Property

    Public Property Response As TPUDSMsg
        Get
            Return _Response
        End Get
        Private Set(ByVal value As TPUDSMsg)
            _Response = value
        End Set
    End Property

    Public Property FunctionalResponses As TPUDSMsg()
        Get
            Return _FunctionalResponses
        End Get
        Private Set(ByVal value As TPUDSMsg())
            _FunctionalResponses = value
        End Set
    End Property

    Public Property ReceivedFunctionalResponses As UInteger
        Get
            Return _ReceivedFunctionalResponses
        End Get
        Private Set(ByVal value As UInteger)
            _ReceivedFunctionalResponses = value
        End Set
    End Property

    Public Property RequestIsFunctional As Boolean
        Get
            Return _RequestIsFunctional
        End Get
        Private Set(ByVal value As Boolean)
            _RequestIsFunctional = value
        End Set
    End Property

    Public Property UUDTResponse As TPUDSMsg
        Get
            Return _UUDTResponse
        End Get
        Private Set(ByVal value As TPUDSMsg)
            _UUDTResponse = value
        End Set
    End Property

    Public ReadOnly Property UUDTError As Boolean
        Get
            Return mUUDTStatus <> TPUDSStatus.PUDS_ERROR_OK AndAlso mUUDTStatus <> TPUDSStatus.PUDS_ERROR_NO_MESSAGE
        End Get
    End Property

    Private mConnectedChannelHandle As TPUDSCANHandle
    Private mLocalNodeAddress As UInteger
    Private mRequest As TPUDSMsg
    Private mServiceInformationOfLastRequest As ServiceInformation
    Private mResponseStatus As TPUDSStatus
    Private mUUDTStatus As TPUDSStatus

    Public Sub New()
        mConnectedChannelHandle = PUDS_NONEBUS
        mLocalNodeAddress = &HF1
        IsCommunicating = False
        FunctionalResponses = New TPUDSMsg(19) {}
    End Sub

    Public Function GetAvailableChannels() As List(Of TPUDSCANHandle)
        Dim status As TPUDSStatus
        Dim availableChannels = New List(Of TPUDSCANHandle)
        Dim condition As UInteger = Nothing

        For Each channel As TPUDSCANHandle In CType(AllChannelHandles, IEnumerable(Of UShort))
            If channel = PUDS_NONEBUS Then Continue For

            If channel <= PUDS_DNGBUS1 Then
                availableChannels.Add(channel)
            Else
                status = UDSApi.GetValue(channel, TPUDSParameter.PUDS_PARAM_CHANNEL_CONDITION, condition, 4)
                If status = TPUDSStatus.PUDS_ERROR_OK AndAlso (condition And PUDS_CHANNEL_AVAILABLE) = PUDS_CHANNEL_AVAILABLE Then availableChannels.Add(channel)
            End If
        Next

        Return availableChannels
    End Function

    Public Function Connect(ByVal canHandle As TPUDSCANHandle, ByVal baudrate As TPUDSBaudrate) As Boolean
        Dim status = Initialize(canHandle, baudrate)

        If status <> TPUDSStatus.PUDS_ERROR_OK Then
            ShowError(status)
            Return False
        Else
            mLocalNodeAddress = &HF1
        End If

        mConnectedChannelHandle = canHandle
        Return True
    End Function

    Public Function ConnectNonPlugAndPlay(ByVal canHandle As TPUDSCANHandle, ByVal baudrate As TPUDSBaudrate, ByVal hardwareType As TPUDSHWType, ByVal ioPort As UInteger, ByVal interrupt As TPUDSCANHandle) As Boolean
        Dim status = Initialize(canHandle, baudrate, hardwareType, ioPort, interrupt)

        If status <> TPUDSStatus.PUDS_ERROR_OK Then
            ShowError(status)
            Return False
        Else
            mLocalNodeAddress = &HF1
        End If

        mConnectedChannelHandle = canHandle
        Return True
    End Function

    Public Function Uninitialize() As Boolean
        If Not IsConnected Then Return False
        Dim status = UDSApi.Uninitialize(mConnectedChannelHandle)

        If status <> TPUDSStatus.PUDS_ERROR_OK Then
            ShowError(status)
            Return False
        End If

        mConnectedChannelHandle = PUDS_NONEBUS
        Return True
    End Function

    Public Function GetHardwareStatus(<Out> ByRef statusOfDevice As String) As Boolean
        statusOfDevice = ""
        If Not IsConnected Then Return False
        Dim status = GetStatus(mConnectedChannelHandle)

        If status <> TPUDSStatus.PUDS_ERROR_OK Then
            ShowError(status)
        Else
            Dim errorDiscription As String

            Select Case status
                Case TPUDSStatus.PUDS_ERROR_OK
                    errorDiscription = "PUDS_ERROR_OK"
                Case TPUDSStatus.PUDS_ERROR_NOT_INITIALIZED
                    errorDiscription = "PUDS_ERROR_NOT_INITIALIZED"
                Case TPUDSStatus.PUDS_ERROR_BUSLIGHT
                    errorDiscription = "PUDS_ERROR_BUSLIGHT"
                Case TPUDSStatus.PUDS_ERROR_BUSHEAVY
                    errorDiscription = "PUDS_ERROR_BUSHEAVY"
                Case TPUDSStatus.PUDS_ERROR_BUSOFF
                    errorDiscription = "PUDS_ERROR_BUSOFF"
                Case Else
                    errorDiscription = "See Documentation"
            End Select

            statusOfDevice = String.Format("Channel status: {0} ({1,8:X}h)", errorDiscription, status)
        End If

        Return status = TPUDSStatus.PUDS_ERROR_OK
    End Function

    Public Function ResetHardware() As Boolean
        If Not IsConnected Then Return False
        Dim status = Reset(mConnectedChannelHandle)
        If status <> TPUDSStatus.PUDS_ERROR_OK Then ShowError(status)
        Return status = TPUDSStatus.PUDS_ERROR_OK
    End Function

    Public Function GetLocalAddress(<Out> ByRef address As UInteger) As Boolean
        Dim status As TPUDSStatus = UDSApi.GetValue(mConnectedChannelHandle, TPUDSParameter.PUDS_PARAM_SERVER_ADDRESS, address, 4)

        If status <> TPUDSStatus.PUDS_ERROR_OK Then
            ShowError(status)
        Else
            If mLocalNodeAddress <> address Then mLocalNodeAddress = address
            If (address And PUDS_SERVER_ADDR_FLAG_ENHANCED_ISO_15765_3) = PUDS_SERVER_ADDR_FLAG_ENHANCED_ISO_15765_3 Then address = address Xor PUDS_SERVER_ADDR_FLAG_ENHANCED_ISO_15765_3
        End If

        Return status = TPUDSStatus.PUDS_ERROR_OK
    End Function

    Public Function SetLocalAddress(ByVal address As UInteger, ByVal isEnhanced As Boolean) As Boolean
        'ISO 14229-1:2006 addresses need to be marked by the flag PUDS_SERVER_ADDR_FLAG_ENHANCED_ISO_15765_3
        If isEnhanced Then address = address Or PUDS_SERVER_ADDR_FLAG_ENHANCED_ISO_15765_3
        Dim status As TPUDSStatus = UDSApi.SetValue(mConnectedChannelHandle, TPUDSParameter.PUDS_PARAM_SERVER_ADDRESS, address, 4)

        If status <> TPUDSStatus.PUDS_ERROR_OK Then
            ShowError(status)
        Else
            mLocalNodeAddress = address
        End If

        Return status = TPUDSStatus.PUDS_ERROR_OK
    End Function

    Public Function AddAddressToFilter(ByVal address As UInteger) As Boolean
        address = address Or PUDS_SERVER_FILTER_LISTEN
        Dim status As TPUDSStatus = UDSApi.SetValue(mConnectedChannelHandle, TPUDSParameter.PUDS_PARAM_SERVER_FILTER, address, 4)

        If status <> TPUDSStatus.PUDS_ERROR_OK Then ShowError(status)
        Return status = TPUDSStatus.PUDS_ERROR_OK
    End Function

    Public Function RemoveAddressFromFilter(ByVal address As UInteger) As Boolean
        address = address Or PUDS_SERVER_FILTER_IGNORE
        Dim status As TPUDSStatus = UDSApi.SetValue(mConnectedChannelHandle, TPUDSParameter.PUDS_PARAM_SERVER_FILTER, address, 4)

        If status <> TPUDSStatus.PUDS_ERROR_OK Then ShowError(status)
        Return status = TPUDSStatus.PUDS_ERROR_OK
    End Function

    Public Function GetUIntParameter(ByVal parameter As TPUDSParameter, <Out> ByRef value As UInteger) As Boolean
        Dim status As TPUDSStatus = UDSApi.GetValue(mConnectedChannelHandle, parameter, value, 4)
        status = UDSApi.GetValue(mConnectedChannelHandle, parameter, value, 4)
        If status <> TPUDSStatus.PUDS_ERROR_OK Then ShowError(status)
        Return status = TPUDSStatus.PUDS_ERROR_OK
    End Function

    Public Function GetStringParameter(ByVal parameter As TPUDSParameter, <Out> ByRef value As String) As Boolean
        Dim stringBuilder = New StringBuilder(256)
        Dim status = GetValue(mConnectedChannelHandle, parameter, stringBuilder, stringBuilder.Capacity)
        If status <> TPUDSStatus.PUDS_ERROR_OK Then ShowError(status)
        value = stringBuilder.ToString
        Return status = TPUDSStatus.PUDS_ERROR_OK
    End Function

    Public Function GetApiVersion(<Out> ByRef information As String) As Boolean
        information = ""
        Dim value As String = Nothing

        If GetStringParameter(TPUDSParameter.PUDS_PARAM_API_VERSION, value) Then
            information = "Api Version: " & value
            Return True
        End If

        Return False
    End Function

    Public Function SetParameter(ByVal parameter As TPUDSParameter, ByVal value As UInteger) As Boolean
        Dim status As TPUDSStatus = UDSApi.SetValue(mConnectedChannelHandle, parameter, value, 4)
        If status <> TPUDSStatus.PUDS_ERROR_OK Then ShowError(status)
        Return status = TPUDSStatus.PUDS_ERROR_OK
    End Function

    Public Function AddMapping(ByVal mapping As MappingInformation) As Boolean
        Dim status = SetValue(mConnectedChannelHandle, TPUDSParameter.PUDS_PARAM_MAPPING_ADD, mapping.MappingPointer, mapping.PointerSize)
        If status <> TPUDSStatus.PUDS_ERROR_OK Then ShowError(status)
        Return status = TPUDSStatus.PUDS_ERROR_OK
    End Function

    Public Function RemoveMapping(ByVal mapping As MappingInformation) As Boolean
        Dim status = SetValue(mConnectedChannelHandle, TPUDSParameter.PUDS_PARAM_MAPPING_REMOVE, mapping.MappingPointerRemoval, mapping.PointerSize)
        If status <> TPUDSStatus.PUDS_ERROR_OK Then ShowError(status)
        Return status = TPUDSStatus.PUDS_ERROR_OK
    End Function

    Public Function SendService(ByVal serviceInformation As ServiceInformation) As Boolean
        Dim status As TPUDSStatus = 0
        mRequest.NETADDRINFO = serviceInformation.NetAddressInformation
        mServiceInformationOfLastRequest = serviceInformation
        RequestIsFunctional = serviceInformation.NetAddressInformation.TA_TYPE = TPUDSAddressingType.PUDS_ADDRESSING_FUNCTIONAL
        If ProtocolIsDetachedFromIdsAndAddresses(serviceInformation.NetAddressInformation.PROTOCOL) Then SetPriority(serviceInformation.Priority)
        Dim byteBuffer As Byte()

        Select Case serviceInformation.Service
            Case TPUDSService.PUDS_SI_TesterPresent
                status = SvcTesterPresent(mConnectedChannelHandle, mRequest, TPUDSSvcParamTP.PUDS_SVC_PARAM_TP_ZSUBF)
            Case TPUDSService.PUDS_SI_DiagnosticSessionControl
                Dim paramDSC As TPUDSSvcParamDSC = serviceInformation.ParametersOfService(0)
                status = SvcDiagnosticSessionControl(mConnectedChannelHandle, mRequest, paramDSC)
            Case TPUDSService.PUDS_SI_ReadDataByIdentifier
                Dim identifiers = CType(serviceInformation.ParametersOfService(0), UShort())
                status = SvcReadDataByIdentifier(mConnectedChannelHandle, mRequest, identifiers, identifiers.Length)
            Case TPUDSService.PUDS_SI_ClearDiagnosticInformation
                Dim groupOfDtc = Convert.ToUInt32(serviceInformation.ParametersOfService(0))
                status = SvcClearDiagnosticInformation(mConnectedChannelHandle, mRequest, groupOfDtc)
            Case TPUDSService.PUDS_SI_ReadDTCInformation
                Dim rdtciType As TPUDSSvcParamRDTCI = serviceInformation.ParametersOfService(0)
                Dim dtcStatusMask = Convert.ToByte(serviceInformation.ParametersOfService(1))
                status = SvcReadDTCInformation(mConnectedChannelHandle, mRequest, rdtciType, dtcStatusMask)
            Case TPUDSService.PUDS_SI_ControlDTCSetting
                Dim dtcSettingType As TPUDSSvcParamCDTCS = serviceInformation.ParametersOfService(0)
                byteBuffer = CType(serviceInformation.ParametersOfService(1), Byte())
                status = SvcControlDTCSetting(mConnectedChannelHandle, mRequest, dtcSettingType, byteBuffer, byteBuffer.Length)
            Case TPUDSService.PUDS_SI_CommunicationControl
                Dim controlType As TPUDSSvcParamCC = serviceInformation.ParametersOfService(0)
                Dim communicationType = Convert.ToByte(serviceInformation.ParametersOfService(1))
                status = SvcCommunicationControl(mConnectedChannelHandle, mRequest, controlType, communicationType)
            Case TPUDSService.PUDS_SI_SecurityAccess
                Dim securityAccessType = Convert.ToByte(serviceInformation.ParametersOfService(0))
                byteBuffer = CType(serviceInformation.ParametersOfService(1), Byte())
                status = SvcSecurityAccess(mConnectedChannelHandle, mRequest, securityAccessType, byteBuffer, byteBuffer.Length)
            Case TPUDSService.PUDS_SI_RequestDownload
                Dim compressionMethod = Convert.ToByte(serviceInformation.ParametersOfService(0))
                Dim encryptingMethod = Convert.ToByte(serviceInformation.ParametersOfService(1))
                Dim memoryAddress = CType(serviceInformation.ParametersOfService(2), Byte())
                Dim memorySize = CType(serviceInformation.ParametersOfService(3), Byte())
                status = SvcRequestDownload(mConnectedChannelHandle, mRequest, compressionMethod, encryptingMethod, memoryAddress, memoryAddress.Length, memorySize, memorySize.Length)
            Case TPUDSService.PUDS_SI_TransferData
                Dim blockSequenceCounter = Convert.ToByte(serviceInformation.ParametersOfService(0))
                byteBuffer = CType(serviceInformation.ParametersOfService(1), Byte())
                status = SvcTransferData(mConnectedChannelHandle, mRequest, blockSequenceCounter, byteBuffer, byteBuffer.Length)
            Case TPUDSService.PUDS_SI_RequestTransferExit
                byteBuffer = CType(serviceInformation.ParametersOfService(0), Byte())
                status = SvcRequestTransferExit(mConnectedChannelHandle, mRequest, byteBuffer, byteBuffer.Length)
            Case TPUDSService.PUDS_SI_RoutineControl
                Dim routineControlType As TPUDSSvcParamRC = serviceInformation.ParametersOfService(0)
                Dim routineIdentifier = Convert.ToUInt16(serviceInformation.ParametersOfService(1))
                byteBuffer = CType(serviceInformation.ParametersOfService(2), Byte())
                status = SvcRoutineControl(mConnectedChannelHandle, mRequest, routineControlType, routineIdentifier, byteBuffer, byteBuffer.Length)
            Case TPUDSService.PUDS_SI_ECUReset
                Dim resetType As TPUDSSvcParamER = serviceInformation.ParametersOfService(0)
                status = SvcECUReset(mConnectedChannelHandle, mRequest, resetType)
            Case Else
                Return False
        End Select

        If status <> TPUDSStatus.PUDS_ERROR_OK Then mResponseStatus = status
        Return status = TPUDSStatus.PUDS_ERROR_OK
    End Function

    Private Sub SetPriority(ByVal prio As Byte)
        Dim isoTpStatus As UInteger = 0, priority As UInteger = prio
        isoTpStatus = IsoTp_SetValue(mConnectedChannelHandle, &HEE, priority, 4)
        If isoTpStatus <> 0 Then Windows.Forms.MessageBox.Show("Error on set prio: " & isoTpStatus.ToString)
    End Sub

    Public Function WaitForResponse() As Boolean
        Dim status As TPUDSStatus
        Dim localResponse As TPUDSMsg = Nothing, transmitConfirmation As TPUDSMsg = Nothing, receiveCount As UInteger = Nothing

        If Not RequestIsFunctional Then
            status = WaitForService(mConnectedChannelHandle, localResponse, mRequest, transmitConfirmation)
            If status = TPUDSStatus.PUDS_ERROR_OK Then Response = localResponse
        Else
            status = WaitForServiceFunctional(mConnectedChannelHandle, FunctionalResponses, FunctionalResponses.Length, receiveCount, True, mRequest, transmitConfirmation)
            If status = TPUDSStatus.PUDS_ERROR_OK Then ReceivedFunctionalResponses = receiveCount
        End If

        If status <> TPUDSStatus.PUDS_ERROR_OK Then
            mResponseStatus = status
        ElseIf mServiceInformationOfLastRequest.Service = TPUDSService.PUDS_SI_DiagnosticSessionControl Then
            Dim paramDSC As TPUDSSvcParamDSC = mServiceInformationOfLastRequest.ParametersOfService(0)
            If paramDSC <> TPUDSSvcParamDSC.PUDS_SVC_PARAM_DSC_DS AndAlso mServiceInformationOfLastRequest.DisableTesterPresent Then DeactivateTesterPresent(mServiceInformationOfLastRequest.NetAddressInformation)
        End If

        Return status = TPUDSStatus.PUDS_ERROR_OK
    End Function

    Public Function DeactivateTesterPresent(ByVal netAddrInfo As TPUDSNetAddrInfo) As Boolean
        'build pointer to session info...
        Dim sessionInfo = New TPUDSSessionInfo
        sessionInfo.NETADDRINFO = netAddrInfo
        Dim sessionSize = Marshal.SizeOf(sessionInfo)
        Dim sessionPtr = Marshal.AllocHGlobal(sessionSize)
        Marshal.StructureToPtr(sessionInfo, sessionPtr, False)
        '...to retreive current session info...
        Dim status = GetValue(mConnectedChannelHandle, TPUDSParameter.PUDS_PARAM_SESSION_INFO, sessionPtr, CUInt(sessionSize))
        '...if an error occured, show error message and exit method...
        If status <> TPUDSStatus.PUDS_ERROR_OK Then
            Marshal.FreeHGlobal(sessionPtr)
            ShowError(status)
            Return False
        End If
        '...else safe current session info in variable...
        sessionInfo = CType(Marshal.PtrToStructure(sessionPtr, GetType(TPUDSSessionInfo)), TPUDSSessionInfo)
        Marshal.FreeHGlobal(sessionPtr)
        '...override session type to prevent sending of tester present...
        sessionInfo.SESSION_TYPE = TPUDSSvcParamDSC.PUDS_SVC_PARAM_DSC_DS
        '...get session pointer with changed session type...
        sessionPtr = Marshal.AllocHGlobal(sessionSize)
        Marshal.StructureToPtr(sessionInfo, sessionPtr, False)
        '...set session info to prevent the sending of tester present...
        status = status Or SetValue(mConnectedChannelHandle, TPUDSParameter.PUDS_PARAM_SESSION_INFO, sessionPtr, CUInt(sessionSize))
        '...free space ocupied by the pointer
        Marshal.FreeHGlobal(sessionPtr)
        If status <> TPUDSStatus.PUDS_ERROR_OK Then ShowError(status)
        Return status = TPUDSStatus.PUDS_ERROR_OK
    End Function

    Public Function SendUUDT(ByVal information As MappingInformation, ByVal dlc As Integer, ByVal data As Byte()) As Boolean
        Dim uudtMessage = New TPUDSMsg
        uudtMessage.NETADDRINFO = information.NetAddressInformation
        uudtMessage.LEN = CUShort(dlc + 4)
        uudtMessage.DATA = New Byte(4094) {}
        uudtMessage.DATA(0) = CByte(information.CanId >> 24 And &HFF)
        uudtMessage.DATA(1) = CByte(information.CanId >> 16 And &HFF)
        uudtMessage.DATA(2) = CByte(information.CanId >> 8 And &HFF)
        uudtMessage.DATA(3) = CByte(information.CanId >> 0 And &HFF)
        Array.Copy(data, 0, uudtMessage.DATA, 4, dlc)
        Dim status = Write(mConnectedChannelHandle, uudtMessage)
        If status <> TPUDSStatus.PUDS_ERROR_OK Then ShowError(status)
        Return status = TPUDSStatus.PUDS_ERROR_OK
    End Function

    Public Function ReceiveUUDT(ByVal argument As MappingInformation) As Boolean
        Dim message As TPUDSMsg = Nothing
        mUUDTStatus = Read(mConnectedChannelHandle, message)

        If mUUDTStatus = TPUDSStatus.PUDS_ERROR_OK Then
            UUDTResponse = message
            Return True
        End If

        Return False
    End Function

    Public Sub ShowResponseError(ByVal Optional isUUDT As Boolean = False)
        If isUUDT Then
            ShowError(mUUDTStatus)
        Else
            ShowError(mResponseStatus)
        End If
    End Sub

    Private Sub ShowError(ByVal status As TPUDSStatus)
        If status = TPUDSStatus.PUDS_ERROR_OK Then Return
        Dim errorText = "An error occurred (" & [Enum].GetName(GetType(TPUDSStatus), status) & "):" & vbCrLf & ""

        Select Case status
            Case TPUDSStatus.PUDS_ERROR_NOT_INITIALIZED
                errorText += "The channel is not initialized"
            Case TPUDSStatus.PUDS_ERROR_ALREADY_INITIALIZED
                errorText += "The channel is already initialized"
            Case TPUDSStatus.PUDS_ERROR_NO_MEMORY
                errorText += "Failed to allocate memory"
            Case TPUDSStatus.PUDS_ERROR_OVERFLOW
                errorText += "A buffer overflow occured (too many channels initialized or too many messages in queue)."
            Case TPUDSStatus.PUDS_ERROR_TIMEOUT
                errorText += "Timeout while trying to access the PCAN-UDS API."
            Case TPUDSStatus.PUDS_ERROR_NO_MESSAGE
                errorText += "No message is available."
            Case TPUDSStatus.PUDS_ERROR_WRONG_PARAM
                errorText += "The given parameter is invalid"
            Case TPUDSStatus.PUDS_ERROR_BUSLIGHT
                errorText += "An error counter on the CAN bus reached the 'light' limit"
            Case TPUDSStatus.PUDS_ERROR_BUSHEAVY
                errorText += "An error counter on the CAN bus reached the 'heavy' limit"
            Case TPUDSStatus.PUDS_ERROR_BUSOFF
                errorText += "The CAN  controller is in bus-off state"
            Case Else

                If (status And TPUDSStatus.PUDS_ERROR_CAN_ERROR) = TPUDSStatus.PUDS_ERROR_CAN_ERROR Then
                    Dim pcanBasicError As UInteger = (status Xor TPUDSStatus.PUDS_ERROR_CAN_ERROR)
                    errorText += "An error occured in the underlying PCAN-Basic API." & vbCrLf & "The PCAN-Basic error code is: 0x" & pcanBasicError.ToString("X8")
                Else
                    errorText += "unknown error code."
                End If
        End Select

        Windows.Forms.MessageBox.Show(errorText, "Error", Windows.Forms.MessageBoxButtons.OK, Windows.Forms.MessageBoxIcon.Error)
    End Sub
End Class
