//  PXCP.h
//
//  ~~~~~~~~~~~~
//
//  PCAN-XCP API
//
//  ~~~~~~~~~~~~
//
//  ------------------------------------------------------------------
//  Author : Keneth Wagner
//	Last change: 06.11.2018 Wagner
//
//  Language: ANSI-C
//  ------------------------------------------------------------------
//
//  Copyright (C) 2018  PEAK-System Technik GmbH, Darmstadt
//  more Info at http://www.peak-system.com 
//
#ifndef __PXCPH__
#define __PXCPH__

////////////////////////////////////////////////////////////
// Inclusion of other needed files
////////////////////////////////////////////////////////////

#ifndef __PCANBASICH__
#include "PCANBasic.h"							         // PCAN-Basic API
#endif

////////////////////////////////////////////////////////////
// Type definitions
////////////////////////////////////////////////////////////

#define TXCPChannel                               WORD            // Represents a PCAN-XCP Channel Handle
#define TXCPHandle                                DWORD           // Represents a PCAN-XCP session Handle (XCP Channel + Slave data)
#define TXCPResult                                DWORD           // Represent the PCAN XCP result and error codes 
#define TXCPQueue                                 DWORD           // Represents the queue selection for queue related function calls

// Limit values
//
#define XCP_MAX_QUEUE_SIZE                        0x7FFF          // Maximum count of asynchronous messages that a queue can retain

// Transport protocols
//
#define XCP_TRANSPORT_ON_CAN                      1
#define XCP_TRANSPORT_ON_TCP                      2
#define XCP_TRANSPORT_ON_LIN                      3

// XCP Package Types
//
#define XCP_MSGTYPE_RES                           0xFF             // Slave-->Master CTO | Command responses
#define XCP_MSGTYPE_ERR                           0xFE             // Slave-->Master CTO | Error packets 
#define XCP_MSGTYPE_EV                            0xFD             // Slave-->Master CTO | Event packets
#define XCP_MSGTYPE_SERV                          0xFC             // Slave-->Master CTO | Service request packets
#define XCP_MSGTYPE_MAX_DAQ                       0xFB             // Slave-->Master DTO | Synchronous data acquisition from 0x0 to XCP_MSGTYPE_MAX_DAQ

// Error Codes
//
// Codes for not sucessfully executed XCP commands
//
#define XCP_ERR_CMD_SYNCH                         0                // Command processor synchronization
#define XCP_ERR_CMD_BUSY                          0x10             // Command was not executed
#define XCP_ERR_DAQ_ACTIVE                        0x11             // Command rejected because DAQ is running
#define XCP_ERR_PGM_ACTIVE                        0x12             // Command rejected because PGM is running
#define XCP_ERR_CMD_UNKNOWN                       0x20             // Unknown command or not implemented optional command
#define XCP_ERR_CMD_SYNTAX                        0x21             // Command syntax invalid
#define XCP_ERR_OUT_OF_RANGE                      0x22             // Command syntax valid but command parameter(s) out of range
#define XCP_ERR_WRITE_PROTECTED                   0x23             // The memory location is write protected
#define XCP_ERR_ACCESS_DENIED                     0x24             // The memory location is not accessible
#define XCP_ERR_ACCESS_LOCKED                     0x25             // Access denied,Seed & Key is required
#define XCP_ERR_PAGE_NOT_VALID                    0x26             // Selected page not available
#define XCP_ERR_MODE_NOT_VALID                    0x27             // Selected page mode not available
#define XCP_ERR_SEGMENT_NOT_VALID                 0x28             // Selected segment not valid
#define XCP_ERR_SEQUENCE                          0x29             // Sequence error
#define XCP_ERR_DAQ_CONFIG                        0x2A             // DAQ configuration not valid
#define XCP_ERR_MEMORY_OVERFLOW                   0x30             // Memory overflow error
#define XCP_ERR_GENERIC                           0x31             // Generic error
#define XCP_ERR_VERIFY                            0x32             // The slave internal program verify routine detects an error
#define XCP_ERR_RESOURCE_TEMPORARY_NOT_ACCESSIBLE 0x33             // Access to the requested resource is temporary not possible
// API return error codes
//
#define XCP_ERR_OK                                (1<<8)           // Acknowledge / no error
#define XCP_ERR_NOT_IMPLEMENTED                   (2<<8)           // Function not available / Operation not implemented
#define XCP_ERR_INVALID_PARAMETER                 (3<<8)           // Invalid parameter value
#define XCP_ERR_MAX_CHANNELS                      (4<<8)           // The maximum amount of registered Slave channels was reached
#define XCP_ERR_INVALID_HANDLE                    (5<<8)           // The given handle is invalid
#define XCP_ERR_INTERNAL_TIMEOUT                  (6<<8)           // A timeout was reached by calling a function synchronously
#define XCP_ERR_QUEUE_EMPTY                       (7<<8)           // The queue being referred is empty
#define XCP_ERR_INSUFFICIENT_BUFFER               (8<<8)           // The size of the given buffer, is not big enough
#define XCP_ERR_TRANSPORT_CHANNEL                  0x80000000      // Flag for a specific error within the underlying transport channel 

#define CAN_MAX_LEN                                8               // Maximum length for a standard CAN message
#define CAN_MAX_LEN_FD                             64              // Maximum length for a CAN-FD message
#define CAN_MAX_CTO                                CAN_MAX_LEN     // Maximum length of a CTO packet on standard CAN
#define CAN_MAX_CTO_PGM                            CAN_MAX_LEN     // Maximum length of a CTO packet on standard CAN, when programming
#define CAN_MAX_DTO                                CAN_MAX_LEN     // Maximum length of a DTO packet on standard CAN

#define XCP_CTO_QUEUE                              1               // Represents the CTO queue of a XCP session
#define XCP_DTO_QUEUE                              2               // Represents the DTO queue of a XCP session

struct TXCPCalibrationPage
{
	BYTE SegmentNumber; 	
	BYTE PageNumber;
};

struct TXCPProgramFormat
{
	BYTE CompressionMethod; 	
	BYTE EncryptionMethod;
	BYTE ProgrammingMehtod;
	BYTE AccessMethod;
};

struct TXCPODTEntry
{
	BYTE BitOffset;
	BYTE DAQSize;
	BYTE DAQAddressExtension;
	DWORD DAQAddress;
};

struct TXCPDAQListConfig
{
	WORD DAQListNumber;
	WORD EventChannelNumber;
	BYTE TransmissionRate;
	BYTE DAQPriority;
};

struct TXCPTransportLayerCAN
{
	DWORD BroadcastID;
	DWORD MasterID;
	DWORD SlaveID;
	bool IncrementalIdUsed;
};

struct TXCPProtocolLayerConfig
{
	WORD T1;
	WORD T2;
	WORD T3;
	WORD T4;
	WORD T5;
	WORD T6;
	WORD T7;
	bool MotorolaFormat;      // DEPRECATED: Value will automatically adjusted on connect (XCP_Connect)
	BYTE AddressGranularity;  // DEPRECATED: Value will automatically adjusted on connect (XCP_Connect)
};

#ifdef __cplusplus
extern "C" {
#define _DEF_ARG =0
#else
#define _DEF_ARG
#endif

////////////////////////////////////////////////////////////
// PCAN-XCP API function declarations
////////////////////////////////////////////////////////////

//------------------------------
// Extras
//------------------------------

/// <summary>
/// Allocates an XCP Channel for CAN communication using PCAN-Basic API
/// </summary>
/// <param name="XcpChannel">Buffer for the XcpChannel representing this CAN Channel</param>
/// <param name="Channel">The handle of a PCAN Channel to be initialized</param>
/// <param name="Btr0Btr1">The speed for the communication (BTR0BTR1 code)</param>
/// <param name="HwType">NON PLUG&PLAY: The type of hardware and operation mode</param>
/// <param name="IOPort">NON PLUG&PLAY: The I/O address for the parallel port</param>
/// <param name="Interupt">NON PLUG&PLAY: Interrupt number of the parallel port</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_InitializeCanChannel(
	TXCPChannel *XcpChannel,
    TPCANHandle Channel, 
    TPCANBaudrate Btr0Btr1, 
    TPCANType HwType _DEF_ARG,
    DWORD IOPort _DEF_ARG, 
    WORD Interrupt _DEF_ARG);

/// <summary>
/// Allocates an XCP Channel for CAN FD communication using PCAN-Basic API 
/// </summary>
/// <param name="XcpChannel">Buffer for the XcpChannel representing this CAN Channel</param>
/// <param name="Channel">The handle of a PCAN Channel to be initialized</param>
/// <param name="BitrateFD">"The speed for the communication (FD bit rate string)"</param>
/// <remarks>See PCAN_BR_* values within the PCAN-Basic help
/// * Parameter and values must be separated by '='
/// * Couples of Parameter/value must be separated by ','
/// * Following Parameter must be filled out: f_clock, data_brp, data_sjw, data_tseg1, data_tseg2,
///   nom_brp, nom_sjw, nom_tseg1, nom_tseg2.
/// * Following Parameters are optional (not used yet): data_ssp_offset, nom_samp
///</remarks>
/// <example>f_clock_mhz=80,nom_brp=0,nom_tseg1=13,nom_tseg2=0,nom_sjw=0,data_brp=0,
/// data_tseg1=13,data_tseg2=0,data_sjw=0</example>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_InitializeCanChannelFD(
	TXCPChannel *XcpChannel,
	TPCANHandle Channel, 
	TPCANBitrateFD bitrate);

/// <summary>
/// Uninitializes a XCP Channel
/// </summary>
/// <param name="XcpChannel">The handle of a Xcp Channel</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_UninitializeChannel(
	TXCPChannel XcpChannel);

/// <summary>
/// Associates an ECU to a XCP Channel to communicate over CAN.
/// </summary>
/// <param name="XcpChannel">The handle of a Xcp Channel</param>
/// <param name="LayerData">The slave (ECU) data for CAN communication</param>
/// <param name="Protocol">Protocol layer's configuration data </param>
/// <param name="XcpHandle">Buffer for the handle representing the session (Channel + Slave)</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_AddSlaveOnCAN(
	TXCPChannel XcpChannel, 
	TXCPTransportLayerCAN LayerData, 
	TXCPProtocolLayerConfig Protocol, 
	TXCPHandle *XcpHandle);

/// <summary>
/// Removes the association between an ECU and a XCP Channel.
/// </summary>
/// <param name="XcpHandle">The handle of the XCP session</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_RemoveSlave(
	TXCPHandle XcpHandle);

/// <summary>
/// Retrieves a CTO or DTO package from the XCP session
/// represented by the given XcpHandle
/// </summary>
/// <param name="XcpHandle">The handle of the XCP session</param>
/// <param name="queueSelect">The queue from where a packet will be removed</param>
/// <param name="packetBuffer">Buffer for the CTO or DTO packet</param>
/// <param name="packetBufferLength">Length of teh packet buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_DequeuePacket(
        TXCPHandle XcpHandle, 
		TXCPQueue queueSelect,
		BYTE *packetBuffer,
		WORD packetBufferLength);

/// <summary>
/// Resets the CTO/DTO queue(s) of a XCP session represented 
/// by the given XcpHandle
/// </summary>
/// <param name="XcpHandle">The handle of the XCP session</param>
/// <param name="queueToClear">Value indicating the queue to be reset</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_ResetQueue(
	TXCPHandle XcpHandle, 
	TXCPQueue queueToClear);

//------------------------------
// Connection
//------------------------------

/// <summary>
/// Command CONNECT: establishes a continuous, logical, point-to-point 
/// connection with a slave device.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="Mode">0: Normal, 1: User defined</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_Connect(
    TXCPHandle XcpHandle, 
	BYTE Mode, 
	BYTE *ctoBuffer,
	BYTE ctoBufferLength);

/// <summary>
/// Command DISCONNECT: brings the slave to the DISCONNECTED state.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_Disconnect(
    TXCPHandle XcpHandle, 
	BYTE *ctoBuffer,
	BYTE ctoBufferLength);

//------------------------------
// Control and Configuration
//------------------------------

/// <summary>
/// Command GET_STATUS: returns all current status information of the slave device.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_GetStatus(
	TXCPHandle XcpHandle, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command SYNCH: used to synchronize command execution after timeout conditions.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_Synchronize(
	TXCPHandle XcpHandle, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command GET_COMM_MODE_INFO: returns optional information on different 
/// Communication Modes supported by the slave.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_GetCommunicationMode(
	TXCPHandle XcpHandle, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command GET_ID: used for automatic session configuration and for 
/// slave device identification.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="Type">The ideintification type (0..4)</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_GetIdentification(
	TXCPHandle XcpHandle, 
	BYTE Type, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command SET_REQUEST: request to save to non-volatile memory.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="Mode">Request mode especification</param>
/// <param name="SessionId">Session Configuration identification</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_SetRequest(
	TXCPHandle XcpHandle, 
	BYTE Mode, 
	WORD SessionId, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command GET_SEED: Get seed for unlocking a protected resource.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="Mode">Seed mode. 0: First part, 1: remaining part</param>
/// <param name="Resource">Resource id, when Mode is 0. Otherwise don't care</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_GetSeed(
	TXCPHandle XcpHandle, 
	BYTE Mode, 
	BYTE Resource, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command UNLOCK: unlocks the slave devices security protection using a 
/// key computed from the seed obtained by a previous GET_SEED sequence.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="keyLength">Indicates the (remaining) number of key bytes</param>
/// <param name="Key">Buffer for the Key's data bytes</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_Unlock(
	TXCPHandle XcpHandle, 
	BYTE keyLength, 
	BYTE* Key, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command TRANSPORT_LAYER_CMD: sends a specific Transport Layer command.
/// For CAN: Works only in blocking mode (ctoBuffer cannot be NULL)
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="subCommand">Transport layer specific command</param>
/// <param name="paramBuffer">Command parameters buffer</param>
/// <param name="paramBufferLength">Length of the Command parameters buffer</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_TransportLayerCommand(
	TXCPHandle XcpHandle, 
	BYTE subCommand, 
	BYTE* paramBuffer, 
	WORD paramBufferLength, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command USER_CMD: sends a user specific command.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="subCommand">User defined command</param>
/// <param name="paramBuffer">Command parameters buffer</param>
/// <param name="paramBufferLength">Length of the Command parameters buffer</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_UserCommand(
	TXCPHandle XcpHandle, 
	BYTE subCommand, 
	BYTE* paramBuffer, 
	WORD paramBufferLength, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

//------------------------------
// Memory Management
//------------------------------

/// <summary>
/// Command SET_MTA: initializes a pointer (32Bit address + 8Bit extension) 
/// for following memory transfer commands: BUILD_CHECKSUM, UPLOAD, DOWNLOAD,
/// DOWNLOAD_NEXT, DOWNLOAD_MAX, MODIFY_BITS, PROGRAM_CLEAR, PROGRAM, 
/// PROGRAM_NEXT and PROGRAM_MAX.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="AddrExtension">Address extension</param>
/// <param name="Addr">Address</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_SetMemoryTransferAddress(
	TXCPHandle XcpHandle, 
	BYTE AddrExtension, 
	DWORD Addr, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command UPLOAD: Upload a data block from slave to master, starting at the 
/// current MTA. The MTA will be post-incremented by the given number of data elements.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="numberOfElements">The amount of elements to retrieve</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_Upload(
	TXCPHandle XcpHandle, 
	BYTE numberOfElements,
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command SHORT_UPLOAD: Upload a data block, of a maximum size of [1..MAX_CTO/AG], 
/// from slave to master. 
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="numberOfElements">The amount of elements to retrieve</param>
/// <param name="AddrExtension">Address extension to set the MTA pointer</param>
/// <param name="Addr">Address to set the MTA pointer</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_ShortUpload(
	TXCPHandle XcpHandle, 
	BYTE numberOfElements, 
	BYTE AddrExtension, 
	DWORD Addr, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command DOWNLOAD: Download a data block from master to slave, starting at the 
/// current MTA. The MTA will be post-incremented by the number of data elements.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="numberOfElements">The amount of elements to be transfered</param>
/// <param name="dataBuffer">Data buffer with the elements to transfer</param>
/// <param name="dataBufferLength">Length of the data buffer</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_Download(
	TXCPHandle XcpHandle, 
	BYTE numberOfElements, 
	BYTE* dataBuffer, 
	BYTE dataBufferLength, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command DOWNLOAD_NEXT: Download consecutive data blocks from master to 
/// slave (Block Mode), starting at the current MTA. The MTA will be post-incremented 
/// by the number of data elements.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="numberOfElements">The amount of elements to be transfered</param>
/// <param name="dataBuffer">Data buffer with the elements to transfer</param>
/// <param name="dataBufferLength">Length of the data buffer</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_DownloadNext(
	TXCPHandle XcpHandle, 
	BYTE numberOfElements, 
	BYTE* dataBuffer, 
	BYTE bufferLength, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command DOWNLOAD_MAX: Download a data block with fixed size (MAX_CTO/AG-1) from 
/// master to slave, starting at the current MTA. The MTA will be post-incremented 
/// by MAX_CTO/AG-1.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="dataBuffer">Data buffer with the elements to transfer</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_DownloadMax(
	TXCPHandle XcpHandle, 
	BYTE* dataBuffer, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command MODIFY_BITS: modifies the bits of the 32 Bit memory 
/// location referred by the MTA.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="ShiftValue">Shift Value</param>
/// <param name="ANDMask">AND Mask (MA)</param>
/// <param name="XORMask">XOR Mask (MX)</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_ModifyBits(
	TXCPHandle XcpHandle, 
	BYTE ShiftValue, 
	WORD ANDMask, 
	WORD XORMask, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

//------------------------------
// Calibration
//------------------------------

/// <summary>
/// Command SET_CAL_PAGE: sets the access mode for a calibration data segment.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="Mode">Access mode</param>
/// <param name="Page">Represents a calibration data page (segment and page number)</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_SetCalibrationPage(
	TXCPHandle XcpHandle, 
	BYTE Mode, 
	TXCPCalibrationPage Page, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command GET_CAL_PAGE: returns the logical number for the calibration data page 
/// that is currently activated for the specified access mode and data segment.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="Mode">Access mode</param>
/// <param name="dataSegmentNumber">Logical data segment number</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_GetCalibrationPage(
	TXCPHandle XcpHandle, 
	BYTE Mode, 
	BYTE dataSegmentNumber, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command GET_PAG_PROCESSOR_INFO: returns general information on paging processor.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_GetPagingProcessorInformation(
	TXCPHandle XcpHandle, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command GET_SEGMENT_INFO: returns information on a specific SEGMENT.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="Mode">Information mode (0..2)</param>
/// <param name="segmentNumber">Segment number, [0,1,..MAX_SEGMENT-1]</param>
/// <param name="segmentInfo">Segment information according to the 'information mode'</param>
/// <param name="mappingIndex">Identifier for address mapping range</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_GetSegmentInformation(
	TXCPHandle XcpHandle, 
	BYTE Mode, 
	BYTE segmentNumber, 
	BYTE segmentInfo, 
	BYTE mappingIndex, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command GET_PAGE_INFO: returns information on a specific PAGE.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="Page">Represents a calibration data page (segment and page number)</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_GetPageInformation(
	TXCPHandle XcpHandle, 
	TXCPCalibrationPage Page, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command SET_SEGMENT_MODE: set the mode for a SEGMENT
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="Mode">Segment mode (0,1)</param>
/// <param name="segmentNumber">Segment number, [0,1,..MAX_SEGMENT-1]</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_SetSegmentMode(
	TXCPHandle XcpHandle, 
	BYTE Mode, 
	BYTE segmentNumber, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command GET_SEGMENT_MODE: get the mode for a SEGMENT
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="segmentNumber">Segment number, [0,1,..MAX_SEGMENT-1]</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_GetSegmentMode(
	TXCPHandle XcpHandle, 
	BYTE segmentNumber, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command COPY_CAL_PAGE: forces the slave to copy one calibration page to another.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="Source">Represents the source calibration data page</param>
/// <param name="Destination">Represents the destination calibration data page</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_CopyCalibrationPage(
	TXCPHandle XcpHandle, 
	TXCPCalibrationPage Source, 
	TXCPCalibrationPage Destination, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

//------------------------------
// Data Adquisition / Stimulation
//------------------------------

/// <summary>
/// Command SET_DAQ_PTR: initializes the DAQ list pointer for a subsequent 
/// operation with XCP_WriteDAQListEntry (WRITE_DAQ) or XCP_ReadDAQListEntry (READ_DAQ).
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="daqListNumber">DAQ List number, [0,1,..MAX_DAQ-1]</param>
/// <param name="odtNumber">Relative ODT number within this DAQ list</param>
/// <param name="odtEntry">Relative ODT entry number within this ODT</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_SetDAQListPointer(
	TXCPHandle XcpHandle, 
	WORD daqListNumber, 
	BYTE odtNumber, 
	BYTE odtEntry, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command WRITE_DAQ: writes one ODT entry to a DAQ list defined by 
/// the DAQ list pointer.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="Entry">Represents an ODT entry for a DAQ List</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_WriteDAQListEntry(
	TXCPHandle XcpHandle, 
	TXCPODTEntry Entry, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command WRITE_DAQ_MULTIPLE: writes consecutive ODT entries to a DAQ list 
/// defined by the DAQ list pointer.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="numberOfElements">Amount of ODT entries to be written</param>
/// <param name="elements">Buffer containing the ODT entries for a DAQ List</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_WriteDAQListEntries(
	TXCPHandle XcpHandle, 
	BYTE numberOfElements, 
	TXCPODTEntry *elements, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command SET_DAQ_LIST_MODE: sets the mode for DAQ list.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="Mode">DAQ list mode bit</param>
/// <param name="Configuration">Represents the configuration of a DAQ List</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_SetDAQListMode(
	TXCPHandle XcpHandle, 
	BYTE Mode, 
	TXCPDAQListConfig Configuration, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command START_STOP_DAQ_LIST: Starts, stops, or selects a DAQ list.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="Mode">0: Stop, 1: Start, 2: Select</param>
/// <param name="daqListNumber">DAQ list number, [0,1,..MAX_DAQ-1]</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_StartStopDAQList(
	TXCPHandle XcpHandle, 
	BYTE Mode, 
	WORD daqListNumber, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command START_STOP_SYNCH: Starts or stops several DAQ lists, synchronously.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="Mode">0: Stop all, 1: Start selected, 2: Stop selected</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_StartStopSynchronizedDAQList(
	TXCPHandle XcpHandle, 
	BYTE Mode, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command READ_DAQ: Reads one ODT entry of a DAQ list defined by the DAQ 
/// list pointer.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_ReadDAQListEntry(
	TXCPHandle XcpHandle, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command GET_DAQ_CLOCK: Gets the DAQ clock from a slave.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_GetDAQClock(
	TXCPHandle XcpHandle, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command GET_DAQ_PROCESSOR_INFO: returns general information on DAQ processor.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_GetDAQProcessorInformation(
	TXCPHandle XcpHandle, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command GET_DAQ_RESOLUTION_INFO: returns information on DAQ 
/// processing resolution.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_GetDAQResolutionInformation(
	TXCPHandle XcpHandle, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command GET_DAQ_LIST_MODE: returns information on the current mode 
/// of a specified DAQ list.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="daqListNumber">DAQ list number, [0,1,..MAX_DAQ-1]</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_GetDAQListMode(
	TXCPHandle XcpHandle, 
	WORD daqListNumber, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command GET_DAQ_EVENT_INFO: returns information on a specific event channel.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="eventChannelNumber">Event channel number, [0,1,..MAX_EVENT_CHANNEL-1]</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_GetEventChannelInformation(
	TXCPHandle XcpHandle, 
	WORD eventChannelNumber, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command CLEAR_DAQ_LIST: clears an specific DAQ list configuration.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="daqListNumber">DAQ list number, [0,1..MAX_DAQ-1]</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_ClearDAQList(
	TXCPHandle XcpHandle, 
	WORD daqListNumber, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command GET_DAQ_LIST_INFO: returns information on a specific DAQ list.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="daqListNumber">DAQ list number, [0,1..MAX_DAQ-1]</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_GetDAQListInformation(
	TXCPHandle XcpHandle, 
	WORD daqListNumber, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command FREE_DAQ: clears all DAQ lists and frees all dynamically allocated 
/// DAQ lists, ODTs and ODT entries.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_FreeDAQLists(
	TXCPHandle XcpHandle, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command ALLOC_DAQ: allocates a number of DAQ lists for a XCP slave device.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="daqCount">Number of DAQ lists to be allocated</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_AllocateDAQLists(
	TXCPHandle XcpHandle, 
	WORD daqCount, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command ALLOC_ODT: allocates a number of ODTs and assigns them to the 
/// specified DAQ list.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="daqListNumber">DAQ list number, [MIN_DAQ, MIN_DAQ+1,..MIN_DAQ+DAQ_COUNT-1]</param>
/// <param name="odtCount">Number of ODTs to be assigned to a DAQ list</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_AllocateODT(
	TXCPHandle XcpHandle, 
	WORD daqListNumber, 
	BYTE odtCount, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command ALLOC_ODT_ENTRY: allocates a number of ODT entries and assigns them to 
/// the specific ODT in this specific DAQ list.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="daqListNumber">DAQ list number, [MIN_DAQ, MIN_DAQ+1,..MIN_DAQ+DAQ_COUNT-1]</param>
/// <param name="odtNumber">Relative ODT number, [0,1,..ODT_COUNT(DAQ list)-1]</param>
/// <param name="EntriesCount">Number of ODT entries to be assigned to ODT</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_AllocateODTEntry(
	TXCPHandle XcpHandle, 
	WORD daqListNumber, 
	BYTE odtNumber, 
	BYTE entriesCount, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

//------------------------------
// Flash Programming
//------------------------------

/// <summary>
/// Command PROGRAM_START: indicates the begin of a non-volatile memory programming
/// sequence.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_ProgramStart(
	TXCPHandle XcpHandle, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command PROGRAM_CLEAR: clears a part of non-volatile memory prior to reprogramming.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="Mode">Access mode, 0: absolute access, 1: functional access</param>
/// <param name="clearRange">Memory range to clear according to the access mode</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_ProgramClear(
	TXCPHandle XcpHandle, 
	BYTE Mode, 
	DWORD clearRange, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command PROGRAM: Programs a non-volatile memory segment inside the slave.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="numberOfElements">Number of data elements to program, [1..(MAX_CTO_PGM-2)/AG]</param>
/// <param name="dataBuffer">Buffer with the data elements to program</param>
/// <param name="dataBufferLength">Length of the data buffer</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_Program(
	TXCPHandle XcpHandle, 
	BYTE numberOfElements, 
	BYTE* dataBuffer, 
	BYTE dataBufferLength, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command PROGRAM_RESET: Indicates the end of a non-volatile memory programming sequence.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_ProgramReset(
	TXCPHandle XcpHandle, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command GET_PGM_PROCESSOR_INFO: returns general information on programming processor.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_GetPGMProcessorInformation(
	TXCPHandle XcpHandle, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command GET_SECTOR_INFO: returns information on a specific SECTOR.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="Mode">Get mode, 0: start address, 1: length, 2: name length</param>
/// <param name="SectorNumber">Sector number, [0,1,..MAX_SECTOR-1]</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_GetSectorInformation(
	TXCPHandle XcpHandle, 
	BYTE Mode, 
	BYTE SectorNumber, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command PROGRAM_PREPARE: indicates the begin of a code download as a precondition
/// for non-volatile memory programming.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="codeSize">Size of the code that will be downloaded</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_ProgramPrepare(
	TXCPHandle XcpHandle, 
	WORD codeSize, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command PROGRAM_FORMAT: describes the format of following, uninterrupted data
/// transfer before programming.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="Format">Represents the data format to be used</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_ProgramFormat(
	TXCPHandle XcpHandle, 
	TXCPProgramFormat Format, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command PROGRAM_NEXT: transmits consecutive data bytes for the PROGRAM command in
/// block transfer mode.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="numberOfElements">The amount of elements to be programed</param>
/// <param name="dataBuffer">Data buffer with the elements to program</param>
/// <param name="dataBufferLength">Length of the data buffer</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_ProgramNext(
	TXCPHandle XcpHandle, 
	BYTE numberOfElements, 
	BYTE* dataBuffer, 
	BYTE dataLength, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command PROGRAM_MAX: Programs a (fixed size) amount of elements into a 
/// non-volatile memory segment, starting at the MTA. The MTA will be 
/// postincremented by MAX_CTO_PGM-1.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="elementsBuffer">Data buffer with the fixed length of MAX_CTO_PGM-1</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_ProgramMax(
	TXCPHandle XcpHandle, 
	BYTE* elementsBuffer, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command PROGRAM_VERIFY: Induces a verification process after a 
/// memory programming session.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="Mode">Mode, 0: internal routine, 1: Verification value</param>
/// <param name="Type">Verification Type, i.e. area(s) to verify</param>
/// <param name="Value">Verification value, when mode is 1</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_ProgramVerify(
	TXCPHandle XcpHandle, 
	BYTE Mode, 
	WORD Type, 
	DWORD Value, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Command BUILD_CHECKSUM: Build a checksum over a memory range starting at the MTA.
/// The MTA will be post-incremented by the block size.
/// </summary>
/// <param name="XcpHandle">The handle of a XCP session</param>
/// <param name="blockSize">The number of elements used for the checksum</param>
/// <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
/// <param name="ctoBufferLength">Length of the CTO buffer</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_BuildChecksum(
	TXCPHandle XcpHandle, 
	DWORD blockSize, 
	BYTE* ctoBuffer, 
	BYTE ctoBufferLength);

/// <summary>
/// Returns a descriptive text of a given TXCPResult error code
/// </summary>
/// <param name="errorCode">A TXCPResult result code</param>
/// <param name="textBuffer">Buffer for a null terminated char array (must be at least 256 in length)</param>
/// <returns>A TXCPResult result code</returns>
TXCPResult WINAPI XCP_GetErrorText(
	TXCPResult errorCode,
	LPSTR textBuffer);

#ifdef __cplusplus
}
#endif
#endif