﻿'
'  ~~~~~~~~~~~~
'
'  PCAN-XCP API
'
'  ~~~~~~~~~~~~
'
'  ------------------------------------------------------------------
'  Author : Keneth Wagner
'  Last change: 06.11.2018 K.Wagner
'
'  Language: VB.Net 
'  ------------------------------------------------------------------
'
'  Copyright (C) 2018  PEAK-System Technik GmbH, Darmstadt
'  more Info at http://www.peak-system.com 
'
Imports System
Imports System.Collections.Generic
Imports System.Text
Imports System.Runtime.InteropServices

Imports Peak.Can.Basic
Imports TPCANHandle = System.UInt16
Imports TPCANBitrateFD = System.String
Imports TPCANTimestampFD = System.UInt64
Imports TXCPChannel = System.UInt16
Imports TXCPHandle = System.UInt32

Namespace Peak.Can.Xcp
    <Flags()> _
    Public Enum TXCPQueue As UInt32
        ''' <summary>
        ''' Queue for packets of kind RES/ERR/EV/SERV
        ''' </summary>
        XCP_CTO_QUEUE = 1
        ''' <summary>
        ''' Queue for packets of kind DAQ
        ''' </summary>
        XCP_DTO_QUEUE = 2
    End Enum

    Public Enum TXCPResult As UInt32
        ' Codes for not sucessfully executed XCP commands
        '
        ''' <summary>
        ''' Command processor synchronization
        ''' </summary>
        XCP_ERR_CMD_SYNCH = 0
        ''' <summary>
        ''' Command was not executed
        ''' </summary>
        XCP_ERR_CMD_BUSY = &H10
        ''' <summary>
        ''' Command rejected because DAQ is running
        ''' </summary>
        XCP_ERR_DAQ_ACTIVE = &H11
        ''' <summary>
        ''' Command rejected because PGM is running
        ''' </summary>
        XCP_ERR_PGM_ACTIVE = &H12
        ''' <summary>
        ''' Unknown command or not implemented optional command
        ''' </summary>
        XCP_ERR_CMD_UNKNOWN = &H20
        ''' <summary>
        ''' Command syntax invalid
        ''' </summary>
        XCP_ERR_CMD_SYNTAX = &H21
        ''' <summary>
        ''' Command syntax valid but command parameter(s) out of range
        ''' </summary>
        XCP_ERR_OUT_OF_RANGE = &H22
        ''' <summary>
        ''' The memory location is write protected
        ''' </summary>
        XCP_ERR_WRITE_PROTECTED = &H23
        ''' <summary>
        ''' The memory location is not accessible
        ''' </summary>
        XCP_ERR_ACCESS_DENIED = &H24
        ''' <summary>
        ''' Access denied,Seed and Key is required
        ''' </summary>
        XCP_ERR_ACCESS_LOCKED = &H25
        ''' <summary>
        ''' Selected page not available
        ''' </summary>
        XCP_ERR_PAGE_NOT_VALID = &H26
        ''' <summary>
        ''' Selected page mode not available
        ''' </summary>
        XCP_RR_MODE_NOT_VALID = &H27
        ''' <summary>
        ''' Selected segment not valid
        ''' </summary>
        XCP_ERR_SEGMENT_NOT_VALID = &H28
        ''' <summary>
        ''' Sequence error
        ''' </summary>
        XCP_ERR_SEQUENCE = &H29
        ''' <summary>
        ''' DAQ configuration not valid
        ''' </summary>
        XCP_ERR_DAQ_CONFIG = &H2A
        ''' <summary>
        ''' Memory overflow error
        ''' </summary>
        XCP_ERR_MEMORY_OVERFLOW = &H30
        ''' <summary>
        ''' Generic error
        ''' </summary>
        XCP_ERR_GENERIC = &H31
        ''' <summary>
        ''' The slave internal program verify routine detects an error
        ''' </summary>
        XCP_ERR_VERIFY = &H32
        ''' <summary>
        ''' Access to the requested resource is temporary not possible
        ''' </summary>
        XCP_ERR_RESOURCE_TEMPORARY_NOT_ACCESSIBLE = &H33

        ' API return error codes
        '
        ''' <summary>
        ''' Acknowledge / no error
        ''' </summary>
        XCP_ERR_OK = (1 << 8)
        ''' <summary>
        ''' Function not available / Operation not implemented
        ''' </summary>
        XCP_ERR_NOT_IMPLEMENTED = (2 << 8)
        ''' <summary>
        ''' Invalid parameter value
        ''' </summary>
        XCP_ERR_INVALID_PARAMETER = (3 << 8)
        ''' <summary>
        ''' The maximum amount of registered Slave channels was reached
        ''' </summary>
        XCP_ERR_MAX_CHANNELS = (4 << 8)
        ''' <summary>
        ''' The given handle is invalid
        ''' </summary>
        XCP_ERROR_INVALID_HANDLE = (5 << 8)
        ''' <summary>
        ''' A timeout was reached by calling a function synchronously
        ''' </summary>
        XCP_ERR_INTERNAL_TIMEOUT = (6 << 8)
        ''' <summary>
        ''' The queue being referred is empty
        ''' </summary>
        XCP_ERR_QUEUE_EMPTY = (7 << 8)
        ''' <summary>
        ''' The size of the given buffer, is not big enough
        ''' </summary>
        XCP_ERR_INSUFFICIENT_BUFFER = (8 << 8)

        ' Transport protocol error flags
        '
        ''' <summary>
        ''' Flag for a specific error within the underlying transport channel 
        ''' </summary>
        XCP_ERR_TRANSPORT_CHANNEL = &H0
    End Enum

    <StructLayout(LayoutKind.Auto, Size:=4)> _
    Public Structure XCPResult
        <MarshalAs(UnmanagedType.U4)> _
        Private m_pxcpResult As UInt32

        Public Sub New(ByVal result As TXCPResult)
            m_pxcpResult = result
        End Sub

        Private Function IsTransportLayerError() As Boolean
            Return (m_pxcpResult And TXCPResult.XCP_ERR_TRANSPORT_CHANNEL) = TXCPResult.XCP_ERR_TRANSPORT_CHANNEL
        End Function

        Public ReadOnly Property XCP() As TXCPResult
            Get
                If IsTransportLayerError() Then
                    Return TXCPResult.XCP_ERR_TRANSPORT_CHANNEL
                Else
                    Return (m_pxcpResult And &H7FFFFFFF)
                End If
            End Get
        End Property

        Public ReadOnly Property TransportLayer() As UInt32
            Get
                Return (m_pxcpResult And &H7FFFFFFF)
            End Get
        End Property

    End Structure

    <StructLayout(LayoutKind.Sequential)> _
    Public Structure TXCPCalibrationPage
        <MarshalAs(UnmanagedType.U1)> _
        Public SegmentNumber As Byte
        <MarshalAs(UnmanagedType.U1)> _
        Public PageNumber As Byte
    End Structure

    <StructLayout(LayoutKind.Sequential)> _
    Public Structure TXCPProgramFormat
        <MarshalAs(UnmanagedType.U1)> _
        Public CompressionMethod As Byte
        <MarshalAs(UnmanagedType.U1)> _
        Public EncryptionMethod As Byte
        <MarshalAs(UnmanagedType.U1)> _
        Public ProgrammingMehtod As Byte
        <MarshalAs(UnmanagedType.U1)> _
        Public AccessMethod As Byte
    End Structure

    <StructLayout(LayoutKind.Sequential)> _
    Public Structure TXCPODTEntry
        <MarshalAs(UnmanagedType.U1)> _
        Public BitOffset As Byte
        <MarshalAs(UnmanagedType.U1)> _
        Public DAQSize As Byte
        <MarshalAs(UnmanagedType.U1)> _
        Public DAQAddressExtension As Byte
        <MarshalAs(UnmanagedType.U4)> _
        Public DAQAddress As UInt32
    End Structure

    <StructLayout(LayoutKind.Sequential)> _
    Public Structure TXCPDAQListConfig
        <MarshalAs(UnmanagedType.U2)> _
        Public DAQListNumber As UInt16
        <MarshalAs(UnmanagedType.U2)> _
        Public EventChannelNumber As UInt16
        <MarshalAs(UnmanagedType.U1)> _
        Public TransmissionRate As Byte
        <MarshalAs(UnmanagedType.U1)> _
        Public DAQPriority As Byte
    End Structure

    <StructLayout(LayoutKind.Sequential)> _
    Public Structure TXCPTransportLayerCAN
        <MarshalAs(UnmanagedType.U4)> _
        Public BroadcastID As UInt32
        <MarshalAs(UnmanagedType.U4)> _
        Public MasterID As UInt32
        <MarshalAs(UnmanagedType.U4)> _
        Public SlaveID As UInt32
        <MarshalAs(UnmanagedType.U1)> _
        Public IncrementalIdUsed As Boolean
    End Structure

    <StructLayout(LayoutKind.Sequential)> _
    Public Structure TXCPProtocolLayerConfig
        <MarshalAs(UnmanagedType.U2)> _
        Public T1 As UInt16
        <MarshalAs(UnmanagedType.U2)> _
        Public T2 As UInt16
        <MarshalAs(UnmanagedType.U2)> _
        Public T3 As UInt16
        <MarshalAs(UnmanagedType.U2)> _
        Public T4 As UInt16
        <MarshalAs(UnmanagedType.U2)> _
        Public T5 As UInt16
        <MarshalAs(UnmanagedType.U2)> _
        Public T6 As UInt16
        <MarshalAs(UnmanagedType.U2)> _
        Public T7 As UInt16
        ''' <summary>
        ''' <remarks>DEPRECATED: Value will automatically adjusted on connect (XCP_Connect)</remarks>
        ''' </summary>
        <MarshalAs(UnmanagedType.U1)> _
        Public MotorolaFormat As Boolean
        ''' <summary>
        ''' <remarks>DEPRECATED: Value will automatically adjusted on connect (XCP_Connect)</remarks>
        ''' </summary>
        <MarshalAs(UnmanagedType.U1)> _
        Public AddressGranularity As Byte
    End Structure

    ''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
    '' PCAN-XCP API function declarations
    ''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
#Region "PCAN-XCP Class"
    ''' <summary>
    ''' PCAN-XCP API class implementation
    ''' </summary>
    Public NotInheritable Class XCPApi
        ''' <summary>
        ''' Maximum count of asynchronous messages that a queue can retain
        ''' </summary>
        Public Const XCP_MAX_QUEUE_SIZE As Integer = &H7FFF

        ''' <summary>
        ''' Maximum length of a CTO packet on CAN
        ''' </summary>
        Public Const CAN_MAX_LEN As Integer = 8
        ''' <summary>
        ''' Maximum length of a CTO packet on CAN-FD
        ''' </summary>
        Public Const CAN_MAX_LEN_FD As Integer = 64

        ''' <summary>
        ''' Maximum length of a CTO packet on standard CAN
        ''' </summary>
        Public Const CAN_MAX_CTO As Integer = CAN_MAX_LEN
        ''' <summary>
        ''' Maximum length of a DTO packet on standard CAN
        ''' </summary>
        Public Const CAN_MAX_DTO As Integer = CAN_MAX_LEN
        ''' <summary>
        ''' Maximum length of a CTO packet on standard CAN, when programming
        ''' </summary>
        Public Const CAN_MAX_CTO_PGM As Integer = CAN_MAX_LEN

        ''' <summary>
        ''' Allocates an XCP Channel for CAN communication using PCAN-Basic API
        ''' </summary>
        ''' <param name="XcpChannel">Buufer for the XcpChannel representing this CAN Channel</param>
        ''' <param name="Channel">The handle of a PCAN Channel to be initialized</param>
        ''' <param name="Btr0Btr1">The speed for the communication (BTR0BTR1 code)</param>
        ''' <param name="HwType">NON PLUG And PLAY: The type of hardware and operation mode</param>
        ''' <param name="IOPort">NON PLUG And PLAY: The I/O address for the parallel port</param>
        ''' <param name="Interrupt">NON PLUG And PLAY: Interrupt number of the parallel port</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_InitializeCanChannel")> _
        Public Shared Function InitializeCanChannel( _
            ByRef XcpChannel As TXCPChannel, _
            ByVal Channel As TPCANHandle, _
            ByVal Btr0Btr1 As TPCANBaudrate, _
            ByVal HwType As TPCANType, _
            ByVal IOPort As UInt32, _
            ByVal Interrupt As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Allocates an XCP Channel for CAN communication using PCAN-Basic API
        ''' </summary>
        ''' <param name="XcpChannel">Buufer for the XcpChannel representing this CAN Channel</param>
        ''' <param name="Channel">The handle of a PCAN Channel to be initialized</param>
        ''' <param name="Btr0Btr1">The speed for the communication (BTR0BTR1 code)</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function InitializeCanChannel( _
            ByRef XcpChannel As TXCPChannel, _
            ByVal Channel As TPCANHandle, _
            ByVal Btr0Btr1 As TPCANBaudrate) As TXCPResult
            Return InitializeCanChannel(XcpChannel, Channel, Btr0Btr1, 0, 0, 0)
        End Function

        ''' <summary>
        ''' Allocates an XCP Channel for CAN-FD communication using PCAN-Basic API
        ''' </summary>
        ''' <param name="XcpChannel">Buufer for the XcpChannel representing this CAN Channel</param>
        ''' <param name="Channel">The handle of a PCAN Channel to be initialized</param>
        ''' <param name="Bitrate">The speed for the communication (FD parameters)</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_InitializeCanChannelFD")> _
        Public Shared Function InitializeCanChannelFD( _
            ByRef XcpChannel As TXCPChannel, _
            ByVal Channel As TPCANHandle, _
            ByVal Bitrate As TPCANBitrateFD) As TXCPResult
        End Function

        ''' <summary>
        ''' Uninitializes a XCP Channel
        ''' </summary>
        ''' <param name="XcpChannel">The handle of a Xcp Channel</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_UninitializeChannel")> _
        Public Shared Function UninitializeChannel( _
            ByVal XcpChannel As TXCPChannel) As TXCPResult
        End Function

        ''' <summary>
        ''' Associates an ECU to a XCP Channel to communicate over CAN.
        ''' </summary>
        ''' <param name="XcpChannel">The handle of a Xcp Channel</param>
        ''' <param name="SlaveData">The slave (ECU) data for CAN communication</param>
        ''' <param name="Protocol">Protocol layer's configuration data </param>
        ''' <param name="XcpHandle">Buffer for the handle representing the session (Channel + Slave)</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_AddSlaveOnCAN")> _
        Public Shared Function AddSlaveOnCAN( _
            ByVal XcpChannel As TXCPChannel, _
            <MarshalAs(UnmanagedType.Struct)> _
            ByVal SlaveData As TXCPTransportLayerCAN, _
            ByVal Protocol As TXCPProtocolLayerConfig, _
            ByRef XcpHandle As TXCPHandle) As TXCPResult
        End Function

        ''' <summary>
        ''' Remove the assocuiation between an ECU and a XCP Channel.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of the XCP session</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_RemoveSlave")> _
        Public Shared Function RemoveSlave( _
            ByVal XcpHandle As TXCPHandle) As TXCPResult
        End Function

        ''' <summary>
        ''' Retrieves an enqueued CTO or DTO packet from the XCP session represented
        ''' by the given XcpHandle.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of the XCP session</param>
        ''' <param name="queueSelect">A TXCPQueue value that identifies the queue from 
        ''' where the packet should be extracted.</param>
        ''' <param name="dtoBuffer">Buffer for the CTO or DTO packet.</param>
        ''' <param name="dtoBufferLength">Length of the packet buffer</param>
        ''' <returns></returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_DequeuePacket")> _
        Public Shared Function DequeuePacket( _
            ByVal XcpHandle As TXCPHandle, _
            <MarshalAs(UnmanagedType.U4)> _
            ByVal queueSelect As TXCPQueue, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=3)> _
            ByVal dtoBuffer As Byte(), _
            ByVal dtoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Resets the CTO/DTO queue(s) of a XCP session represented 
        ''' by the given XcpHandle
        ''' </summary>
        ''' <param name="XcpHandle">The handle of the XCP session</param>
        ''' <param name="QueueToClear">Value indicating the queue to be reset</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_ResetQueue")> _
        Public Shared Function ResetQueue( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal QueueToClear As TXCPQueue) As TXCPResult
        End Function

        ''------------------------------
        '' Commands
        ''------------------------------

        ''' <summary>
        ''' Command CONNECT: establishes a continuous, logical, point-to-point 
        ''' connection with a slave device.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Mode">0: Normal, 1: User defined</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_Connect")> _
        Public Shared Function Connect( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Mode As Byte, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=3)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command CONNECT: establishes a continuous, logical, point-to-point 
        ''' connection with a slave device.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Mode">0: Normal, 1: User defined</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function Connect( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Mode As Byte) As TXCPResult
            Return Connect(XcpHandle, Mode, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command DISCONNECT: brings the slave to the “DISCONNECTED” state.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_Disconnect")> _
        Public Shared Function Disconnect( _
            ByVal XcpHandle As TXCPHandle, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=2)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command DISCONNECT: brings the slave to the “DISCONNECTED” state.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function Disconnect( _
            ByVal XcpHandle As TXCPHandle) As TXCPResult
            Return Disconnect(XcpHandle, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command GET_STATUS: returns all current status information of the slave device.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_GetStatus")> _
        Public Shared Function GetStatus( _
            ByVal XcpHandle As TXCPHandle, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=2)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function


        ''' <summary>
        ''' Command GET_STATUS: returns all current status information of the slave device.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function GetStatus( _
            ByVal XcpHandle As TXCPHandle) As TXCPResult
            Return GetStatus(XcpHandle, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command SYNCH: used to synchronize command execution after timeout conditions.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_Synchronize")> _
        Public Shared Function Synchronize( _
             ByVal XcpHandle As TXCPHandle, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=2)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command SYNCH: used to synchronize command execution after timeout conditions.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function Synchronize( _
            ByVal XcpHandle As TXCPHandle) As TXCPResult
            Return Synchronize(XcpHandle, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command GET_COMM_MODE_INFO: returns optional information on different 
        ''' Communication Modes supported by the slave.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_GetCommunicationMode")> _
        Public Shared Function GetCommunicationMode( _
             ByVal XcpHandle As TXCPHandle, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=2)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function


        ''' <summary>
        ''' Command GET_COMM_MODE_INFO: returns optional information on different 
        ''' Communication Modes supported by the slave.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function GetCommunicationMode( _
            ByVal XcpHandle As TXCPHandle) As TXCPResult
            Return GetCommunicationMode(XcpHandle, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command GET_ID: used for automatic session configuration and for 
        ''' slave device identification.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Type">The ideintification type (0..4)</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_GetIdentification")> _
        Public Shared Function GetIdentification( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Type As Byte, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=3)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As Byte) As TXCPResult
        End Function

        ''' <summary>
        ''' Command GET_ID: used for automatic session configuration and for 
        ''' slave device identification.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Type">The ideintification type (0..4)</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function GetIdentification( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Type As Byte) As TXCPResult
            Return GetIdentification(XcpHandle, Type, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command SET_REQUEST: request to save to non-volatile memory.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Mode">Request mode especification</param>
        ''' <param name="SessionId">Session Configuration identification</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_SetRequest")> _
        Public Shared Function SetRequest( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Mode As Byte, _
            ByVal SessionId As UInt16, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=4)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command SET_REQUEST: request to save to non-volatile memory.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Mode">Request mode especification</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function SetRequest( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Mode As Byte, _
            ByVal SessionId As UInt16) As TXCPResult
            Return SetRequest(XcpHandle, Mode, SessionId, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command GET_SEED: Get seed for unlocking a protected resource.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Mode">Seed mode. 0: First part, 1: remaining part</param>
        ''' <param name="Resource">Resource id, when Mode is 0. Otherwise don't care</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_GetSeed")> _
        Public Shared Function GetSeed( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Mode As Byte, _
            ByVal Resource As Byte, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=4)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command GET_SEED: Get seed for unlocking a protected resource.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Mode">Seed mode. 0: First part, 1: remaining part</param>
        ''' <param name="Resource">Resource id, when Mode is 0. Otherwise don't care</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function GetSeed( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Mode As Byte, _
            ByVal Resource As Byte) As TXCPResult
            Return GetSeed(XcpHandle, Mode, Resource, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command UNLOCK: unlocks the slave device’s security protection using a 
        ''' ´key´ computed from the ´seed´ obtained by a previous GET_SEED sequence.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="KeyLength">Indicates the (remaining) number of key bytes</param>
        ''' <param name="Key">Buffer for the Key's data bytes</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_Unlock")> _
        Public Shared Function Unlock( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal KeyLength As Byte, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=1)> _
            ByVal Key As Byte(), _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=4)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command UNLOCK: unlocks the slave device’s security protection using a 
        ''' ´key´ computed from the ´seed´ obtained by a previous GET_SEED sequence.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Key">Buffer for the Key's data bytes</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function Unlock( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Key As Byte()) As TXCPResult
            Return Unlock(XcpHandle, Key.Length, Key, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command SET_MTA: initializes a pointer (32Bit address + 8Bit extension) 
        ''' for following memory transfer commands: BUILD_CHECKSUM, UPLOAD, DOWNLOAD,
        ''' DOWNLOAD_NEXT, DOWNLOAD_MAX, MODIFY_BITS, PROGRAM_CLEAR, PROGRAM, 
        ''' PROGRAM_NEXT and PROGRAM_MAX.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="AddrExtension">Address extension</param>
        ''' <param name="Addr">Address</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_SetMemoryTransferAddress")> _
        Public Shared Function SetMemoryTransferAddress( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal AddrExtension As Byte, _
            ByVal Addr As UInt32, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=4)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command SET_MTA: initializes a pointer (32Bit address + 8Bit extension) 
        ''' for following memory transfer commands: BUILD_CHECKSUM, UPLOAD, DOWNLOAD,
        ''' DOWNLOAD_NEXT, DOWNLOAD_MAX, MODIFY_BITS, PROGRAM_CLEAR, PROGRAM, 
        ''' PROGRAM_NEXT and PROGRAM_MAX.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="AddrExtension">Address extension</param>
        ''' <param name="Addr">Address</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function SetMemoryTransferAddress( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal AddrExtension As Byte, _
            ByVal Addr As UInt32) As TXCPResult
            Return SetMemoryTransferAddress(XcpHandle, AddrExtension, Addr, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command UPLOAD: Upload a data block from slave to master, starting at the 
        ''' current MTA. The MTA will be post-incremented by the given number of data elements.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="NumberOfElements">The amount of elements to retrieve</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_Upload")> _
        Public Shared Function Upload( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal NumberOfElements As Byte, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=3)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command UPLOAD: Upload a data block from slave to master, starting at the 
        ''' current MTA. The MTA will be post-incremented by the given number of data elements.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="NumberOfElements">The amount of elements to retrieve</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function Upload( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal NumberOfElements As Byte) As TXCPResult
            Return Upload(XcpHandle, NumberOfElements, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command SHORT_UPLOAD: Upload a data block, of a maximum size of [1..MAX_CTO/AG], 
        ''' from slave to master. 
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="NumberOfElements">The amount of elements to retrieve</param>
        ''' <param name="AddrExtension">Address extension to set the MTA pointer</param>
        ''' <param name="Addr">Address to set the MTA pointer</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_ShortUpload")> _
        Public Shared Function ShortUpload( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal NumberOfElements As Byte, _
            ByVal AddrExtension As Byte, _
            ByVal Addr As UInt32, _
             <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=5)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command SHORT_UPLOAD: Upload a data block, of a maximum size of [1..MAX_CTO/AG], 
        ''' from slave to master. 
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="NumberOfElements">The amount of elements to retrieve</param>
        ''' <param name="AddrExtension">Address extension to set the MTA pointer</param>
        ''' <param name="Addr">Address to set the MTA pointer</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function ShortUpload( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal NumberOfElements As Byte, _
            ByVal AddrExtension As Byte, _
            ByVal Addr As UInt32) As TXCPResult
            Return ShortUpload(XcpHandle, NumberOfElements, AddrExtension, Addr, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command BUILD_CHECKSUM: Build a checksum over a memory range starting at the MTA.
        ''' The MTA will be post-incremented by the block size.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="BlockSize">The number of elements used for the checksum</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_BuildChecksum")> _
        Public Shared Function BuildChecksum( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal BlockSize As UInt32, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=3)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command BUILD_CHECKSUM: Build a checksum over a memory range starting at the MTA.
        ''' The MTA will be post-incremented by the block size.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="BlockSize">The number of elements used for the checksum</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function BuildChecksum( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal BlockSize As UInt32) As TXCPResult
            Return BuildChecksum(XcpHandle, BlockSize, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command TRANSPORT_LAYER_CMD: sends a specific Transport Layer command.
        ''' For CAN: Works only in blocking mode (ctoBuffer cannot be NULL)
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="SubCommand">Transport layer specific command</param>
        ''' <param name="Parameters">Command parameters buffer</param>
        ''' <param name="ParametersLength">Length of the Command parameters buffer</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_TransportLayerCommand")> _
        Public Shared Function TransportLayerCommand( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal SubCommand As Byte, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=3)> _
            ByVal Parameters As Byte(), _
            ByVal ParametersLength As Byte, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=5)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command TRANSPORT_LAYER_CMD: sends a specific Transport Layer command.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="SubCommand">Transport layer specific command</param>
        ''' <param name="Parameters">Command parameters buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function TransportLayerCommand( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal SubCommand As Byte, _
            ByVal Parameters As Byte()) As TXCPResult
            Return TransportLayerCommand(XcpHandle, SubCommand, Parameters, Parameters.Length, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command USER_CMD: sends a user specific command.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="SubCommand">User defined command</param>
        ''' <param name="Parameters">Command parameters buffer</param>
        ''' <param name="ParametersLength">Length of the Command parameters buffer</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_UserCommand")> _
        Public Shared Function UserCommand( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal SubCommand As Byte, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=3)> _
            ByVal Parameters As Byte(), _
            ByVal ParametersLength As Byte, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=5)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function


        ''' <summary>
        ''' Command USER_CMD: sends a user specific command.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="SubCommand">User defined command</param>
        ''' <param name="Parameters">Command parameters buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function UserCommand( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal SubCommand As Byte, _
            ByVal Parameters As Byte()) As TXCPResult
            Return UserCommand(XcpHandle, SubCommand, Parameters, Parameters.Length, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command DOWNLOAD: Download a data block from master to slave, starting at the 
        ''' current MTA. The MTA will be post-incremented by the number of data elements.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="NumberOfElements">The amount of elements to be transfered</param>
        ''' <param name="Data">Data buffer with the elements to transfer</param>
        ''' <param name="DataLength">Length of the data buffer</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_Download")> _
        Public Shared Function Download( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal NumberOfElements As Byte, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=3)> ByVal _
            Data As Byte(), _
            ByVal DataLength As Byte, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=5)> ByVal _
            ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command DOWNLOAD: Download a data block from master to slave, starting at the 
        ''' current MTA. The MTA will be post-incremented by the number of data elements.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="NumberOfElements">The amount of elements to be transfered</param>
        ''' <param name="Data">Data buffer with the elements to transfer</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function Download( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal NumberOfElements As Byte, _
            ByVal Data As Byte()) As TXCPResult
            Return Download(XcpHandle, NumberOfElements, Data, Data.Length, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command DOWNLOAD_NEXT: Download consecutive data blocks from master to 
        ''' slave (Block Mode), starting at the current MTA. The MTA will be post-incremented 
        ''' by the number of data elements.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="NumberOfElements">The amount of elements to be transfered</param>
        ''' <param name="Data">Data buffer with the elements to transfer</param>
        ''' <param name="DataLength">Length of the data buffer</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_DownloadNext")> _
        Public Shared Function DownloadNext( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal NumberOfElements As Byte, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=3)> ByVal _
            Data As Byte(), _
            ByVal DataLength As Byte, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=5)> ByVal _
            ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command DOWNLOAD_NEXT: Download consecutive data blocks from master to 
        ''' slave (Block Mode), starting at the current MTA. The MTA will be post-incremented 
        ''' by the number of data elements.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="NumberOfElements">The amount of elements to be transfered</param>
        ''' <param name="Data">Data buffer with the elements to transfer</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function DownloadNext( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal NumberOfElements As Byte, _
            ByVal Data As Byte()) As TXCPResult
            Return DownloadNext(XcpHandle, NumberOfElements, Data, Data.Length, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command DOWNLOAD_MAX: Download a data block with fixed size (MAX_CTO/AG-1) from 
        ''' master to slave, starting at the current MTA. The MTA will be post-incremented 
        ''' by MAX_CTO/AG-1.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Data">Data buffer with the elements to transfer</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_DownloadMax")> _
        Public Shared Function DownloadMax( _
            ByVal XcpHandle As TXCPHandle, _
            <MarshalAs(UnmanagedType.LPArray)> ByVal _
            Data As Byte(), _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=3)> ByVal _
            ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function


        ''' <summary>
        ''' Command DOWNLOAD_MAX: Download a data block with fixed size (MAX_CTO/AG-1) from 
        ''' master to slave, starting at the current MTA. The MTA will be post-incremented 
        ''' by MAX_CTO/AG-1.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Data">Data buffer with the elements to transfer</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function DownloadMax( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Data As Byte()) As TXCPResult
            Return DownloadMax(XcpHandle, Data, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command MODIFY_BITS: modifies the bits of the 32 Bit memory 
        ''' location referred by the MTA.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="ShiftValue">Shift Value</param>
        ''' <param name="ANDMask">AND Mask (MA)</param>
        ''' <param name="XORMask">XOR Mask (MX)</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_ModifyBits")> _
        Public Shared Function ModifyBits( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal ShiftValue As Byte, _
            ByVal ANDMask As UInt16, _
            ByVal XORMask As UInt16, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=5)> ByVal _
            ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command MODIFY_BITS: modifies the bits of the 32 Bit memory 
        ''' location referred by the MTA.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="ShiftValue">Shift Value</param>
        ''' <param name="ANDMask">AND Mask (MA)</param>
        ''' <param name="XORMask">XOR Mask (MX)</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function ModifyBits( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal ShiftValue As Byte, _
            ByVal ANDMask As UInt16, _
            ByVal XORMask As UInt16) As TXCPResult
            Return ModifyBits(XcpHandle, ShiftValue, ANDMask, XORMask, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command SET_CAL_PAGE: sets the access mode for a calibration data segment.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Mode">Access mode</param>
        ''' <param name="Page">Represents a calibration data page (segment and page number)</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_SetCalibrationPage")> _
        Public Shared Function SetCalibrationPage( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Mode As Byte, _
            ByVal Page As TXCPCalibrationPage, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=4)> ByVal _
            ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command SET_CAL_PAGE: sets the access mode for a calibration data segment.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Mode">Access mode</param>
        ''' <param name="Page">Represents a calibration data page (segment and page number)</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function SetCalibrationPage( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Mode As Byte, _
            ByVal Page As TXCPCalibrationPage) As TXCPResult
            Return SetCalibrationPage(XcpHandle, Mode, Page, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command GET_CAL_PAGE: returns the logical number for the calibration data page 
        ''' that is currently activated for the specified access mode and data segment.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Mode">Access mode</param>
        ''' <param name="DataSegmentNumber">Logical data segment number</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_GetCalibrationPage")> _
        Public Shared Function GetCalibrationPage( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Mode As Byte, _
            ByVal DataSegmentNumber As Byte, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=4)> ByVal _
            ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command GET_CAL_PAGE: returns the logical number for the calibration data page 
        ''' that is currently activated for the specified access mode and data segment.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Mode">Access mode</param>
        ''' <param name="DataSegmentNumber">Logical data segment number</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function GetCalibrationPage( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Mode As Byte, _
            ByVal DataSegmentNumber As Byte) As TXCPResult
            Return GetCalibrationPage(XcpHandle, Mode, DataSegmentNumber, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command GET_PAG_PROCESSOR_INFO: returns general information on paging processor.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_GetPagingProcessorInformation")> _
        Public Shared Function GetPagingProcessorInformation( _
            ByVal XcpHandle As TXCPHandle, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=2)> ByVal _
            ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command GET_PAG_PROCESSOR_INFO: returns general information on paging processor.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function GetPagingProcessorInformation( _
            ByVal XcpHandle As TXCPHandle) As TXCPResult
            Return GetPagingProcessorInformation(XcpHandle, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command GET_SEGMENT_INFO: returns information on a specific SEGMENT.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Mode">Information mode (0..2)</param>
        ''' <param name="SegmentNumber">Segment number, [0,1,..MAX_SEGMENT-1]</param>
        ''' <param name="SegmentInfo">Segment information according to the 'information mode'</param>
        ''' <param name="MappingIndex">Identifier for address mapping range</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_GetSegmentInformation")> _
        Public Shared Function GetSegmentInformation( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Mode As Byte, _
            ByVal SegmentNumber As Byte, _
            ByVal SegmentInfo As Byte, _
            ByVal MappingIndex As Byte, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=6)> ByVal _
            ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command GET_SEGMENT_INFO: returns information on a specific SEGMENT.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Mode">Information mode (0..2)</param>
        ''' <param name="SegmentNumber">Segment number, [0,1,..MAX_SEGMENT-1]</param>
        ''' <param name="SegmentInfo">Segment information according to the 'information mode'</param>
        ''' <param name="MappingIndex">Identifier for address mapping range</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function GetSegmentInformation( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Mode As Byte, _
            ByVal SegmentNumber As Byte, _
            ByVal SegmentInfo As Byte, _
            ByVal MappingIndex As Byte) As TXCPResult
            Return GetSegmentInformation(XcpHandle, Mode, SegmentNumber, SegmentInfo, MappingIndex, Nothing, 0)
        End Function


        ''' <summary>
        ''' Command GET_PAGE_INFO: returns information on a specific PAGE.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Page">Represents a calibration data page (segment and page number)</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_GetPageInformation")> _
        Public Shared Function GetPageInformation( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Page As TXCPCalibrationPage, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=3)> ByVal _
            ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command GET_PAGE_INFO: returns information on a specific PAGE.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Page">Represents a calibration data page (segment and page number)</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function GetPageInformation( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Page As TXCPCalibrationPage) As TXCPResult
            Return GetPageInformation(XcpHandle, Page, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command SET_SEGMENT_MODE: set the mode for a SEGMENT
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Mode">Segment mode (0,1)</param>
        ''' <param name="segmentNumber">Segment number, [0,1,..MAX_SEGMENT-1]</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_SetSegmentMode")> _
        Public Shared Function SetSegmentMode( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Mode As Byte, _
            ByVal SegmentNumber As Byte, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=4)> ByVal _
            ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command SET_SEGMENT_MODE: set the mode for a SEGMENT
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Mode">Segment mode (0,1)</param>
        ''' <param name="segmentNumber">Segment number, [0,1,..MAX_SEGMENT-1]</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function SetSegmentMode( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Mode As Byte, _
            ByVal SegmentNumber As Byte) As TXCPResult
            Return SetSegmentMode(XcpHandle, Mode, SegmentNumber, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command GET_SEGMENT_MODE: get the mode for a SEGMENT
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="SegmentNumber">Segment number, [0,1,..MAX_SEGMENT-1]</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_GetSegmentMode")> _
        Public Shared Function GetSegmentMode( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal SegmentNumber As Byte, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=3)> ByVal _
            ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command GET_SEGMENT_MODE: get the mode for a SEGMENT
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="SegmentNumber">Segment number, [0,1,..MAX_SEGMENT-1]</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function GetSegmentMode( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal SegmentNumber As Byte) As TXCPResult
            Return GetSegmentMode(XcpHandle, SegmentNumber, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command COPY_CAL_PAGE: forces the slave to copy one calibration page to another.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Source">Represents the source calibration data page</param>
        ''' <param name="Destination">Represents the destination calibration data page</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_CopyCalibrationPage")> _
        Public Shared Function CopyCalibrationPage( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Source As TXCPCalibrationPage, _
            ByVal Destination As TXCPCalibrationPage, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=4)> ByVal _
            ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command COPY_CAL_PAGE: forces the slave to copy one calibration page to another.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Source">Represents the source calibration data page</param>
        ''' <param name="Destination">Represents the destination calibration data page</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function CopyCalibrationPage( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Source As TXCPCalibrationPage, _
            ByVal Destination As TXCPCalibrationPage) As TXCPResult
            Return CopyCalibrationPage(XcpHandle, Source, Destination, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command SET_DAQ_PTR: initializes the DAQ list pointer for a subsequent 
        ''' operation with XCP_WriteDAQListEntry (WRITE_DAQ) or XCP_ReadDAQListEntry (READ_DAQ).
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="DaqListNumber">DAQ List number, [0,1,..MAX_DAQ-1]</param>
        ''' <param name="OdtNumber">Relative ODT number within this DAQ list</param>
        ''' <param name="OdtEntry">Relative ODT entry number within this ODT</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_SetDAQListPointer")> _
        Public Shared Function SetDAQListPointer( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal DaqListNumber As UInt16, _
            ByVal OdtNumber As Byte, _
            ByVal OdtEntry As Byte, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=5)> ByVal _
            ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command SET_DAQ_PTR: initializes the DAQ list pointer for a subsequent 
        ''' operation with XCP_WriteDAQListEntry (WRITE_DAQ) or XCP_ReadDAQListEntry (READ_DAQ).
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="DaqListNumber">DAQ List number, [0,1,..MAX_DAQ-1]</param>
        ''' <param name="OdtNumber">Relative ODT number within this DAQ list</param>
        ''' <param name="OdtEntry">Relative ODT entry number within this ODT</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function SetDAQListPointer( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal DaqListNumber As UInt16, _
            ByVal OdtNumber As Byte, _
            ByVal OdtEntry As Byte) As TXCPResult
            Return SetDAQListPointer(XcpHandle, DaqListNumber, OdtNumber, OdtEntry, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command WRITE_DAQ: writes one ODT entry to a DAQ list defined by 
        ''' the DAQ list pointer.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Entry">Represents an ODT entry for a DAQ List</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_WriteDAQListEntry")> _
        Public Shared Function WriteDAQListEntry( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Entry As TXCPODTEntry, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=3)> ByVal _
            ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command WRITE_DAQ: writes one ODT entry to a DAQ list defined by 
        ''' the DAQ list pointer.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Entry">Represents an ODT entry for a DAQ List</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function WriteDAQListEntry( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Entry As TXCPODTEntry) As TXCPResult
            Return WriteDAQListEntry(XcpHandle, Entry, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command SET_DAQ_LIST_MODE: sets the mode for DAQ list.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Mode">DAQ list mode bit</param>
        ''' <param name="Configuration">Represents the configuration of a DAQ List</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_SetDAQListMode")> _
        Public Shared Function SetDAQListMode( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Mode As Byte, _
            ByVal Configuration As TXCPDAQListConfig, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=4)> ByVal _
            ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command SET_DAQ_LIST_MODE: sets the mode for DAQ list.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Mode">DAQ list mode bit</param>
        ''' <param name="Configuration">Represents the configuration of a DAQ List</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function SetDAQListMode( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Mode As Byte, _
            ByVal Configuration As TXCPDAQListConfig) As TXCPResult
            Return SetDAQListMode(XcpHandle, Mode, Configuration, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command START_STOP_DAQ_LIST: Starts, stops, or selects a DAQ list.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Mode">0: Stop, 1: Start, 2: Select</param>
        ''' <param name="DaqListNumber">DAQ list number, [0,1,..MAX_DAQ-1]</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_StartStopDAQList")> _
        Public Shared Function StartStopDAQList( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Mode As Byte, _
            ByVal DaqListNumber As UInt16, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=4)> ByVal _
            ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command START_STOP_DAQ_LIST: Starts, stops, or selects a DAQ list.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Mode">0: Stop, 1: Start, 2: Select</param>
        ''' <param name="DaqListNumber">DAQ list number, [0,1,..MAX_DAQ-1]</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function StartStopDAQList( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Mode As Byte, _
            ByVal DaqListNumber As UInt16) As TXCPResult
            Return StartStopDAQList(XcpHandle, Mode, DaqListNumber, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command START_STOP_SYNCH: Starts or stops several DAQ lists, synchronously.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Mode">0: Stop all, 1: Start selected, 2: Stop selected</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_StartStopSynchronizedDAQList")> _
        Public Shared Function StartStopSynchronizedDAQList( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Mode As Byte, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=3)> ByVal _
            ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command START_STOP_SYNCH: Starts, stops, or selects several DAQ lists, synchronously.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Mode">0: Stop all, 1: Start selected, 2: Stop selected</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function StartStopSynchronizedDAQList( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Mode As Byte) As TXCPResult
            Return StartStopSynchronizedDAQList(XcpHandle, Mode, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command WRITE_DAQ_MULTIPLE: writes consecutive ODT entries to a DAQ list 
        ''' defined by the DAQ list pointer.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="NumberOfElements">Amount of ODT entries to be written</param>
        ''' <param name="Elements">Buffer containing the ODT entries for a DAQ List</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_WriteDAQListEntries")> _
        Public Shared Function WriteDAQListEntries( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal NumberOfElements As Byte, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=1)> _
            ByVal Elements As TXCPODTEntry(), _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=4)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command WRITE_DAQ_MULTIPLE: writes consecutive ODT entries to a DAQ list 
        ''' defined by the DAQ list pointer.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Elements">Buffer containing the ODT entries for a DAQ List</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function WriteDAQListEntries( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Elements As TXCPODTEntry()) As TXCPResult
            Return WriteDAQListEntries(XcpHandle, Elements.Length, Elements, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command READ_DAQ: Reads one ODT entry of a DAQ list defined by the DAQ 
        ''' list pointer.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_ReadDAQListEntry")> _
        Public Shared Function ReadDAQListEntry( _
            ByVal XcpHandle As TXCPHandle, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=2)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command READ_DAQ: Reads one ODT entry of a DAQ list defined by the DAQ 
        ''' list pointer.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function ReadDAQListEntry( _
            ByVal XcpHandle As TXCPHandle) As TXCPResult
            Return ReadDAQListEntry(XcpHandle, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command GET_DAQ_CLOCK: Gets the DAQ clock from a slave.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_GetDAQClock")> _
        Public Shared Function GetDAQClock( _
            ByVal XcpHandle As TXCPHandle, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=2)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command GET_DAQ_CLOCK: Gets the DAQ clock from a slave.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function GetDAQClock( _
            ByVal XcpHandle As TXCPHandle) As TXCPResult
            Return GetDAQClock(XcpHandle, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command GET_DAQ_PROCESSOR_INFO: returns general information on DAQ processor.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_GetDAQProcessorInformation")> _
        Public Shared Function GetDAQProcessorInformation( _
            ByVal XcpHandle As TXCPHandle, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=2)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command GET_DAQ_PROCESSOR_INFO: returns general information on DAQ processor.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function GetDAQProcessorInformation( _
            ByVal XcpHandle As TXCPHandle) As TXCPResult
            Return GetDAQProcessorInformation(XcpHandle, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command GET_DAQ_RESOLUTION_INFO: returns information on DAQ 
        ''' processing resolution.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_GetDAQResolutionInformation")> _
        Public Shared Function GetDAQResolutionInformation( _
            ByVal XcpHandle As TXCPHandle, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=2)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command GET_DAQ_RESOLUTION_INFO: returns information on DAQ 
        ''' processing resolution.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function GetDAQResolutionInformation( _
            ByVal XcpHandle As TXCPHandle) As TXCPResult
            Return GetDAQResolutionInformation(XcpHandle, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command GET_DAQ_LIST_MODE: returns information on the current mode 
        ''' of a specified DAQ list.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="DaqListNumber">DAQ list number, [0,1,..MAX_DAQ-1]</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_GetDAQListMode")> _
        Public Shared Function GetDAQListMode( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal DaqListNumber As UInt16, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=3)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command GET_DAQ_LIST_MODE: returns information on the current mode 
        ''' of a specified DAQ list.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="DaqListNumber">DAQ list number, [0,1,..MAX_DAQ-1]</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function GetDAQListMode( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal DaqListNumber As UInt16) As TXCPResult
            Return GetDAQListMode(XcpHandle, DaqListNumber, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command GET_DAQ_EVENT_INFO: returns information on a specific event channel.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="EventChannelNumber">Event channel number, [0,1,..MAX_EVENT_CHANNEL-1]</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_GetEventChannelInformation")> _
        Public Shared Function GetEventChannelInformation( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal EventChannelNumber As UInt16, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=3)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command GET_DAQ_EVENT_INFO: returns information on a specific event channel.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="EventChannelNumber">Event channel number, [0,1,..MAX_EVENT_CHANNEL-1]</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function GetEventChannelInformation( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal EventChannelNumber As UInt16) As TXCPResult

            Return GetEventChannelInformation(XcpHandle, EventChannelNumber, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command CLEAR_DAQ_LIST: clears an specific DAQ list configuration.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="DaqListNumber">DAQ list number, [0,1..MAX_DAQ-1]</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_ClearDAQList")> _
        Public Shared Function ClearDAQList( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal DaqListNumber As UInt16, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=3)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command CLEAR_DAQ_LIST: clears an specific DAQ list configuration.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="DaqListNumber">DAQ list number, [0,1..MAX_DAQ-1]</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function ClearDAQList( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal DaqListNumber As UInt16) As TXCPResult
            Return ClearDAQList(XcpHandle, DaqListNumber, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command GET_DAQ_LIST_INFO: returns information on a specific DAQ list.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="DaqListNumber">DAQ list number, [0,1..MAX_DAQ-1]</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_GetDAQListInformation")> _
        Public Shared Function GetDAQListInformation( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal DaqListNumber As UInt16, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=3)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command GET_DAQ_LIST_INFO: returns information on a specific DAQ list.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="DaqListNumber">DAQ list number, [0,1..MAX_DAQ-1]</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function GetDAQListInformation( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal DaqListNumber As UInt16) As TXCPResult
            Return GetDAQListInformation(XcpHandle, DaqListNumber, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command FREE_DAQ: clears all DAQ lists and frees all dynamically allocated 
        ''' DAQ lists, ODTs and ODT entries.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_FreeDAQLists")> _
        Public Shared Function FreeDAQLists( _
            ByVal XcpHandle As TXCPHandle, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=2)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command FREE_DAQ: clears all DAQ lists and frees all dynamically allocated 
        ''' DAQ lists, ODTs and ODT entries.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function FreeDAQLists( _
            ByVal XcpHandle As TXCPHandle) As TXCPResult
            Return FreeDAQLists(XcpHandle, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command ALLOC_DAQ: allocates a number of DAQ lists for a XCP slave device.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="DaqCount">Number of DAQ lists to be allocated</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_AllocateDAQLists")> _
        Public Shared Function AllocateDAQLists( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal DaqCount As UInt16, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=3)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command ALLOC_DAQ: allocates a number of DAQ lists for a XCP slave device.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="DaqCount">Number of DAQ lists to be allocated</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function AllocateDAQLists( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal DaqCount As UInt16) As TXCPResult
            Return AllocateDAQLists(XcpHandle, DaqCount, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command ALLOC_ODT: allocates a number of ODTs and assigns them to the 
        ''' specified DAQ list.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="DaqListNumber">DAQ list number, [MIN_DAQ, MIN_DAQ+1,..MIN_DAQ+DAQ_COUNT-1]</param>
        ''' <param name="OdtCount">Number of ODTs to be assigned to a DAQ list</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_AllocateODT")> _
        Public Shared Function AllocateODT( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal DaqListNumber As UInt16, _
            ByVal OdtCount As Byte, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=4)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command ALLOC_ODT: allocates a number of ODTs and assigns them to the 
        ''' specified DAQ list.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="DaqListNumber">DAQ list number, [MIN_DAQ, MIN_DAQ+1,..MIN_DAQ+DAQ_COUNT-1]</param>
        ''' <param name="OdtCount">Number of ODTs to be assigned to a DAQ list</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function AllocateODT( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal DaqListNumber As UInt16, _
            ByVal OdtCount As Byte) As TXCPResult
            Return AllocateODT(XcpHandle, DaqListNumber, OdtCount, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command ALLOC_ODT_ENTRY: allocates a number of ODT entries and assigns them to 
        ''' the specific ODT in this specific DAQ list.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="DaqListNumber">DAQ list number, [MIN_DAQ, MIN_DAQ+1,..MIN_DAQ+DAQ_COUNT-1]</param>
        ''' <param name="OdtNumber">Relative ODT number, [0,1,..ODT_COUNT(DAQ list)-1]</param>
        ''' <param name="EntriesCount">Number of ODT entries to be assigned to ODT</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_AllocateODTEntry")> _
        Public Shared Function AllocateODTEntry( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal DaqListNumber As UInt16, _
            ByVal OdtNumber As Byte, _
            ByVal EntriesCount As Byte, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=5)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command ALLOC_ODT_ENTRY: allocates a number of ODT entries and assigns them to 
        ''' the specific ODT in this specific DAQ list.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="DaqListNumber">DAQ list number, [MIN_DAQ, MIN_DAQ+1,..MIN_DAQ+DAQ_COUNT-1]</param>
        ''' <param name="OdtNumber">Relative ODT number, [0,1,..ODT_COUNT(DAQ list)-1]</param>
        ''' <param name="EntriesCount">Number of ODT entries to be assigned to ODT</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function AllocateODTEntry( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal DaqListNumber As UInt16, _
            ByVal OdtNumber As Byte, _
            ByVal EntriesCount As Byte) As TXCPResult
            Return AllocateODTEntry(XcpHandle, DaqListNumber, OdtNumber, EntriesCount, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command PROGRAM_START: indicates the begin of a non-volatile memory programming
        ''' sequence.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_ProgramStart")> _
        Public Shared Function ProgramStart( _
            ByVal XcpHandle As TXCPHandle, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=2)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command PROGRAM_START: indicates the begin of a non-volatile memory programming
        ''' sequence.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function ProgramStart( _
            ByVal XcpHandle As TXCPHandle) As TXCPResult
            Return ProgramStart(XcpHandle, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command PROGRAM_CLEAR: clears a part of non-volatile memory prior to reprogramming.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Mode">Access mode, 0: absolute access, 1: functional access</param>
        ''' <param name="ClearRange">Memory range to clear according to the access mode</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_ProgramClear")> _
        Public Shared Function ProgramClear( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Mode As Byte, _
            ByVal ClearRange As UInt32, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=4)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command PROGRAM_CLEAR: clears a part of non-volatile memory prior to reprogramming.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Mode">Access mode, 0: absolute access, 1: functional access</param>
        ''' <param name="ClearRange">Memory range to clear according to the access mode</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function ProgramClear( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Mode As Byte, _
            ByVal ClearRange As UInt32) As TXCPResult
            Return ProgramClear(XcpHandle, Mode, ClearRange, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command PROGRAM: Programs a non-volatile memory segment inside the slave.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="NumberOfElements">Number of data elements to program, [1..(MAX_CTO_PGM-2)/AG]</param>
        ''' <param name="Data">Buffer with the data elements to program</param>
        ''' <param name="DataLength">Length of the data buffer</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_Program")> _
        Public Shared Function Program( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal NumberOfElements As Byte, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=3)> _
            ByVal Data As Byte(), _
            ByVal DataLength As Byte, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=5)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command PROGRAM: Programs a non-volatile memory segment inside the slave.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="NumberOfElements">Number of data elements to program, [1..(MAX_CTO_PGM-2)/AG]</param>
        ''' <param name="Data">Buffer with the data elements to program</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function Program( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal NumberOfElements As Byte, _
            ByVal Data As Byte()) As TXCPResult
            Return Program(XcpHandle, NumberOfElements, Data, Data.Length, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command PROGRAM_RESET: Indicates the end of a non-volatile memory programming sequence.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_ProgramReset")> _
        Public Shared Function ProgramReset( _
            ByVal XcpHandle As TXCPHandle, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=2)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command PROGRAM_RESET: Indicates the end of a non-volatile memory programming sequence.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function ProgramReset( _
            ByVal XcpHandle As TXCPHandle) As TXCPResult
            Return ProgramReset(XcpHandle, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command GET_PGM_PROCESSOR_INFO: returns general information on programming processor.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_GetPGMProcessorInformation")> _
        Public Shared Function GetPGMProcessorInformation( _
            ByVal XcpHandle As TXCPHandle, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=2)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command GET_PGM_PROCESSOR_INFO: returns general information on programming processor.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function GetPGMProcessorInformation( _
            ByVal XcpHandle As TXCPHandle) As TXCPResult
            Return GetPGMProcessorInformation(XcpHandle, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command GET_SECTOR_INFO: returns information on a specific SECTOR.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Mode">Get mode, 0: start address, 1: length, 2: name length</param>
        ''' <param name="SectorNumber">Sector number, [0,1,..MAX_SECTOR-1]</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_GetSectorInformation")> _
        Public Shared Function GetSectorInformation( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Mode As Byte, _
            ByVal SectorNumber As Byte, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=4)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command GET_SECTOR_INFO: returns information on a specific SECTOR.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Mode">Get mode, 0: start address, 1: length, 2: name length</param>
        ''' <param name="SectorNumber">Sector number, [0,1,..MAX_SECTOR-1]</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function GetSectorInformation( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Mode As Byte, _
            ByVal SectorNumber As Byte) As TXCPResult
            Return GetSectorInformation(XcpHandle, Mode, SectorNumber, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command PROGRAM_PREPARE: indicates the begin of a code download as a precondition
        ''' for non-volatile memory programming.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="CodeSize">Size of the code that will be downloaded</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_ProgramPrepare")> _
        Public Shared Function ProgramPrepare( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal CodeSize As UInt16, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=3)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command PROGRAM_PREPARE: indicates the begin of a code download as a precondition
        ''' for non-volatile memory programming.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="CodeSize">Size of the code that will be downloaded</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function ProgramPrepare( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal CodeSize As UInt16) As TXCPResult
            Return ProgramPrepare(XcpHandle, CodeSize, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command PROGRAM_FORMAT: describes the format of following, uninterrupted data
        ''' transfer before programming.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Format">Represents the data format to be used</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_ProgramFormat")> _
        Public Shared Function ProgramFormat( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Format As TXCPProgramFormat, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=3)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command PROGRAM_FORMAT: describes the format of following, uninterrupted data
        ''' transfer before programming.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Format">Represents the data format to be used</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function ProgramFormat( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Format As TXCPProgramFormat) As TXCPResult
            Return ProgramFormat(XcpHandle, Format, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command PROGRAM_NEXT: transmits consecutive data bytes for the PROGRAM command in
        ''' block transfer mode.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="NumberOfElements">The amount of elements to be programed</param>
        ''' <param name="Data">Data buffer with the elements to program</param>
        ''' <param name="DataLength">Length of the data buffer</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_ProgramNext")> _
        Public Shared Function ProgramNext( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal NumberOfElements As Byte, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=3)> _
            ByVal Data As Byte(), _
            ByVal DataLength As Byte, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=5)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command PROGRAM_NEXT: transmits consecutive data bytes for the PROGRAM command in
        ''' block transfer mode.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="NumberOfElements">The amount of elements to be programed</param>
        ''' <param name="Data">Data buffer with the elements to program</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function ProgramNext( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal NumberOfElements As Byte, _
            ByVal Data As Byte()) As TXCPResult
            Return ProgramNext(XcpHandle, NumberOfElements, Data, Data.Length, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command PROGRAM_MAX: Programs a (fixed size) amount of elements into a 
        ''' non-volatile memory segment, starting at the MTA. The MTA will be 
        ''' postincremented by MAX_CTO_PGM-1.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Elements">Data buffer with the fixed length of MAX_CTO_PGM-1</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_ProgramMax")> _
        Public Shared Function ProgramMax( _
            ByVal XcpHandle As TXCPHandle, _
            <MarshalAs(UnmanagedType.LPArray)> _
            ByVal Elements As Byte(), _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=3)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command PROGRAM_MAX: Programs a (fixed size) amount of elements into a 
        ''' non-volatile memory segment, starting at the MTA. The MTA will be 
        ''' postincremented by MAX_CTO_PGM-1.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Elements">Data buffer with the fixed length of MAX_CTO_PGM-1</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function ProgramMax( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Elements As Byte()) As TXCPResult
            Return ProgramMax(XcpHandle, Elements, Nothing, 0)
        End Function

        ''' <summary>
        ''' Command PROGRAM_VERIFY: Induces a verification process after a 
        ''' memory programming session.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Mode">Mode, 0: internal routine, 1: Verification value</param>
        ''' <param name="Type">Verification Type, i.e. area(s) to verify</param>
        ''' <param name="Value">Verification value, when mode is 1</param>
        ''' <param name="ctoBuffer">Buffer for the slave responce (CTO package)</param>
        ''' <param name="ctoBufferLength">Length of the CTO buffer</param>
        ''' <returns>A TXCPResult result code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_ProgramVerify")> _
        Public Shared Function ProgramVerify( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Mode As Byte, _
            ByVal Type As UInt16, _
            ByVal Value As UInt32, _
            <MarshalAs(UnmanagedType.LPArray, SizeParamIndex:=5)> _
            ByVal ctoBuffer As Byte(), _
            ByVal ctoBufferLength As UInt16) As TXCPResult
        End Function

        ''' <summary>
        ''' Command PROGRAM_VERIFY: Induces a verification process after a 
        ''' memory programming session.
        ''' </summary>
        ''' <param name="XcpHandle">The handle of a XCP session</param>
        ''' <param name="Mode">Mode, 0: internal routine, 1: Verification value</param>
        ''' <param name="Type">Verification Type, i.e. area(s) to verify</param>
        ''' <param name="Value">Verification value, when mode is 1</param>
        ''' <returns>A TXCPResult result code</returns>
        Public Shared Function ProgramVerify( _
            ByVal XcpHandle As TXCPHandle, _
            ByVal Mode As Byte, _
            ByVal Type As UInt16, _
            ByVal Value As UInt32) As TXCPResult
            Return ProgramVerify(XcpHandle, Mode, Type, Value, Nothing, 0)
        End Function

        ''' <summary>
        ''' Returns a descriptive text of a given TXCPResult error code
        ''' </summary>
        ''' <param name="anError">A TXCPResult error code</param>
        ''' <param name="StringBuffer">Buffer for the text (must be at least 256 in length)</param>
        ''' <returns>A TXCPResult error code</returns>
        <DllImport("PXCP.dll", EntryPoint:="XCP_GetErrorText")> _
        Public Shared Function GetErrorText( _
            ByVal anError As TXCPResult, _
            ByVal StringBuffer As StringBuilder) As TXCPResult
        End Function

    End Class
#End Region

End Namespace
