/* SPDX-License-Identifier: GPL-2.0-only */
/*
 * Copyright (C) 2025 Stephane Grosjean <stephane.grosjean@hms-networks.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the version 2 of the GNU General Public License
 * as published by the Free Software Foundation
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 */
#include "plin_main.h"

#ifdef DEBUG
#define DEBUG_CHRDEV
#elif defined(DEBUG_TRACE)
#define DEBUG_CHRDEV_TRACE
#endif

#ifdef DEBUG_CHRDEV
#define DEBUG_CHRDEV_TRACE
#endif

#ifdef PLIN_CHRDEV_API

#include <linux/slab.h>
#include <linux/poll.h>

#include "plin_usb.h"

#if __GNUC__ <= 4
/* GCC 4 is not able to check copy_from_user() destination size correctly here:
 *
 * 	err = copy_from_user((void *)&usb_cmd.init_hw, argp, argl);
 *
 * => use the raw version instead, to avoid this useless warning:
 *
 * warning: call to ‘copy_from_user_overflow’ declared with attribute warning:
 * copy_from_user() buffer size is not provably correct [enabled by default]
 */
#define copy_from_user			_copy_from_user
#endif

/* When a device is currently in use, then setting some commands is forbidden.
 * In that case, PLIN_DEV_INUSE_STATUS is the status that is returned back
 * to the user. If it is 0, then the cmd is silently ignored (see
 * dmesg). Otherwise, the ioctl() fails and errno value is set to
 * -PLIN_DEV_INUSE_STATUS. Default is -EBUSY.
 */
#define PLIN_DEV_INUSE_STATUS		-EBUSY

#define INIT_USR_RXFIFO_WITH_SLEEP

/*
 * static inline void plin_mutex_lock(struct plin_dev *dev)
 * static inline void plin_mutex_unlock(struct plin_dev *dev)
 */
static inline void plin_mutex_lock(struct plin_dev *dev)
{
	//mutex_lock(&dev->chrdev.mutex_lock);
}

static inline void plin_mutex_unlock(struct plin_dev *dev)
{
	//mutex_unlock(&dev->chrdev.mutex_lock);
}

/*
 * static inline void plin_chrdev_lock_write(struct plin_dev *dev)
 * static inline void plin_chrdev_unlock_write(struct plin_dev *dev)
 *
 * Ensure that the device write() method is exclusively used by one task at
 * a time.
 */
static inline void plin_chrdev_lock_write(struct plin_dev *dev)
{
	mutex_lock(&dev->chrdev.mutex_write);
}

static inline void plin_chrdev_unlock_write(struct plin_dev *dev)
{
	mutex_unlock(&dev->chrdev.mutex_write);
}

/*
 * static inline void plin_chrdev_lock_ioctl(struct plin_dev *dev)
 * static inline int plin_chrdev_unlock_ioctl(struct plin_dev *dev, int err)
 *
 * Ensure that the device ioctl() method is exclusively used by one task at
 * a time. In our case, this mutex protect access to the USB command buffer
 * of a device when USB commands are directly forwarded to the USB device.
 */
static inline void plin_chrdev_lock_ioctl(struct plin_dev *dev)
{
	mutex_lock(&dev->chrdev.mutex_ioctl);
}

static inline int plin_chrdev_unlock_ioctl(struct plin_dev *dev, int err)
{
	mutex_unlock(&dev->chrdev.mutex_ioctl);
	return err;
}

/*
 * static struct plin_dev *plin_chrdev_get_dev(struct plin_dev *dev)
 * static void plin_chrdev_put_dev(struct plin_dev *dev, int err)
 */
static struct plin_dev *plin_chrdev_get_dev(struct plin_dev *dev)
{
	if (!plin_get_dev(dev))
		return NULL;

	plin_mutex_lock(dev);

	return dev;
}

static int plin_chrdev_put_dev(struct plin_dev *dev, int err)
{
	plin_mutex_unlock(dev);

	return plin_put_dev(dev, err);
}

/*
 * static int plin_is_minor_ok(struct plin_dev *dev, void *arg)
 */
static int plin_is_minor_ok(struct plin_dev *dev, void *arg)
{
	struct inode *inode = arg;

	return (drv_major(&plin) == MAJOR(inode->i_rdev) &&
		dev_minor(dev) == MINOR(inode->i_rdev));
}

/*
 * static int plin_open(struct inode *inode, struct file *filep)
 */
static int plin_open(struct inode *inode, struct file *filep)
{
	struct plin_udata *usr_prv;
	struct plin_dev *dev;
	int err = 0;

#ifdef DEBUG_CHRDEV_TRACE
	int major = MAJOR(inode->i_rdev);
	int minor = MINOR(inode->i_rdev);

	pr_info(DRV_NAME ": %s(major=%d, minor=%d, filep=%p)\n",
		__func__, major, minor, filep);
#endif

	dev = plin_get_dev(plin_foreach_dev(plin_is_minor_ok, inode));
	if (!dev)
		return -ENODEV;

	usr_prv = kzalloc(sizeof(*usr_prv), GFP_KERNEL);
	if (!usr_prv) {
		pr_err(DRV_NAME ": not enough memory to open device %s\n",
			dev_full_name(dev));

		return plin_put_dev(dev, -ENOMEM);
	}

	err = kfifo_alloc(&usr_prv->rx_fifo,
			  rxfifolen * sizeof(struct plin_msg),
			  GFP_KERNEL);
	if (err) {
		kfree(usr_prv);
		return plin_put_dev(dev, -ENOMEM);
	}

	init_waitqueue_head(&usr_prv->rx_wait);

	filep->private_data = usr_prv;

	usr_prv->f_flags = filep->f_flags;
	usr_prv->dev = dev;

	plin_mutex_lock(dev);

	/* link user to device list */
	list_add_tail(&usr_prv->link, &dev_usr_list(dev));

	if (!dev_usr_count(dev)++) {
		if (dev_open(dev)) {
			err = dev_open(dev)(dev, usr_prv);
			if (err) {
				list_del(&usr_prv->link);
				kfifo_free(&usr_prv->rx_fifo);
				kfree(usr_prv);

				goto lbl_exit;
			}
		}
	}

#ifdef INIT_USR_RXFIFO_WITH_SLEEP
	/* if device bus state is SLEEP then push the corresponding event into
	 * the user rx_fifo
	 */
	if (dev->flags & PLIN_DEV_BUS_SLEEP) {
		struct plin_msg msg;

		/* "Only the timestamp is valid." */
		memset(&msg, '\0', sizeof(msg));
		msg.type = PLIN_MSG_SLEEP;
		msg.ts_us = dev->sleep_ts;

		kfifo_in(&usr_prv->rx_fifo, &msg, sizeof(struct plin_msg));

#ifdef DEBUG_CHRDEV_TRACE
		pr_info(DRV_NAME ": %s(): PLIN_MSG_SLEEP simulated\n",
			__func__);
#endif
	}
#endif

lbl_exit:
	return plin_chrdev_put_dev(dev, err);
}

/*
 * static int plin_release(struct inode *inode, struct file *filep)
 *
 * Called each time a process closes the device.
 */
static int plin_release(struct inode *inode, struct file *filep)
{
	struct plin_udata *usr_prv = (struct plin_udata *)filep->private_data;
	struct plin_dev *dev = plin_chrdev_get_dev(usr_prv->dev);

#ifdef DEBUG_CHRDEV_TRACE
	pr_info(DRV_NAME ": %s(%p, %s): user_count=%d\n", __func__, filep,
		(dev) ? dev_full_name(dev) : "???",
		(dev) ? dev_usr_count(dev) : -1);
#endif

	if (dev) {

		list_del(&usr_prv->link);

		/* last user does close the device itself */
		if (dev_usr_count(dev) > 0)
			if (!--dev_usr_count(dev))
				if (dev_close(dev))
					dev_close(dev)(dev, usr_prv);

		plin_chrdev_put_dev(dev, 0);
	}

	kfifo_free(&usr_prv->rx_fifo);
	kfree(usr_prv);

	return 0;
}

#ifdef DEBUG_CHRDEV_TRACE
static const char *plin_msg_text[] = {
	[PLIN_MSG_FRAME] = "FRAME",
	[PLIN_MSG_SLEEP] = "SLEEP",
	[PLIN_MSG_WAKEUP] = "WAKEUP",
	[PLIN_MSG_AUTOBAUD_TO] = "AUTOBAUD_TO",
	[PLIN_MSG_AUTOBAUD_OK] = "AUTOBAUD_OK",
	[PLIN_MSG_OVERRUN] = "OVERRUN",
};
#endif

/*
 * static ssize_t plin_read(struct file *filep, char __user *buf, size_t count,
 *			    loff_t *f_pos)
 */
static ssize_t plin_read(struct file *filep, char __user *buf, size_t count,
			 loff_t *f_pos)
{
	struct plin_udata *usr_prv = (struct plin_udata *)filep->private_data;
	struct plin_dev *dev = plin_chrdev_get_dev(usr_prv->dev);
	unsigned int copied;
	int err;

#ifdef DEBUG_CHRDEV_TRACE
	pr_info(DRV_NAME ": %s(%p, %s)\n", __func__, filep,
		(dev) ? dev_full_name(dev) : "???");
#endif

	if (!dev)
		return -ENODEV;

#ifdef USES_DEV_RXFIFO
	/* get any pending event in the device rx queue first */
	err = kfifo_to_user(&dev->rx_fifo, buf, count, &copied);
	if (err || copied)
		goto lbl_exit;
#endif

	while (kfifo_is_empty(&usr_prv->rx_fifo)) {

		plin_mutex_unlock(dev);

		if (filep->f_flags & O_NONBLOCK) {
			return plin_put_dev(dev, -EAGAIN);
		}

#ifdef DEBUG_CHRDEV_TRACE
		pr_info(DRV_NAME ": %s() is going to sleep...\n", __func__);
#endif
		err = wait_event_interruptible(usr_prv->rx_wait,
				atomic_read(&dev->ref_count) <= 0 ||
				!kfifo_is_empty(&usr_prv->rx_fifo));
		if (err) {
			return plin_put_dev(dev, err);
		}

#ifdef DEBUG_CHRDEV_TRACE
		pr_info(DRV_NAME ": %s() is awaken!\n", __func__);
#endif

		plin_mutex_lock(dev);

		if (atomic_read(&dev->ref_count) <= 0) {
			/* this should happend only if the device is unplugged
			 * when an application is waiting for reading: calling
			 * _put_dev() next does free the attached device.
			 */
#ifdef DEBUG_CHRDEV_TRACE
			pr_info(DRV_NAME ": %s(%u) ref_count=%d !\n",
				__func__, __LINE__,
				atomic_read(&dev->ref_count));
#endif
			return plin_chrdev_put_dev(dev, -ENODEV);
		}
	}

	err = kfifo_to_user(&usr_prv->rx_fifo, buf, count, &copied);
	if (!err)
		err = copied;

#ifdef USES_DEV_RXFIFO
lbl_exit:
#endif
#ifdef DEBUG_CHRDEV_TRACE
	if (err >= 0) {
		struct plin_msg msg;

		if (!copy_from_user(&msg, buf, sizeof(msg))) {
			dump_mem("data read", &msg, sizeof(msg));
			pr_info(DRV_NAME ": %s < %s\n",
				(msg.type < ARRAY_SIZE(plin_msg_text)) ?
					plin_msg_text[msg.type] : "???",
				dev_full_name(dev));
		} else {
			pr_err(DRV_NAME
			       ": %s(%u): copy_from_user(p=%p, l=%lu) "
			       "failure\n", __func__, __LINE__,
			       buf, (unsigned long )sizeof(msg));
		}
	}

	pr_info(DRV_NAME ": %s() returns %d\n", __func__, err);
#endif
	return plin_chrdev_put_dev(dev, err);
}

/*
 * static ssize_t plin_write(struct file *filep, const char __user *buf,
 *			     size_t count, loff_t *f_pos)
 *
 * Note: "during master mode only, when scheduler is not running"
 */
static ssize_t plin_write(struct file *filep, const char __user *buf,
			  size_t count, loff_t *f_pos)
{
	struct plin_udata *usr_prv = (struct plin_udata *)filep->private_data;
	struct plin_dev *dev = plin_get_dev(usr_prv->dev);
	struct plin_msg msg;
	int err;

#ifdef DEBUG_CHRDEV_TRACE
	pr_info(DRV_NAME ": %s(%p, %s)\n", __func__, filep,
		(dev) ? dev_full_name(dev) : "???");
#endif

	if (!dev)
		return -ENODEV;

	if (!dev->write) {
		return plin_put_dev(dev, -ENOTSUPP);
	}

	err = copy_from_user(&msg, buf, sizeof(msg));
	if (err) {
		pr_err(DRV_NAME
		       ": %s(%u): copy_from_user(p=%p, l=%lu) failure\n",
		       __func__, __LINE__, buf, (unsigned long )sizeof(msg));
		return plin_put_dev(dev, -EFAULT);
	}

#ifdef DEBUG_CHRDEV_TRACE
	pr_info(DRV_NAME ": %s > %s\n",
		(msg.type < ARRAY_SIZE(plin_msg_text)) ?
			plin_msg_text[msg.type] : "???", dev_full_name(dev));
#endif

	plin_chrdev_lock_write(dev);

	while (dev->write(dev, &msg) == -EAGAIN) {

		plin_chrdev_unlock_write(dev);

		if ((filep->f_flags & O_NONBLOCK) || (!dev->write_busy)) {
			err = -EAGAIN;
			goto lbl_exit;
		}
		
		err = wait_event_interruptible(dev->tx_wait,
				       atomic_read(&dev->ref_count) <= 0 ||
				       !dev->write_busy(dev));
		if (err)
			goto lbl_exit;

		if (atomic_read(&dev->ref_count) <= 0) {

			/* this should arrive only if the device is unplugged
			 * when an application is waiting for writing: calling
			 * _put_dev() next does free the attached device.
			 */
#ifdef DEBUG_CHRDEV_TRACE
			pr_info(DRV_NAME ": %s(%u) ref_count=%d !\n",
				__func__, __LINE__,
				atomic_read(&dev->ref_count));
#endif
			err = -ENODEV;
			goto lbl_exit;
		}

		plin_chrdev_lock_write(dev);
	}

	plin_chrdev_unlock_write(dev);

	if (!err)
		err = sizeof(msg);

lbl_exit:
#ifdef DEBUG_CHRDEV_TRACE
	pr_info(DRV_NAME ": %s() returns %d\n", __func__, err);
#endif
	return plin_put_dev(dev, err);
}


#ifdef DEBUG_CHRDEV_TRACE
static const char *plin_cmd_text[] = {
	[PLIN_USB_CMD_INIT_HW] = "INIT_HW",
	[PLIN_USB_CMD_RST_HW_CFG] = "RST_HW_CFG",
	[PLIN_USB_CMD_SET_FRM_ENTRY] = "SET_FRM_ENTRY",
	[PLIN_USB_CMD_GET_FRM_ENTRY] = "GET_FRM_ENTRY",
	[PLIN_USB_CMD_START_AUTO_BAUD] = "START_AUTO_BAUD",
	[PLIN_USB_CMD_GET_BAUDRATE] = "GET_BAUDRATE",
	[PLIN_USB_CMD_SET_ID_FILTER] = "SET_ID_FILTER",
	[PLIN_USB_CMD_GET_ID_FILTER] = "GET_ID_FILTER",
	[PLIN_USB_CMD_GET_MODE] = "GET_MODE",
	[PLIN_USB_CMD_SET_IDENT_STR] = "SET_IDENT_STR",
	[PLIN_USB_CMD_GET_IDENT_STR] = "GET_IDENT_STR",
	[PLIN_USB_CMD_IDENTIFY_BUS] = "IDENTIFY_BUS",
	[PLIN_USB_CMD_GET_FW_VER] = "GET_FW_VER",
	[PLIN_USB_CMD_START_KEEP_ALIVE] = "START_KEEP_ALIVE",
	[PLIN_USB_CMD_RESUME_KEEP_ALIVE] = "RESUME_KEEP_ALIVE",
	[PLIN_USB_CMD_SUSPEND_KEEP_ALIVE] = "SUSPEND_KEEP_ALIVE",
	[PLIN_USB_CMD_ADD_SCHED_SLOT] = "ADD_SCHED_SLOT",
	[PLIN_USB_CMD_DEL_SCHED_SLOT] = "DEL_SCHED_SLOT",
	[PLIN_USB_CMD_GET_SLOTS_COUNT] = "GET_SLOTS_COUNT",
	[PLIN_USB_CMD_GET_SCHED_SLOT] = "GET_SCHED_SLOT",
	[PLIN_USB_CMD_SET_SCHED_BRKPT] = "SET_SCHED_BRKPT",
	[PLIN_USB_CMD_START_SCHED] = "START_SCHED",
	[PLIN_USB_CMD_RESUME_SCHED] = "RESUME_SCHED",
	[PLIN_USB_CMD_SUSPEND_SCHED] = "SUSPEND_SCHED",
	[PLIN_USB_CMD_GET_STATUS] = "GET_STATUS",
	[PLIN_USB_CMD_RESET] = "RESET",
	[PLIN_USB_CMD_UPDATE_BYTE_ARRAY] = "UPDATE_BYTE_ARRAY",
	[PLIN_USB_CMD_XMT_WAKE_UP] = "XMT_WAKE_UP",
	[PLIN_USB_CMD_RSP_REMAP] = "RSP_REMAP",
	[PLIN_USB_CMD_LED_STATE] = "LED_STATE",
};
#endif

/*
 * static int plin_ioctl_usb(struct plin_dev *dev, unsigned int cmd,
 *			     void __user *argp, struct plin_udata *usr_prv)
 */
static int plin_ioctl_usb(struct plin_dev *dev, unsigned int cmd,
			  void __user *argp, struct plin_udata *usr_prv)
{
	struct plin_usb_cmd usb_cmd;
	int err, argl = _IOC_SIZE(cmd), read_rsp;
	int idx, i, l;

#ifdef DEBUG_CHRDEV_TRACE
	pr_info(DRV_NAME ": %s(%s, dir=%d, nr=%d, size=%d)\n",
		__func__, dev_full_name(dev),
		_IOC_DIR(cmd), _IOC_NR(cmd), _IOC_SIZE(cmd));
#endif

	switch (_IOC_NR(cmd)) {

	case PLIN_USB_CMD_INIT_HW:
	case PLIN_USB_CMD_RST_HW_CFG:
	case PLIN_USB_CMD_SET_FRM_ENTRY:
	case PLIN_USB_CMD_GET_FRM_ENTRY:
	case PLIN_USB_CMD_START_AUTO_BAUD:
	case PLIN_USB_CMD_GET_BAUDRATE:
	case PLIN_USB_CMD_SET_ID_FILTER:
	case PLIN_USB_CMD_GET_ID_FILTER:
	case PLIN_USB_CMD_GET_MODE:
	case PLIN_USB_CMD_SET_IDENT_STR:
	case PLIN_USB_CMD_GET_IDENT_STR:
	case PLIN_USB_CMD_IDENTIFY_BUS:
	case PLIN_USB_CMD_GET_FW_VER:
	case PLIN_USB_CMD_START_KEEP_ALIVE:
	case PLIN_USB_CMD_RESUME_KEEP_ALIVE:
	case PLIN_USB_CMD_SUSPEND_KEEP_ALIVE:
	case PLIN_USB_CMD_ADD_SCHED_SLOT:
	case PLIN_USB_CMD_DEL_SCHED_SLOT:
	case PLIN_USB_CMD_GET_SLOTS_COUNT:
	case PLIN_USB_CMD_GET_SCHED_SLOT:
	case PLIN_USB_CMD_SET_SCHED_BRKPT:
	case PLIN_USB_CMD_START_SCHED:
	case PLIN_USB_CMD_RESUME_SCHED:
	case PLIN_USB_CMD_SUSPEND_SCHED:
	case PLIN_USB_CMD_GET_STATUS:
	case PLIN_USB_CMD_RESET:
	case PLIN_USB_CMD_UPDATE_BYTE_ARRAY:
	case PLIN_USB_CMD_XMT_WAKE_UP:
	case PLIN_USB_CMD_RSP_REMAP:
	case PLIN_USB_CMD_LED_STATE:
		break;
	default:
		pr_err(DRV_NAME ": invalid ioctl() USB cmd %u\n",
			_IOC_NR(cmd));
		return -ENOTTY;
	}

#ifdef DEBUG_CHRDEV_TRACE
	pr_info(DRV_NAME ": %s > %s\n",
		plin_cmd_text[_IOC_NR(cmd)], dev_full_name(dev));
#endif

	if (argl + 4 > sizeof(usb_cmd)) {
		pr_err(DRV_NAME ": too large arg: %u > %lu bytes\n",
			argl + 4, (unsigned long)sizeof(usb_cmd));
		return -ERANGE;
	}

	if (_IOC_DIR(cmd) & _IOC_WRITE) {
		err = copy_from_user((void *)&usb_cmd.init_hw, argp, argl);
		if (err) {
			pr_err(DRV_NAME
			       ": %s(%u): copy_from_user(p=%p, l=%d) failure\n",
			       __func__, __LINE__, argp, argl);
			return -EFAULT;
		}
	}

	/* initialize the USB command */
	usb_cmd.id = (u8 )_IOC_NR(cmd);
	usb_cmd.device = 0;
	usb_cmd.bus = dev_idx(dev);
	usb_cmd.client = 0;

	/* _IOC_READ bit set means that cmd is waiting for a response */
	read_rsp = (_IOC_DIR(cmd) & _IOC_READ);

	/* protect the device from mutual access */
	plin_chrdev_lock_ioctl(dev);

	/* pre-process the USB command */
	switch (usb_cmd.id) {

	case PLIN_USB_CMD_INIT_HW:

		/* if this bus has been setup and I'm not the only one to play
		 * with, then I can't change its setup: ignore this command if
		 * mode and bitrate I set are the same than the current ones.
		 */
		if ((dev->mode != PLIN_MODE_NONE) && (dev_usr_count(dev) > 1)) {

			/* can't init the device in any other mode than the
			 * current one
			 */
			if (dev->mode != usb_cmd.init_hw.mode) {
				pr_warn(DRV_NAME
					": can't set mode %d while "
					"%s is being used (mode=%d)\n",
					usb_cmd.init_hw.mode,
					dev_full_name(dev), dev->mode);

				return plin_chrdev_unlock_ioctl(dev, -EBUSY);
			}

			/* can't init the device with another baudrate than the
			 * current one
			 */
			if (dev->baudrate !=
				le16_to_cpu(usb_cmd.init_hw.baudrate)) {
				pr_warn(DRV_NAME
					": can't set baudrate %d while "
					"%s is being used (baudrate=%d)\n",
					le16_to_cpu(usb_cmd.init_hw.baudrate),
					dev_full_name(dev), dev->baudrate);

				return plin_chrdev_unlock_ioctl(dev, -EBUSY);
			}

			/* Ok */
			return plin_chrdev_unlock_ioctl(dev, 0);
		}

		/* Fall through */

	/* these commands can be done by anyone, whatever the mode is */
	case PLIN_USB_CMD_GET_FRM_ENTRY:
	case PLIN_USB_CMD_GET_BAUDRATE:
	case PLIN_USB_CMD_GET_ID_FILTER:
	case PLIN_USB_CMD_GET_MODE:
	case PLIN_USB_CMD_IDENTIFY_BUS:
	case PLIN_USB_CMD_GET_FW_VER:
	case PLIN_USB_CMD_GET_STATUS:
	case PLIN_USB_CMD_GET_SLOTS_COUNT:
	case PLIN_USB_CMD_GET_SCHED_SLOT:
	case PLIN_USB_CMD_LED_STATE:
	case PLIN_USB_CMD_UPDATE_BYTE_ARRAY:

	/* sure of these two ones? */
	case PLIN_USB_CMD_XMT_WAKE_UP:
	case PLIN_USB_CMD_RESET:
		break;

	case PLIN_USB_CMD_SET_IDENT_STR:
	case PLIN_USB_CMD_GET_IDENT_STR:
		/* WARNING: 48 bytes array is structured as below by the
		 * Windows driver:
		 *                     1 2                      4 4  4
		 *  0                  9 0                      3 4  7
		 * +-------- 20 --------+---------- 24 ----------+--4-+
		 * | reserved for       | IDSTR                  | IDNUM
		 * | future use         |                        |
		 *
		 * User SHOULD prefer using PLIOSETUSERSTR/PLIOGETUSERSTR or
		 * PLIOSETUSERID/PLIOGETUSERID ioctl instead.
		 */
		break;

	case PLIN_USB_CMD_RSP_REMAP:
		/* hook: when command is PLIN_USB_CMD_RSP_REMAP, MUST read
		 * set_get field to know if we really need to wait for a
		 * response:
		 */
		read_rsp = (usb_cmd.rsp_remap.set_get ==
						PLIN_USB_RSP_REMAP_GET);
		/* anyone can do that */
		if (read_rsp)
			break;

		/* Fall through */
		fallthrough;
	default:
		/* reject any other command if the device mode is set and there
		 * are more than one user of it
		 */
		if ((dev->mode != PLIN_MODE_NONE) && (dev_usr_count(dev) > 1)) {
			pr_warn(DRV_NAME
#if !PLIN_DEV_INUSE_STATUS
				": cmd %u ignored while "
#else
				": cmd %u failed because "
#endif
				"%u users are using %s and its mode is %u\n",
				usb_cmd.id, dev_usr_count(dev),
				dev_full_name(dev), dev->mode);

			return plin_chrdev_unlock_ioctl(dev,
						PLIN_DEV_INUSE_STATUS);
		}

		break;
	}

#ifdef DEBUG_CHRDEV_TRACE
	dump_mem("cmd sent",  &usb_cmd, argl+4);
#endif
	/* write the command */
	err = dev->send_direct_cmd(dev, &usb_cmd, argl+4);
	if (err) {
		pr_err(DRV_NAME ": failed to send USB cmd %d (err %d)\n",
			_IOC_NR(cmd), err);

		return plin_chrdev_unlock_ioctl(dev, err);
	}

	/* post-process the command */
	switch (usb_cmd.id) {
	case PLIN_USB_CMD_RST_HW_CFG:

		/* reset the cached values */
		dev->mode = PLIN_MODE_NONE;
		dev->baudrate = 0;
		break;
		
	case PLIN_USB_CMD_INIT_HW:

		/* cache the device settings if any other user wanted to
		 * init the hw too
		 */
		dev->mode = usb_cmd.init_hw.mode;
		dev->baudrate = le16_to_cpu(usb_cmd.init_hw.baudrate);
		break;

	case PLIN_USB_CMD_UPDATE_BYTE_ARRAY:
		/* cache modified data bytes only */
		idx = usb_cmd.update_data.id & PLIN_FRM_ID_MAX;
		i = (usb_cmd.update_data.idx < PLIN_DAT_LEN) ? \
			usb_cmd.update_data.idx : PLIN_DAT_LEN-1;
		l = (i + usb_cmd.update_data.len <= PLIN_DAT_LEN) ? \
			usb_cmd.update_data.len : (PLIN_DAT_LEN - i);
		memcpy(dev->frm_entry_cache[idx].d + i,
		       usb_cmd.update_data.d, l);
		break;

	case PLIN_USB_CMD_SET_FRM_ENTRY:
		/* cache initial data bytes to be able to given them back in
		 * PLIN_USB_CMD_GET_FRM_ENTRY response.
		 */
		idx = usb_cmd.frm_entry.id & PLIN_FRM_ID_MAX;
		dev->frm_entry_cache[idx] = usb_cmd.frm_entry;
		break;
	}

	/* these commands are waiting for a response */
	if (read_rsp) {
		struct plin_usb_cmd usb_rsp;

		err = dev->read_direct_rsp(dev, &usb_rsp, sizeof(usb_rsp));
		if (err) {
			pr_err(DRV_NAME ": failed to get USB rsp %d (err %d)\n",
				_IOC_NR(cmd), err);
		} else {
#ifdef DEBUG_CHRDEV_TRACE
			pr_info(DRV_NAME ": %s < %s\n",
				plin_cmd_text[usb_rsp.id], dev_full_name(dev));
			dump_mem("rsp read",  &usb_rsp, sizeof(usb_rsp));
#endif
			if (usb_rsp.id != usb_cmd.id) {
				pr_err(DRV_NAME
				       ": got rsp %u while waiting for %u\n",
				       usb_rsp.id, usb_cmd.id);
				err = -EBADMSG;

			} else if (usb_rsp.bus != usb_cmd.bus) {
				pr_err(DRV_NAME
				       ": got rsp %u from bus %u "
				       "instead of %u\n",
				       usb_rsp.id, usb_rsp.bus, usb_cmd.bus);
				err = -EBADMSG;

			} else if (usb_rsp.client != usb_cmd.client) {
				pr_err(DRV_NAME
				       ": got rsp %u to client %u "
				       "instead of %u\n",
				       usb_rsp.id, usb_rsp.client,
				       usb_cmd.client);
				err = -EBADMSG;

			} else {

				/* otherwise, post-process response received
				 * before copying it into user space buffer
				 */
				switch (usb_rsp.id) {
				case PLIN_USB_CMD_GET_FRM_ENTRY:
					idx = usb_cmd.frm_entry.id & PLIN_FRM_ID_MAX;
					l = (usb_rsp.frm_entry.len <= PLIN_DAT_LEN) ? \
						usb_rsp.frm_entry.len : PLIN_DAT_LEN;
					memcpy(usb_rsp.frm_entry.d,
					       dev->frm_entry_cache[idx].d, l);
					break;
				}

				err = copy_to_user(argp,
						   (void *)&usb_rsp.init_hw,
						   argl);
				if (err) {
					pr_err(DRV_NAME ": %s(%u): "
					       "copy_to_user(p=%p, l=%d) "
					       "failure\n",
					       __func__, __LINE__, argp, argl);

					err = -EFAULT;
				}
			}
		}
	}

	return plin_chrdev_unlock_ioctl(dev, err);
}

#if 0
static int __plin_send_cmd_ex(struct plin_dev *dev,
			      struct plin_usb_cmd *usb_cmd, int l, int read_rsp)
{
	int err;

	/* initialize the USB command */
	usb_cmd->device = 0;
	usb_cmd->bus = dev_idx(dev);
	usb_cmd->client = 0;

#ifdef DEBUG_CHRDEV_TRACE
	pr_info(DRV_NAME ": %s > %s\n",
		plin_cmd_text[usb_cmd->id], dev_full_name(dev));
	dump_mem("cmd sent",  usb_cmd, l);
#endif

	/* write the command */
	err = dev->send_direct_cmd(dev, usb_cmd, l);
	if (err) {
		pr_err(DRV_NAME ": failed to send USB cmd %d (err %d)\n",
			usb_cmd->id, err);

		return err;
	}

	/* these commands are waiting for a response */
	if (read_rsp) {
		struct plin_usb_cmd usb_rsp;

		err = dev->read_direct_rsp(dev, &usb_rsp, sizeof(usb_rsp));
		if (err) {
			pr_err(DRV_NAME ": failed to get USB rsp %d (err %d)\n",
				usb_cmd->id, err);
		} else {
#ifdef DEBUG_CHRDEV_TRACE
			pr_info(DRV_NAME ": %s < %s\n",
				plin_cmd_text[usb_rsp.id], dev_full_name(dev));
			dump_mem("rsp read",  &usb_rsp, sizeof(usb_rsp));
#endif
			if (usb_rsp.id != usb_cmd->id) {
				pr_err(DRV_NAME
				       ": got rsp %u while waiting for %u\n",
				       usb_rsp.id, usb_cmd->id);
				err = -EBADMSG;

			} else if (usb_rsp.bus != usb_cmd->bus) {
				pr_err(DRV_NAME
				       ": got rsp %u from bus %u "
				       "instead of %u\n",
				       usb_rsp.id, usb_rsp.bus, usb_cmd->bus);
				err = -EBADMSG;

			} else if (usb_rsp.client != usb_cmd->client) {
				pr_err(DRV_NAME
				       ": got rsp %u to client %u "
				       "instead of %u\n",
				       usb_rsp.id, usb_rsp.client,
				       usb_cmd->client);
				err = -EBADMSG;
			} else {
				*usb_cmd = usb_rsp;
			}
		}
	}

	return err;
}

static int __plin_get_rsp(struct plin_dev *dev, struct plin_usb_cmd *usb_cmd,
			  int cmd_len)
{
	return __plin_send_cmd_ex(dev, usb_cmd, cmd_len, 1);
}

static int __plin_send_cmd(struct plin_dev *dev, struct plin_usb_cmd *usb_cmd,
			   int cmd_len)
{
	return __plin_send_cmd_ex(dev, usb_cmd, cmd_len, 0);
}

/*
 * static int plin_ioctl_ident(struct plin_dev *dev, unsigned int cmd,
 *			       void __user *argp, struct plin_udata *usr_prv)
 */
static int _plin_ioctl_ident(struct plin_dev *dev, unsigned int cmd,
			     void __user *argp, struct plin_udata *usr_prv)
{
	struct plin_usb_cmd usb_cmd = { .id = PLIN_USB_CMD_GET_IDENT_STR, };
	int err, argl = _IOC_SIZE(cmd);

#ifdef DEBUG_CHRDEV_TRACE
	pr_info(DRV_NAME ": %s(%s, dir=%d, nr=%d, size=%d)\n",
		__func__, dev_full_name(dev),
		_IOC_DIR(cmd), _IOC_NR(cmd), _IOC_SIZE(cmd));
#endif

	/* protect the device from mutual access */
	plin_chrdev_lock_ioctl(dev);

	/* 1st, always read current ident string */
	switch (cmd) {

	case PLIOGETUSERSTR:
	case PLIOGETUSERID:
	case PLIOSETUSERSTR:
	case PLIOSETUSERID:
		err = __plin_get_rsp(dev, &usb_cmd, argl+4);
		break;

	default:
		pr_err(DRV_NAME ": invalid ioctl() ident cmd %u\n",
			_IOC_NR(cmd));
		return plin_chrdev_unlock_ioctl(dev, -ENOTTY);
	}

	/* 2nd, for SET commands, overwrite ident string with user data */
	usb_cmd.id = PLIN_USB_CMD_SET_IDENT_STR;

	switch (cmd) {

	case PLIOSETUSERSTR:
		err = copy_from_user((void *)usb_cmd.ident_str.user_str,
				     argp + offsetof(struct plin_usb_ident_str,
						     user_str),
				     PLIN_USB_IDENT_USER_LEN);
		break;

	case PLIOSETUSERID:
		err = copy_from_user((void *)&usb_cmd.ident_str.user_id,
				     argp + offsetof(struct plin_usb_ident_str,
						     user_id),
				     sizeof(__le32));
		break;

	case PLIOGETUSERSTR:
		usb_cmd.id = 0;
		err = copy_to_user(argp + offsetof(struct plin_usb_ident_str,
						   user_str),
				   (void *)usb_cmd.ident_str.user_str,
				   PLIN_USB_IDENT_USER_LEN);
		break;

	case PLIOGETUSERID:
		usb_cmd.id = 0;
		err = copy_to_user(argp + offsetof(struct plin_usb_ident_str,
						   user_id),
				   (void *)&usb_cmd.ident_str.user_id,
				   sizeof(__le32));
		break;
	}

	if (err) {
		pr_err(DRV_NAME
		       ": %s(%u): copy_[from/to]_user(p=%p) failure\n",
		       __func__, __LINE__, argp);
		return plin_chrdev_unlock_ioctl(dev, -EFAULT);
	}

	if (usb_cmd.id == PLIN_USB_CMD_SET_IDENT_STR)
		err = __plin_send_cmd(dev, &usb_cmd, argl+4);

	return plin_chrdev_unlock_ioctl(dev, err);
}
#endif

static int plin_ioctl_ident(struct plin_dev *dev, unsigned int cmd,
			     void __user *argp, struct plin_udata *usr_prv)
{
	struct plin_usb_ident_str ident_str;
	int err = -ENOTSUPP, argl = _IOC_SIZE(cmd);

#ifdef DEBUG_CHRDEV_TRACE
	pr_info(DRV_NAME ": %s(%s, dir=%d, nr=%d, size=%d)\n",
		__func__, dev_full_name(dev),
		_IOC_DIR(cmd), _IOC_NR(cmd), _IOC_SIZE(cmd));
#endif

	/* if command sets a user value, get the whole struct at once */
	if (_IOC_DIR(cmd) & _IOC_WRITE) {
		err = copy_from_user(&ident_str, argp, argl);
		if (err) {
			pr_err(DRV_NAME
			       ": %s(%u): copy_from_user(p=%p) failure\n",
			       __func__, __LINE__, argp);
			return -EFAULT;
		}
	}

	/* protect the device from mutual access */
	plin_chrdev_lock_ioctl(dev);

	switch (cmd) {

	case PLIOGETUSERSTR:
		if (dev->get_id_str_nolock)
			err = dev->get_id_str_nolock(dev, ident_str.user_str,
						     PLIN_USB_IDENT_USER_LEN);
		break;

	case PLIOGETUSERID:
		if (dev->get_id_num_nolock)
			err = dev->get_id_num_nolock(dev, &ident_str.user_id);

		break;

	case PLIOSETUSERSTR:
		if (dev->set_id_str)
			err = dev->set_id_str(dev, ident_str.user_str,
					      PLIN_USB_IDENT_USER_LEN);
		break;

	case PLIOSETUSERID:
		if (dev->set_id_num)
			err = dev->set_id_num(dev, ident_str.user_id);

		break;

	default:
		err = -ENOTTY;
		pr_err(DRV_NAME ": invalid ioctl() ident cmd %u\n",
			_IOC_NR(cmd));
	}

	plin_chrdev_unlock_ioctl(dev, err);

	/* if ident cmds gets a user field, then copy the whole ident struct */
	if (_IOC_DIR(cmd) & _IOC_READ) {
		err = copy_to_user(argp, &ident_str, argl);
		if (err)
			pr_err(DRV_NAME
			       ": %s(%u): copy_to_user(p=%p) failure\n",
			       __func__, __LINE__, argp);
	}

	return err;
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,36)
static int plin_ioctl(struct inode *inode,
		      struct file *filep, unsigned int cmd, unsigned long arg)
#else
static long plin_ioctl(struct file *filep, unsigned int cmd, unsigned long arg)
#endif
{
	struct plin_udata *usr_prv = (struct plin_udata *)filep->private_data;
	struct plin_dev *dev = plin_get_dev(usr_prv->dev);
	int err = 0;

#ifdef DEBUG_CHRDEV_TRACE
	pr_info(DRV_NAME ": %s(%p, %s, cmd=%u, arg=%lu)\n",
		__func__, filep, dev ? dev_full_name(dev) : "???", cmd, arg);
#endif

	if (!dev)
		return -ENODEV;

	switch (_IOC_TYPE(cmd)) {
	case 'i':
		err = plin_ioctl_ident(dev, cmd, (void __user *)arg, usr_prv);
		break;
	case 'u':
		err = plin_ioctl_usb(dev, cmd, (void __user *)arg, usr_prv);
		break;
	default:
		pr_err(DRV_NAME ": %u invalid ioctl() cmd\n", cmd);
		err = -ENOTTY;
	}

	return plin_put_dev(dev, err);
}

#ifdef PLIN_CONFIG_COMPAT
static long plin_compat_ioctl(struct file *filep, unsigned int cmd,
			      unsigned long arg)
{
	struct plin_udata *usr_prv = (struct plin_udata *)filep->private_data;
	struct plin_dev *dev = plin_chrdev_get_dev(usr_prv->dev);
	void __user *argp = compat_ptr(arg);
	int err = 0;

#ifdef DEBUG_CHRDEV_TRACE
	pr_info(DRV_NAME ": %s(%s, cmd=%d, arg=%lu)\n",
		__func__, dev_full_name(dev), cmd, arg);
#endif

	if (!dev)
		return -ENODEV;

	switch (_IOC_TYPE(cmd)) {
	case 'u':
		err = plin_ioctl_usb(dev, cmd, (void __user *)arg, usr_priv);
		break;
	default:
		pr_err(DRV_NAME ": %u invalid ioctl() cmd\n", cmd);
		err = -ENOTTY;
	}

	return plin_chrdev_put_dev(dev, err);
}
#endif

/*
 * static unsigned int plin_poll(struct file *filep, poll_table *wait)
 */
static unsigned int plin_poll(struct file *filep, poll_table *wait)
{
	struct plin_udata *usr_prv = (struct plin_udata *)filep->private_data;
	struct plin_dev *dev = plin_chrdev_get_dev(usr_prv->dev);
	unsigned int mask = 0;

#ifdef DEBUG_CHRDEV_TRACE
	pr_info(DRV_NAME ": %s(%s)\n", __func__,
		(dev) ? dev_full_name(dev) : "???");
#endif

	/* return on ops that could be performed without blocking */
	poll_wait(filep, &usr_prv->rx_wait, wait);

	if (!kfifo_is_empty(&usr_prv->rx_fifo))
		mask |= POLLIN | POLLRDNORM;

	poll_wait(filep, &dev->tx_wait, wait);

	if (dev->write_busy && !dev->write_busy(dev))
		mask |= POLLOUT | POLLWRNORM;

	return plin_chrdev_put_dev(dev, mask);
}

static struct file_operations plin_fops = {
	owner:      THIS_MODULE,
	open:       plin_open,
	release:    plin_release,
	read:       plin_read,
	write:      plin_write,
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,36)
	ioctl:      plin_ioctl,
#else
	unlocked_ioctl: plin_ioctl,
#endif
#ifdef PLIN_CONFIG_COMPAT
	compat_ioctl: plin_compat_ioctl,
#endif
	poll:       plin_poll,
};

static struct plin_udata *plin_chrdev_foreach_user(struct plin_dev *dev,
					int (*f)(struct plin_udata *, void *),
					void *arg)
{
	struct list_head *pd;

	list_for_each(pd, &dev_usr_list(dev)) {
		struct plin_udata *u = list_entry(pd, struct plin_udata, link);
		int err = f(u, arg);

		if (err > 0)
			return u;
		if (err < 0)
			break;
	}

	return NULL;
}

/*
 * int plin_chrdev_init_dev(struct plin_dev *dev)
 *
 * Called when a LIN device is created, to initialize all what should be to
 * allow chardev access.
 */
int plin_chrdev_init_dev(struct plin_dev *dev)
{
	INIT_LIST_HEAD(&dev_usr_list(dev));

	mutex_init(&dev->chrdev.mutex_lock);
	mutex_init(&dev->chrdev.mutex_write);
	mutex_init(&dev->chrdev.mutex_ioctl);

	return 0;
}

static int do_wake_up_interruptible(struct plin_udata *usr_prv, void *arg)
{
	wake_up_interruptible(&usr_prv->rx_wait);

	kfifo_free(&usr_prv->rx_fifo);

	return 0;
}

/*
 * void plin_chrdev_free_dev(struct plin_dev *dev)
 *
 * Called when a device is to be deleted.
 */
void plin_chrdev_free_dev(struct plin_dev *dev)
{
	/* wakeup any task that is waiting for incoming msgs:
	 * the rx_fifo is to be deleted!
	 */
	plin_chrdev_foreach_user(dev, do_wake_up_interruptible, NULL);

	wake_up_interruptible(&dev->tx_wait);
}

static int do_plin_rx_msg(struct plin_udata *usr_prv, void *arg)
{
	struct plin_msg *msg = arg;
	int err = 0;

	if (!kfifo_is_full(&usr_prv->rx_fifo)) {

		switch (msg->type) {
		case PLIN_MSG_SLEEP:

#ifndef INIT_USR_RXFIFO_WITH_SLEEP
			/* don't push SLEEP event if user already knows it */
			if (usr_prv->flags & PLIN_DEV_BUS_SLEEP)
				break;
#endif
			/* Fall through */
			fallthrough;

		case PLIN_MSG_FRAME:
		case PLIN_MSG_WAKEUP:
		case PLIN_MSG_AUTOBAUD_TO:
		case PLIN_MSG_AUTOBAUD_OK:
		case PLIN_MSG_OVERRUN:

			kfifo_in(&usr_prv->rx_fifo, msg,
				 sizeof(struct plin_msg));
#ifdef DEBUG_RX
#ifdef DEBUG_MINOR
			if (dev_minor(usr_prv->dev) == DEBUG_MINOR)
#endif
				pr_info(DRV_NAME
					": msg ts=%llu us type=%u id=%02xh "
					"len=%u < %s\n",
					msg->ts_us, msg->type, msg->id,
					msg->len, dev_full_name(usr_prv->dev));
#endif

			/* something new exists in rx fifo: wake reader up */
			wake_up_interruptible(&usr_prv->rx_wait);
			break;

		default:

			/* WTF is that message? */
			pr_warn(DRV_NAME
				": unknown msg [type=%u ts=%llu us "
				"id=%02xh len=%u] < %s: it is discarded\n",
				msg->type, msg->ts_us, msg->id,
				msg->len, dev_full_name(usr_prv->dev));

			usr_prv->dev->stats.rx_discarded++;

			/* return error value to stop parsing that buffer,
			 * filled with wrong msg record
			 */
			err = -EINVAL;
		}

	} else {
		pr_warn(DRV_NAME ": %s: msg type=%d lost (user Rx fifo full)\n",
			dev_full_name(usr_prv->dev), msg->type);

		usr_prv->dev->stats.rx_fifo_over++;
		if (msg->type == PLIN_MSG_FRAME)
			usr_prv->dev->stats.rx_dropped++;
	}

	return err;
}

/*
 * int plin_chrdev_rx_msg(struct plin_dev *dev, struct plin_msg *msg)
 *
 * Called when a LIN message is to be given to any reading user process.
 */
int plin_chrdev_rx_msg(struct plin_dev *dev, struct plin_msg *msg)
{
	plin_chrdev_foreach_user(dev, do_plin_rx_msg, msg);

	return 0;
}

/*
 * int plin_chrdev_init(void)
 *
 * Called when the LIN driver module is loaded.
 */
int plin_chrdev_init(void)
{
	int err;

	drv_major(&plin) = 0;

	err = register_chrdev(drv_major(&plin), DRV_NAME, &plin_fops);
	if (err < 0)  {
		pr_err(DRV_NAME ": failed to register chrdev (err %d)\n", err);

		return err;
	}

	drv_major(&plin) = err;

	return 0;
}

void plin_chrdev_exit(void)
{
	unregister_chrdev(drv_major(&plin), DRV_NAME);
}

#else /* PLIN_CHRDEV_API */

int plin_chrdev_init(void)
{
	drv_major(&plin) = 0;
	return 0;
}

void plin_chrdev_exit(void) {}

int plin_chrdev_init_dev(struct plin_dev *dev)
{
	return 0;
}
void plin_chrdev_free_dev(struct plin_dev *dev) {}

#endif /* PLIN_CHRDEV_API */
