/*
 * Copyright (C) 2025 Stephane Grosjean <stephane.grosjean@hms-networks.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the version 2 of the GNU General Public License
 * as published by the Free Software Foundation
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 */
#include "plin_main.h"

#include <linux/slab.h>
#include <linux/stringify.h>

MODULE_AUTHOR("Stephane Grosjean <stephane.grosjean@hms-networks.com>");
MODULE_DESCRIPTION("LIN driver for PEAK-System LIN interfaces family");
MODULE_LICENSE("GPL v2");
MODULE_VERSION(DRV_VER_STR);

#if defined(DEBUG)
#define DRV_BUILD_OPTS	"(-DDEBUG) "
#elif defined(DEBUG_TRACE)
#define DRV_BUILD_OPTS	"(-DDEBUG_TRACE) "
#elif defined(DEBUG_CHRDEV_TRACE)
#define DRV_BUILD_OPTS	"(-DDEBUG_CHRDEV_TRACE) "
#elif defined(DEBUG_USB_TRACE)
#define DRV_BUILD_OPTS	"(-DDEBUG_USB_TRACE) "
#else
#define DRV_BUILD_OPTS	""
#endif

/* the driver object */
struct plin_driver plin = {
	.name = DRV_NAME,
};

#ifndef NO_USB
#include "plin_usb.h"

#define PLIN_NOUSB_DEF		0

static uint nousb = PLIN_NOUSB_DEF;
module_param(nousb, uint, 0644);
MODULE_PARM_DESC(nousb, " ignore USB interfaces (default="
			__stringify(PLIN_NOUSB_DEF) ")");
#endif

#define PLIN_RXFIFO_LEN_MIN		16
#define PLIN_RXFIFO_LEN_MAX		512
#define PLIN_RXFIFO_LEN_DEF		128

uint rxfifolen = PLIN_RXFIFO_LEN_DEF;
module_param(rxfifolen, uint, 0644);
MODULE_PARM_DESC(rxfifolen,
		 " size of user Rx fifo (def="
		 __stringify(PLIN_RXFIFO_LEN_DEF) ")");

#define DUMP_MAX	64
#define DUMP_WIDTH	16
#ifdef DEBUG
#define DUMP_OUT	KERN_DEBUG
#else
#define DUMP_OUT	KERN_INFO
#endif

/*
 * void dump_mem(char *prompt, void *p, int l)
 */
void dump_mem(char *prompt, void *p, int l)
{
	printk(DUMP_OUT "%s: dumping %s (%d bytes):\n",
	       DRV_NAME, prompt ? prompt : "memory", l);
	print_hex_dump(DUMP_OUT, DRV_NAME ": ", DUMP_PREFIX_NONE,
		       DUMP_WIDTH, 1, p, l, false);
}

/*
 * u8 plin_frm_get_pid(u8 id)
 *
 * This function computes the Protect IDentifier Field of a LIN frame.
 *
 * The Protected IDentifier Field (PID) consists of 2 sub-fields:
 * - the frame identifier (made of bits 0 to 5)
 * - the parity (bits 6 and 7):
 *    b6 = IDO + ID1 + ID2 + ID4
 *    b7 = !(ID1 + ID3 + ID4 + ID5)
 *
 * Frame IDs are split into 3 categories:
 * 0x00 - 0x3b	normal signal/data carrying frames
 * 0x3c - 0x3d	diagnostic and configuration data frames (classic CS only)
 * 0x3e - 0x3f	reserved
 */
u8 plin_frm_get_pid(u8 id)
{
	u8 b6 = !!(id & (1 << 0)) ^ !!(id & (1 << 1)) ^
		!!(id & (1 << 2)) ^ !!(id & (1 << 4));
	u8 b7 = !(!!(id & (1 << 1)) ^ !!(id & (1 << 3)) ^
		!!(id & (1 << 4)) ^ !!(id & (1 << 5)));
	
	return (b7 << 7) | (b6 << 6) | (id & PLIN_FRM_ID_MASK);
}

/*
 * struct plin_intf *_plin_alloc_intf(int size_of, struct plin_intf_desc *desc)
 */
struct plin_intf *_plin_alloc_intf(int size_of, struct plin_intf_desc *desc)
{
	struct plin_intf *intf;

	if (size_of < sizeof(struct plin_intf))
		size_of = sizeof(struct plin_intf);

	size_of += desc->ctrl_count * sizeof(struct plin_dev *);
	intf = kzalloc(size_of, GFP_KERNEL);

#ifdef DEBUG_TRACE
	pr_info(DRV_NAME ": %s(): intf=%p\n", __func__, intf);
#endif

	if (intf) {
		intf->desc = desc;
		intf->same_idx = desc->intf_count++;
		intf->idx = -1;		/* unlinked */

		INIT_LIST_HEAD(&intf->dev_list);

		spin_lock_init(&intf->isr_lock);
	}

	return intf;
}

/*
 * void plin_free_intf(struct plin_intf *intf)
 */
void plin_free_intf(struct plin_intf *intf)
{
	struct list_head *ptr, *next_ptr;

#ifdef DEBUG_TRACE
	pr_info(DRV_NAME ": %s(intf=%p)\n", __func__, intf);
#endif

	/* free memory used by each controller */
	list_for_each_safe(ptr, next_ptr, &intf->dev_list) {
		struct plin_dev *dev = list_entry(ptr, struct plin_dev, link);

		plin_unlink_dev(dev);
		plin_free_dev(dev);
	}

	if (intf->free_intf)
		intf->free_intf(intf);

	--intf->desc->intf_count;

	kfree(intf);
}

void plin_link_intf(struct plin_intf *intf)
{
#ifdef DEBUG_TRACE
	pr_info(DRV_NAME ": %s(intf=%p)\n", __func__, intf);
#endif

	if (intf->idx >= 0)
		return;

	intf->idx = plin.intf_list_count++;

	list_add_tail(&intf->link, &plin.intf_list);
}

void plin_unlink_intf(struct plin_intf *intf)
{
	struct list_head *ptr;

#ifdef DEBUG_TRACE
	pr_info(DRV_NAME ": %s(intf=%p)\n", __func__, intf);
#endif

	/* unlink each device */
	list_for_each(ptr, &intf->dev_list) {
		struct plin_dev *dev = list_entry(ptr, struct plin_dev, link);

		plin_sysfs_destroy_dev(dev);
	}

	plin_sysfs_destroy_intf(intf);

	list_del(&intf->link);

	intf->idx = -1;

	plin.intf_list_count--;
}

/*
 * struct plin_intf *plin_foreach_intf(int (*f)(struct plin_intf *, void *arg),
 * 				       void *arg)
 */
struct plin_intf *plin_foreach_intf(int (*f)(struct plin_intf *, void *arg),
				    void *arg)
{
	struct list_head *pi;

#ifdef DEBUG_TRACE
	pr_info(DRV_NAME ": %s()\n", __func__);
#endif

	list_for_each(pi, &plin.intf_list) {
		struct plin_intf *intf = list_entry(pi, struct plin_intf, link);

		if (f(intf, arg))
			return intf;
	}

	return NULL;
}

/*
 * struct plin_dev *plin_alloc_dev(int sizeof_dev)
 */
struct plin_dev *_plin_alloc_dev(int sizeof_dev)
{
	int err;
	struct plin_dev *dev = kzalloc(sizeof_dev < sizeof(struct plin_dev) ? \
					sizeof(struct plin_dev) : sizeof_dev,
					GFP_KERNEL);

#ifdef DEBUG_TRACE
	pr_info(DRV_NAME ": %s(): dev=%p\n", __func__, dev);
#endif

	if (!dev)
		goto lbl_exit;

#ifdef USES_DEV_RXFIFO
	err = kfifo_alloc(&dev->rx_fifo,
			  rxfifolen * sizeof(struct plin_msg),
			  GFP_KERNEL);
	if (err)
		goto lbl_err;
#endif

	atomic_set(&dev->ref_count, 0);

	spin_lock_init(&dev->tx_lock);
	init_waitqueue_head(&dev->tx_wait);

	strncpy(dev->full_name, "LIN?", sizeof(dev->full_name));
	dev->idx = -1;

	err = plin_chrdev_init_dev(dev);
	if (!err)
		goto lbl_exit;

#ifdef USES_DEV_RXFIFO
lbl_err:
#endif
	kfree(dev);
	dev = NULL;

lbl_exit:
	return dev;
}

/*
 * void plin_free_dev(struct plin_dev *dev)
 */
void plin_free_dev(struct plin_dev *dev)
{
#ifdef DEBUG_TRACE
	pr_info(DRV_NAME ": %s(%s): ref_count=%d usr_count=%d\n", __func__,
		dev_full_name(dev), atomic_read(&dev->ref_count),
		dev_usr_count(dev));
#endif

	if (atomic_read(&dev->ref_count) > 0) {

		atomic_set(&dev->ref_count, 1 - atomic_read(&dev->ref_count));

		plin_chrdev_free_dev(dev);
		return;
	}

	if (dev->free_dev)
		dev->free_dev(dev);

#ifdef USES_DEV_RXFIFO
	kfifo_free(&dev->rx_fifo);
#endif
	kfree(dev);
}

/*
 * static int plin_is_minor_used(struct plin_dev *dev, void *arg)
 */
static int plin_is_minor_used(struct plin_dev *dev, void *arg)
{
	return dev_minor(dev) == *(int *)arg;
}

void plin_link_dev(struct plin_intf *intf, struct plin_dev *dev)
{
#ifdef DEBUG_TRACE
	pr_info(DRV_NAME ": %s(dev=%p)\n", __func__, dev);
#endif

	dev->intf = intf;
	dev->idx = intf->dev_list_count++;

	/* look for a free minor */
	for(dev_minor(dev) = 0;
		plin_foreach_dev(plin_is_minor_used, &dev_minor(dev));
						 	dev_minor(dev)++);

	list_add_tail(&dev->link, &intf->dev_list);

	snprintf(dev->full_name, sizeof(dev->full_name), "%s #%u LIN%u",
		dev_intf_name(dev), dev_intf_idx(dev), dev_idx(dev)+1);
}

void plin_unlink_dev(struct plin_dev *dev)
{
	if (dev->idx < 0)
		return;

#ifdef DEBUG_TRACE
	pr_info(DRV_NAME ": %s(%s)\n", __func__, dev_full_name(dev));
#endif

	dev->intf->dev_list_count--;
	plin.dev_list_count--;

	dev->intf = NULL;

	list_del(&dev->link);

	dev->idx = -1;
}

/*
 * static int plin_do_get_dev(struct plin_dev *dev, void *arg)
 */
static int plin_do_get_dev(struct plin_dev *dev, void *arg)
{
	return (dev == arg);
}

/*
 * struct plin_dev *__plin_get_dev(struct plin_dev *dev)
 *
 * Get the device object if it exists.
 */
struct plin_dev *__plin_get_dev(struct plin_dev *dev)
{
	struct plin_dev *the_dev = plin_foreach_dev(plin_do_get_dev, dev);
	if (the_dev) {

		/* the device can't be accessed and will be destroyed */
		if (atomic_read(&the_dev->ref_count) < 0)
			return NULL;

		atomic_inc(&the_dev->ref_count);
	}

	return the_dev;
}

/*
 * int __plin_put_dev(struct plin_dev *dev, int err)
 *
 * Give back the device object. If no more referenced, then it is destroyed.
 */
int __plin_put_dev(struct plin_dev *dev, int err)
{
	/* the device is waiting for everyone to put it before being
	 * destroyed...
	 */
	if (atomic_read(&dev->ref_count) < 0) {
		atomic_inc(&dev->ref_count);

	/* last user before deletion: */
	} else if (atomic_read(&dev->ref_count) == 0) {
		plin_free_dev(dev);
		err = -ENODEV;

	/* normal use: device is no more used by current user */
	} else if (atomic_read(&dev->ref_count) > 0) {
		atomic_dec(&dev->ref_count);
	}

	return err;
}

/*
 * struct plin_dev *plin_intf_foreach_dev(struct plin_intf *intf,
 * 					 int (*f)(struct plin_dev *, void *arg),
 *					 void *arg)
 */
struct plin_dev *plin_intf_foreach_dev(struct plin_intf *intf,
				      int (*f)(struct plin_dev *, void *arg),
				      void *arg)
{
	struct list_head *pd;

#ifdef DEBUG_TRACE
	pr_info(DRV_NAME ": %s(): intf=%p\n", __func__, intf);
#endif

	list_for_each(pd, &intf->dev_list) {
		struct plin_dev *dev = list_entry(pd, struct plin_dev, link);

		if (f(dev, arg))
			return dev;
	}

	return NULL;
}

/*
 * struct plin_dev *plin_foreach_dev(int (*f)(struct plin_dev *, void *arg),
 * 				     void *arg)
 */
struct plin_dev *plin_foreach_dev(int (*f)(struct plin_dev *, void *arg),
				  void *arg)
{
	struct list_head *pi;

#ifdef DEBUG_TRACE
	pr_info(DRV_NAME ": %s()\n", __func__);
#endif

	list_for_each(pi, &plin.intf_list) {
		struct plin_intf *intf = list_entry(pi, struct plin_intf, link);
		struct plin_dev *dev = plin_intf_foreach_dev(intf, f, arg);

		if (dev)
			return dev;
	}

	return NULL;
}

/*
 * int plin_rx_msg(struct plin_dev *dev, struct plin_msg *msg)
 */
int plin_rx_msg(struct plin_dev *dev, struct plin_msg *msg)
{
#ifdef DEBUG_TRACE
	pr_info(DRV_NAME ": %s()\n", __func__);
#endif
	if (dev_usr_count(dev) > 0)
		plin_chrdev_rx_msg(dev, msg);

#ifdef USES_DEV_RXFIFO
	else if (!kfifo_is_full(&dev->rx_fifo))
		kfifo_in(&dev->rx_fifo, msg, sizeof(struct plin_msg));
#endif
	else
		dev->stats.rx_lost++;

	dev->stats.rx_msgs++;
	dev->stats.rx_bytes += msg->len;

	switch (msg->type) {
	case PLIN_MSG_FRAME:
		dev->stats.rx_frames++;

		/* fall through */
		fallthrough;
	case PLIN_MSG_WAKEUP:
		dev->flags &= ~PLIN_DEV_BUS_SLEEP;

		/* fall through */
		fallthrough;
	default:
		break;

	case PLIN_MSG_SLEEP:
		/* "Only the timestamp is valid." */
		dev->sleep_ts = msg->ts_us;
		dev->flags |= PLIN_DEV_BUS_SLEEP;
		break;
	}

	return 0;
}

/*
 * int __init plin_init(void)
 */
static int __init plin_init(void)
{
	int err;

#ifdef DEBUG_TRACE
	pr_info(DRV_NAME ": %s()\n", __func__);
#endif

	if (rxfifolen > PLIN_RXFIFO_LEN_MAX)
		rxfifolen = PLIN_RXFIFO_LEN_MAX;
	else if (rxfifolen < PLIN_RXFIFO_LEN_MIN)
		rxfifolen = PLIN_RXFIFO_LEN_MIN;

	INIT_LIST_HEAD(&plin.intf_list);

	//pr_info(DRV_NAME ": loading driver v%s\n", DRV_VER_STR);

	err = plin_chrdev_init();
	if (err)
		return err;

#ifndef NO_USB
	if (!nousb) {
		err = plin_usb_init();
		if (err)
			return err;
	}
#endif

	plin_sysfs_init();

	pr_info(DRV_NAME ": LIN driver %sv%s loaded\n",
		DRV_BUILD_OPTS, DRV_VER_STR);

	return 0;
}

/*
 * static int plin_do_cleanup(struct plin_dev *dev, void *arg)
 *
 * Do cleanup a device.
 */
static int plin_do_cleanup(struct plin_dev *dev, void *arg)
{
#ifdef DEBUG_TRACE
	pr_info(DRV_NAME ": %s(dev=%p)\n", __func__, dev);
#endif

	//wake_up_interruptible(&dev->rx_wait);

	if (dev->cleanup)
		dev->cleanup(dev);

	return 0;
}

/*
 * void __exit plin_exit(void)
 */
static void __exit plin_exit(void)
{
#ifdef DEBUG_TRACE
	pr_info(DRV_NAME ": %s()\n", __func__);
#endif

	/* do disconnect all opened sessions now!
	 * (especially before calling usb_deregister())
	 */
	plin_foreach_dev(plin_do_cleanup, NULL);

#ifndef NO_USB
	if (!nousb)
		plin_usb_exit();
#endif

	plin_sysfs_exit();

	plin_chrdev_exit();

	pr_info(DRV_NAME ": removed\n");
}

module_init(plin_init);
module_exit(plin_exit);
