/* SPDX-License-Identifier: GPL-2.0-only */
/*
 * Copyright (C) 2025 Stephane Grosjean <stephane.grosjean@hms-networks.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the version 2 of the GNU General Public License
 * as published by the Free Software Foundation
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 */
#ifndef __PLIN_MAIN_H__
#define __PLIN_MAIN_H__

#ifndef REMOVE_PRIVATE_SECTION
/* stand-alone debugging... */
//#define DEBUG
//#undef DEBUG
#endif

#ifdef DEBUG
#define DEBUG_TRACE
#define DEBUG_TRACE_PUT_GET
#define DEBUG_RX
#define DEBUG_TX
#else
//#define DEBUG_TRACE
//#define DEBUG_TRACE_PUT_GET
//#define DEBUG_RX
//#define DEBUG_TX
#endif

/* if defined, only debug this minor */
//#define DEBUG_MINOR		4

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/kfifo.h>

#define PLIN_SYSFS_API
#define PLIN_CHRDEV_API

#include "plin.h"
#include "plin_version.h"
#include "plin_sysfs.h"
#include "plin_chrdev.h"

#define DRV_NAME		"plin"

#define _DRV_VER_STR		__stringify(DRV_VER_MAJOR)"."\
				__stringify(DRV_VER_MINOR)"."\
				__stringify(DRV_VER_SUBMINOR)

#ifdef DRV_VER_RC
#define DRV_VER_STR		_DRV_VER_STR"-rc"__stringify(DRV_VER_RC)
#else
#define DRV_VER_STR		_DRV_VER_STR
#endif

#ifndef fallthrough
#define fallthrough
#endif

/* LIN frame identifier range is 0..63 */
#define PLIN_FRM_ID_MASK	PLIN_FRM_ID_MAX

struct plin_driver {
	char *name;
	struct plin_driver_chrdev chrdev;
	struct class *class;
	struct list_head intf_list;
	int intf_list_count;
	int dev_list_count;
};

#define drv_major(drv)		((drv)->chrdev.major)

struct plin_intf;

struct plin_intf_desc {
	char *name;
	u32 devid;
	int ctrl_count;

	int intf_count;
};

#define PLIN_INTF(n, d, c)	\
	{ .name = n, .devid = d, .ctrl_count = c, }

struct plin_dev;

struct plin_intf {
	struct list_head link;
	spinlock_t isr_lock;
	int idx;
	int same_idx;
	struct plin_intf_desc *desc;
	struct list_head dev_list;
	int dev_list_count;
	int fw_ver[3];
	struct device *sysfs_dev;

	void (*free_intf)(struct plin_intf *intf);
};

#define intf_minor(i)		((i)->idx + 100)
#define intf_idx(i)		((i)->idx)
#define intf_desc(i)		((i)->desc)
#define intf_name(i)		(intf_desc(i)->name)
#define intf_ctrl_count(i)	(intf_desc(i)->ctrl_count)

struct plin_dev_stats {
	unsigned long tx_msgs;
	unsigned long rx_msgs;
	unsigned long rx_frames;
	unsigned long rx_bytes;
	unsigned long rx_fifo_over;
	unsigned long rx_lost;
	unsigned long rx_dropped;
	unsigned long rx_discarded;
	unsigned long tx_frames;
	unsigned long tx_bytes;
};

#define PLIN_DEV_BUS_SLEEP	0x00000001UL

struct plin_dev {
	struct list_head link;
	char full_name[128];

	int idx;
	struct plin_intf *intf;

	int mode;
	u16 baudrate;
	u32 flags;
	u64 sleep_ts;

	atomic_t ref_count;

	struct plin_dev_sysfs sysfs;
	struct plin_dev_stats stats;
	struct plin_chrdev_data chrdev;

	spinlock_t tx_lock;
	wait_queue_head_t tx_wait;

	struct plin_usb_frm_entry frm_entry_cache[PLIN_FRM_ID_MAX+1];

#ifdef USES_DEV_RXFIFO
	struct kfifo rx_fifo;
#endif
	void (*free_dev)(struct plin_dev *dev);

	int (*identify)(struct plin_dev *dev);
	int (*get_id_str_nolock)(struct plin_dev *dev, char *buf, int len);
	int (*set_id_str)(struct plin_dev *dev, const char *buf, int len);
	int (*get_id_num_nolock)(struct plin_dev *dev, u32 *user_num);
	int (*set_id_num)(struct plin_dev *dev, u32 user_num);
	int (*set_led_state)(struct plin_dev *dev, u8 state);

	int (*write)(struct plin_dev *dev, struct plin_msg *msg);
	int (*write_busy)(struct plin_dev *dev);

	void (*cleanup)(struct plin_dev *dev);

	int (*send_direct_cmd)(struct plin_dev *dev, void *b, int l);
	int (*read_direct_rsp)(struct plin_dev *dev, void *b, int l);
};

#define dev_idx(d)		((d)->idx)
#define dev_intf(d)		((d)->intf)
#define dev_full_name(d)	((d)->full_name)
#define dev_intf_name(d)	intf_name(dev_intf(d))
#define dev_intf_idx(d)		intf_idx(dev_intf(d))

#define plin_alloc_intf(t, d)	(t *)_plin_alloc_intf(sizeof(t), d)
#define plin_alloc_dev(t)	(t *)_plin_alloc_dev(sizeof(t))

extern uint rxfifolen;
extern struct plin_driver plin;

extern void dump_mem(char *prompt, void *p, int l);

extern u8 plin_frm_get_pid(u8 id);

extern struct plin_intf *_plin_alloc_intf(int size_of, struct plin_intf_desc *);
extern void plin_free_intf(struct plin_intf *intf);

extern void plin_link_intf(struct plin_intf *intf);
extern void plin_unlink_intf(struct plin_intf *intf);

extern struct plin_dev *_plin_alloc_dev(int sizeof_dev);
extern void plin_free_dev(struct plin_dev *dev);

extern void plin_link_dev(struct plin_intf *intf, struct plin_dev *dev);
extern void plin_unlink_dev(struct plin_dev *dev);

extern struct plin_dev *__plin_get_dev(struct plin_dev *dev);

#ifdef DEBUG_TRACE_PUT_GET
static inline struct plin_dev *_plin_get_dev(struct plin_dev *dev,
					     const char *f, unsigned int l)
{
	pr_info(DRV_NAME ": %s(l=%u): get(): ref=%d\n", f, l,
		atomic_read(&dev->ref_count));
	return __plin_get_dev(dev);
}

#define plin_get_dev(d)		_plin_get_dev(d, __func__, __LINE__)
#else
#define plin_get_dev(d)		__plin_get_dev(d)
#endif

extern int __plin_put_dev(struct plin_dev *dev, int err);

#ifdef DEBUG_TRACE_PUT_GET
static inline int _plin_put_dev(struct plin_dev *dev, int err,
				const char *f, unsigned int l)
{
	pr_info(DRV_NAME ": %s(l=%u): put(err=%d): ref=%d\n", f, l,
		err, atomic_read(&dev->ref_count));
	return __plin_put_dev(dev, err);
}

#define plin_put_dev(d, e)		_plin_put_dev(d, e, __func__, __LINE__)
#else
#define plin_put_dev(d, e)		__plin_put_dev(d, e)
#endif

extern struct plin_intf *plin_foreach_intf(int (*f)(struct plin_intf *,
						    void *arg), void *arg);

extern struct plin_dev *plin_intf_foreach_dev(struct plin_intf *intf,
					int (*f)(struct plin_dev *, void *arg),
					void *arg);
extern struct plin_dev *plin_foreach_dev(int (*f)(struct plin_dev *, void *arg),
					void *arg);

extern int plin_rx_msg(struct plin_dev *dev, struct plin_msg *msg);

#endif /* __PLIN_MAIN_H__ */
