/* SPDX-License-Identifier: GPL-2.0-only */
/*
 * Copyright (C) 2025 Stephane Grosjean <stephane.grosjean@hms-networks.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the version 2 of the GNU General Public License
 * as published by the Free Software Foundation
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 */
#include "plin_main.h"

#if defined(DEBUG)
#define DEBUG_SYSFS
#elif defined(DEBUG_TRACE)
#define DEBUG_SYSFS_TRACE
#endif

#ifdef DEBUG_SYSFS
#define DEBUG_SYSFS_TRACE
#endif

#include <linux/device.h>
#include <linux/ctype.h>

static char *plin_mode_str[] = {
	[PLIN_MODE_NONE] = "none",
	[PLIN_MODE_SLAVE] = "slave",
	[PLIN_MODE_MASTER] = "master",
};

const char *plin_sysfs_get_mode_str(int mode)
{
	return (mode < ARRAY_SIZE(plin_mode_str) && plin_mode_str[mode]) ?
		plin_mode_str[mode] : NULL;
}

#ifdef PLIN_SYSFS_API

#include <linux/kdev_t.h>

/* linux < 2.6.27 use device_create_drvdata() */
#ifndef device_create_drvdata
#define device_create_drvdata   device_create
#endif

/* write permissions:
 * S_IWUSR	only user root is allowed to write
 * S_IWUGO	everyone is allowed to write
 */

#if LINUX_VERSION_CODE < KERNEL_VERSION(3, 11, 0)
static int plin_sysfs_add_attr(struct device *dev, struct attribute *attr)
{
	return sysfs_add_file_to_group(&dev->kobj, attr, NULL);
}

static int plin_sysfs_add_attrs(struct device *dev, struct attribute **attrs)
{
	int err = 0;
	struct attribute **ppa;

#ifdef DEBUG_SYSFS_TRACE
	pr_info(DRV_NAME ": %s(%p=\"%s\")\n", __func__, dev, dev->kobj.name);
#endif
	for (ppa = attrs; *ppa; ppa++) {
		err = plin_sysfs_add_attr(dev, *ppa);
		if (err) {
			pr_err(DRV_NAME
				": failed to add \"%s\" to \"%s\" (err %d)\n",
				(*ppa)->name, dev->kobj.name, err);
			break;
		}
	}

	return err;
}

static void plin_sysfs_del_attr(struct device *dev, struct attribute *attr)
{
	sysfs_remove_file_from_group(&dev->kobj, attr, NULL);
}

static void plin_sysfs_del_attrs(struct device *dev, struct attribute **attrs)
{
	struct attribute **ppa;

#ifdef DEBUG_SYSFS_TRACE
	pr_info(DRV_NAME ": %s(%p=\"%s\")\n", __func__, dev, dev->kobj.name);
#endif
	for (ppa = attrs; *ppa; ppa++)
		plin_sysfs_del_attr(dev, *ppa);
}
#endif

static inline ssize_t show_non_0_u32(char *buf, u32 v)
{
	int l = 0;

	if (v > 0)
		l = snprintf(buf, PAGE_SIZE-1, "%u", v);

	buf[l++] = '\n';

	return l;
}

static inline ssize_t show_u32(char *buf, u32 v)
{
	return snprintf(buf, PAGE_SIZE, "%u\n", v);
}

static inline ssize_t show_int(char *buf, int v)
{
	return snprintf(buf, PAGE_SIZE, "%d\n", v);
}

static inline ssize_t show_str(char *buf, char *str)
{
	return snprintf(buf, PAGE_SIZE, "%s\n", str);
}

static inline struct plin_intf *to_plin_intf(struct device *dev)
{
	return (struct plin_intf *)dev_get_drvdata(dev);
}

static ssize_t show_fw_ver(struct device *dev,
			   struct device_attribute *attr, char *buf)
{
	struct plin_intf *intf = to_plin_intf(dev);
	int l = 0;

	if (intf->fw_ver[0] >= 0) {
		l = snprintf(buf+l, PAGE_SIZE, "%u", intf->fw_ver[0]);
		if (intf->fw_ver[1] >= 0) {
			l += snprintf(buf+l, PAGE_SIZE-l, ".%u",
					intf->fw_ver[1]);

			if (intf->fw_ver[2] >= 0)
				l += snprintf(buf+l, PAGE_SIZE-l, ".%u",
						intf->fw_ver[2]);
		}

		if (l >= PAGE_SIZE)
			l = PAGE_SIZE - 1;
	}

	buf[l++] = '\n';

	return l;
}

#define PLIN_INTF_ATTR(_v, _name, _show) \
	struct device_attribute plin_intf_attr_##_v = \
				__ATTR(_name, S_IRUGO, _show, NULL)

#define PLIN_INTF_ATTR_RW(_v, _name, _show, _store) \
	struct device_attribute plin_intf_attr_##_v = \
				__ATTR(_name, S_IRUGO|S_IWUSR, _show, _store)

static PLIN_INTF_ATTR(fw_ver, fw_ver, show_fw_ver);

static struct attribute *plin_sysfs_intf_attrs[] = {
	&plin_intf_attr_fw_ver.attr,
	NULL
};

#if LINUX_VERSION_CODE >= KERNEL_VERSION(3, 11, 0)
static struct attribute_group plin_sysfs_intf_attrs_group = {
	/* NULL ".name" => attrs will be created under plinxxx node 
	 * .name = "plin-dev", 
	 */
	.attrs = plin_sysfs_intf_attrs,
};
static const struct attribute_group *plin_sysfs_intf_attrs_groups[] = {
	&plin_sysfs_intf_attrs_group,
	NULL,
};
#endif

/*
 * Create a node for a LIN interface
 */
static
void __plin_sysfs_create_intf_ex(struct plin_intf *intf, struct device *parent)
{
	char tmp[32], *ps, *pd;

	/* no need to create anything if sysfs device not created */
	if (!plin.class) {
		intf->sysfs_dev = NULL;
		return;
	}

	for (ps = intf_name(intf), pd = tmp; *ps; ps++)
		if ((*ps >= 'A') && (*ps <= 'Z'))
			*pd++ = *ps + 0x20;
		else if (*ps <= ' ')
			*pd++ = '_';
		else
			*pd++ = *ps;

	*pd = '\0';

#ifdef DEBUG_SYSFS_TRACE
	pr_info(DRV_NAME ": %s(%s, %d): tmp=\"%s\"\n",
		__func__, intf_name(intf), intf->idx, tmp);
#endif

	/* Note: when using something else than MKDEV(0, 0), then Udev will
	 * create a corresponding entry under /dev. Here, we don't want any
	 * /dev/pcan-usb_pro_fd-0 entry!
	 */
#if LINUX_VERSION_CODE < KERNEL_VERSION(3, 11, 0)
	/* Should use WAIT_FOR key in Udev rules... */
	intf->sysfs_dev = device_create_drvdata(plin.class, parent,
				//MKDEV(drv_major(&plin), intf_minor(intf)),
				MKDEV(0, 0),
				intf, "%s-%u", tmp, intf_idx(intf));

	if (!IS_ERR(intf->sysfs_dev)) {
		plin_sysfs_add_attrs(intf->sysfs_dev, plin_sysfs_intf_attrs);
#else
	/* since 3.11, it is possible to add attrs when creating the device
	 * node, that is *BEFORE* the UEVENT is being sent to userspace!
	 * Doing this, Udev rules does not need of WAIT_FOR key anymore! */
	intf->sysfs_dev = device_create_with_groups(plin.class, parent,
				//MKDEV(drv_major(&plin), intf_minor(intf)),
				MKDEV(0, 0),
				intf, plin_sysfs_intf_attrs_groups,
				"%s-%u", tmp, intf_idx(intf));

	if (!IS_ERR(intf->sysfs_dev)) {
#endif /* KERNEL_VERSION(3, 11, 0) */

#ifdef DEBUG_SYSFS
		pr_info(DRV_NAME ": %s(%p=\"%s\")\n", __func__, intf,
			intf->sysfs_dev->kobj.name);
#endif
	} else {
		pr_warn(DRV_NAME ": %s(): device_create() failure err=%ld\n",
			__func__, (long )IS_ERR(intf->sysfs_dev));

		intf->sysfs_dev = NULL;
	}
}

static int plin_sysfs_do_create_dev(struct plin_dev *dev, void *arg)
{
	plin_sysfs_create_dev(dev);
	return 0;
}

static int plin_sysfs_do_create_intf(struct plin_intf *intf, void *arg)
{
	__plin_sysfs_create_intf_ex(intf, arg);
	plin_intf_foreach_dev(intf, plin_sysfs_do_create_dev, arg);

	return 0;
}

void plin_sysfs_create_intf(struct plin_intf *intf)
{
	plin_sysfs_do_create_intf(intf, NULL);
}

/*
 * While no chrdev driver is register, can't rely on any major number,
 * therefore, can't rely on device_destroy() to delete any created device.
 * Do as drivers/gpio/gpiolib-sysfs.c does, that is handle the deletion of the
 * device by ourself.
 */
#if LINUX_VERSION_CODE < KERNEL_VERSION(3, 9, 0)
static int plin_sysfs_match_dev(struct device *dev, void *arg)
#else
static int plin_sysfs_match_dev(struct device *dev, const void *arg)
#endif
{
	return dev == arg;
}

static void __plin_sysfs_destroy_dev(struct device *the_dev)
{
#if 0
	/* can't use device_destroy() here while interfaces are created
	 * with MKDEV(0,0)
	 */
	device_destroy(plin.class, MKDEV(drv_major(&plin), intf_minor(intf)));
#else
	struct device *dev;

	//mutex_lock(&sysfs_lock);
	dev = class_find_device(plin.class, NULL, the_dev,
				plin_sysfs_match_dev);

	if (!dev)
		goto lbl_unlock;

	/* see device_destroy() */
	device_unregister(dev);

lbl_unlock:
	//mutex_unlock(&sysfs_lock);

	put_device(dev);
#endif
}

/*
 * Destroy sysfs interface node
 */
void plin_sysfs_destroy_intf(struct plin_intf *intf)
{
	if (intf->sysfs_dev) {
#ifdef DEBUG_SYSFS_TRACE
		pr_info(DRV_NAME ": %s(%p=\"%s\")\n", __func__, intf,
			intf->sysfs_dev->kobj.name);
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(3, 11, 0)
		plin_sysfs_del_attrs(intf->sysfs_dev, plin_sysfs_intf_attrs);
#endif

		__plin_sysfs_destroy_dev(intf->sysfs_dev);
		intf->sysfs_dev = NULL;
	}
}

#define PLIN_DEVICE_ATTR(_v, _name, _show) \
	struct device_attribute plin_dev_attr_##_v = \
				__ATTR(_name, S_IRUGO, _show, NULL)

#define PLIN_DEVICE_ATTR_RW(_v, _name, _show, _store) \
	struct device_attribute plin_dev_attr_##_v = \
				__ATTR(_name, S_IRUGO|S_IWUSR, _show, _store)

static inline struct plin_dev *to_plin_dev(struct device *_dev)
{
	return (struct plin_dev *)dev_get_drvdata(_dev);
}

static ssize_t plin_sysfs_get_led(struct device *_dev,
				  struct device_attribute *attr, char *buf)
{
#ifdef DEBUG_SYSFS_TRACE
	struct plin_dev *dev = to_plin_dev(_dev);

	pr_info(DRV_NAME ": %s(%s)\n", __func__, dev_full_name(dev));
#endif

	return show_u32(buf, 0);
}

static ssize_t plin_sysfs_set_led(struct device *_dev,
				  struct device_attribute *attr,
				  const char *buf, size_t count)
{
	char *endptr;
	struct plin_dev *dev = to_plin_dev(_dev);
	u32 tmp32 = simple_strtoul(buf, &endptr, 0);

#ifdef DEBUG_SYSFS_TRACE
	pr_info(DRV_NAME ": %s: %s(buf=\"%s\", count=%u)\n",
		dev_full_name(dev), __func__, buf, (unsigned int)count);
#endif

	if (*endptr != '\n')
		return -EINVAL;

	switch (tmp32) {
	
	default:
		if (dev->identify) {
			int err = dev->identify(dev);
			if (err)
				return err;

			break;
		}

		/* fall through */
		fallthrough;
	case 0:
	case 1:
		if (dev->set_led_state) {
			int err = dev->set_led_state(dev, tmp32);
			if (err)
				return err;

			break;
		}
	}

	return count;
}

static ssize_t plin_sysfs_get_baudrate(struct device *_dev,
				       struct device_attribute *attr, char *buf)
{
	struct plin_dev *dev = to_plin_dev(_dev);

#ifdef DEBUG_SYSFS_TRACE
	pr_info(DRV_NAME ": %s(%s)\n", __func__, dev_full_name(dev));
#endif

	return show_non_0_u32(buf, dev->baudrate);
}

static ssize_t plin_sysfs_get_mode(struct device *_dev,
				   struct device_attribute *attr, char *buf)
{
	struct plin_dev *dev = to_plin_dev(_dev);
	const char *mode_str;
	int l = 0;

#ifdef DEBUG_SYSFS_TRACE
	pr_info(DRV_NAME ": %s(%s)\n", __func__, dev_full_name(dev));
#endif

	mode_str = plin_sysfs_get_mode_str(dev->mode);
	if (mode_str) {
		//l = snprintf(buf, PAGE_SIZE, mode_str);
		l = strscpy(buf, mode_str, PAGE_SIZE-1);
	}

	buf[l++] = '\n';

	return l;
}

static ssize_t plin_sysfs_get_id_str(struct device *_dev,
				     struct device_attribute *attr, char *buf)
{
	struct plin_dev *dev = to_plin_dev(_dev);
	int i, l;

#ifdef DEBUG_SYSFS_TRACE
	pr_info(DRV_NAME ": %s(%s)\n", __func__, dev_full_name(dev));
#endif

	if (!dev->get_id_str_nolock)
		return -ENOTSUPP;

	l = dev->get_id_str_nolock(dev, buf, PAGE_SIZE-1);
	if (l < 0)
		return l;

	/* since it might be a non-null-terminated string, search the
	 * end of the string now. 0xff means not used.
	 */
	for (i = 0; i < l && buf[i] && buf[i] != 0xff; i++) {
		if (!isascii(buf[i]) || !isprint(buf[i]))
			buf[i] = '.';
	}
	
	buf[i++] = '\n';

	return i;
}

static ssize_t plin_sysfs_set_id_str(struct device *_dev,
				     struct device_attribute *attr,
				     const char *buf, size_t count)
{
	struct plin_dev *dev = to_plin_dev(_dev);
	int err;

#ifdef DEBUG_SYSFS_TRACE
	pr_info(DRV_NAME ": %s: %s(buf=\"%s\", count=%u)\n",
		dev_full_name(dev), __func__, buf, (unsigned int)count);
#endif

	if (!dev->set_id_str)
		return -ENOTSUPP;

	/* remove ending \n */
	err = dev->set_id_str(dev, buf, count-1);
	if (err < 0)
		return err;

	return count;
}

static ssize_t plin_sysfs_get_id_num(struct device *_dev,
				     struct device_attribute *attr, char *buf)
{
	struct plin_dev *dev = to_plin_dev(_dev);
	u32 user_num;
	int err;

#ifdef DEBUG_SYSFS_TRACE
	pr_info(DRV_NAME ": %s(%s)\n", __func__, dev_full_name(dev));
#endif

	if (!dev->get_id_num_nolock)
		return -ENOTSUPP;

	err = dev->get_id_num_nolock(dev, &user_num);
	if (err < 0)
		return err;

	if (user_num == 0xffffffff) {
		*buf = '\n';
		return 1;
	}

	return show_u32(buf, user_num);
}

static ssize_t plin_sysfs_set_id_num(struct device *_dev,
				     struct device_attribute *attr,
				     const char *buf, size_t count)
{
	struct plin_dev *dev = to_plin_dev(_dev);
	char *endptr;
	u32 user_num;
	int err;

#ifdef DEBUG_SYSFS_TRACE
	pr_info(DRV_NAME ": %s: %s(buf=\"%s\", count=%u)\n",
		dev_full_name(dev), __func__, buf, (unsigned int)count);
#endif

	if (!dev->set_id_num)
		return -ENOTSUPP;

	user_num = simple_strtoul(buf, &endptr, 0);
	if (*endptr != '\n')
		return -EINVAL;

	err = dev->set_id_num(dev, user_num);
	if (err < 0)
		return err;

	return count;
}

static PLIN_DEVICE_ATTR_RW(led, led, plin_sysfs_get_led, plin_sysfs_set_led);
static PLIN_DEVICE_ATTR_RW(id_str, id_str,
			   plin_sysfs_get_id_str, plin_sysfs_set_id_str);
static PLIN_DEVICE_ATTR_RW(id_num, id_num,
			   plin_sysfs_get_id_num, plin_sysfs_set_id_num);
static PLIN_DEVICE_ATTR(baudrate, baudrate, plin_sysfs_get_baudrate);
static PLIN_DEVICE_ATTR(mode, mode, plin_sysfs_get_mode);

static struct attribute *plin_sysfs_dev_common_attrs[] = {
	&plin_dev_attr_led.attr,
	&plin_dev_attr_id_str.attr,
	&plin_dev_attr_id_num.attr,
	&plin_dev_attr_baudrate.attr,
	&plin_dev_attr_mode.attr,
	NULL
};

static ssize_t plin_sysfs_get_rx_frames(struct device *_dev,
				struct device_attribute *attr, char *buf)
{
	struct plin_dev *dev = to_plin_dev(_dev);

#ifdef DEBUG_SYSFS_TRACE
	pr_info(DRV_NAME ": %s(%s)\n", __func__, dev_full_name(dev));
#endif

	return show_u32(buf, dev->stats.rx_frames);
}

static ssize_t plin_sysfs_get_rx_bytes(struct device *_dev,
				struct device_attribute *attr, char *buf)
{
	struct plin_dev *dev = to_plin_dev(_dev);

#ifdef DEBUG_SYSFS_TRACE
	pr_info(DRV_NAME ": %s(%s)\n", __func__, dev_full_name(dev));
#endif

	return show_u32(buf, dev->stats.rx_bytes);
}

static ssize_t plin_sysfs_get_rx_over_errors(struct device *_dev,
				struct device_attribute *attr, char *buf)
{
	struct plin_dev *dev = to_plin_dev(_dev);

#ifdef DEBUG_SYSFS_TRACE
	pr_info(DRV_NAME ": %s(%s)\n", __func__, dev_full_name(dev));
#endif

	return show_u32(buf, dev->stats.rx_fifo_over);
}

static ssize_t plin_sysfs_get_rx_dropped(struct device *_dev,
				struct device_attribute *attr, char *buf)
{
	struct plin_dev *dev = to_plin_dev(_dev);

#ifdef DEBUG_SYSFS_TRACE
	pr_info(DRV_NAME ": %s(%s)\n", __func__, dev_full_name(dev));
#endif

	return show_u32(buf, dev->stats.rx_dropped);
}

static ssize_t plin_sysfs_get_rx_discarded(struct device *_dev,
				struct device_attribute *attr, char *buf)
{
	struct plin_dev *dev = to_plin_dev(_dev);

#ifdef DEBUG_SYSFS_TRACE
	pr_info(DRV_NAME ": %s(%s)\n", __func__, dev_full_name(dev));
#endif

	return show_u32(buf, dev->stats.rx_discarded);
}

static ssize_t plin_sysfs_get_rx_lost(struct device *_dev,
				      struct device_attribute *attr, char *buf)
{
	struct plin_dev *dev = to_plin_dev(_dev);

#ifdef DEBUG_SYSFS_TRACE
	pr_info(DRV_NAME ": %s(%s)\n", __func__, dev_full_name(dev));
#endif

	return show_u32(buf, dev->stats.rx_lost);
}

static ssize_t plin_sysfs_get_tx_frames(struct device *_dev,
				struct device_attribute *attr, char *buf)
{
	struct plin_dev *dev = to_plin_dev(_dev);

#ifdef DEBUG_SYSFS_TRACE
	pr_info(DRV_NAME ": %s(%s)\n", __func__, dev_full_name(dev));
#endif

	return show_u32(buf, dev->stats.tx_frames);
}

static ssize_t plin_sysfs_get_tx_bytes(struct device *_dev,
				struct device_attribute *attr, char *buf)
{
	struct plin_dev *dev = to_plin_dev(_dev);

#ifdef DEBUG_SYSFS_TRACE
	pr_info(DRV_NAME ": %s(%s)\n", __func__, dev_full_name(dev));
#endif

	return show_u32(buf, dev->stats.tx_bytes);
}

static PLIN_DEVICE_ATTR(rx_frames, rx_frames, plin_sysfs_get_rx_frames);
static PLIN_DEVICE_ATTR(rx_bytes, rx_bytes, plin_sysfs_get_rx_bytes);
static PLIN_DEVICE_ATTR(rx_over_errors, rx_over_errors,
			plin_sysfs_get_rx_over_errors);
static PLIN_DEVICE_ATTR(rx_dropped, rx_dropped, plin_sysfs_get_rx_dropped);
static PLIN_DEVICE_ATTR(rx_discarded, rx_discarded, plin_sysfs_get_rx_discarded);
static PLIN_DEVICE_ATTR(rx_lost, rx_lost, plin_sysfs_get_rx_lost);
static PLIN_DEVICE_ATTR(tx_frames, tx_frames, plin_sysfs_get_tx_frames);
static PLIN_DEVICE_ATTR(tx_bytes, tx_bytes, plin_sysfs_get_tx_bytes);

static struct attribute *plin_sysfs_dev_stats_attrs[] = {
	&plin_dev_attr_rx_bytes.attr,
	&plin_dev_attr_rx_frames.attr,
	&plin_dev_attr_rx_over_errors.attr,
	&plin_dev_attr_rx_dropped.attr,
	&plin_dev_attr_rx_discarded.attr,
	&plin_dev_attr_rx_lost.attr,
	&plin_dev_attr_tx_bytes.attr,
	&plin_dev_attr_tx_frames.attr,
	NULL
};

#if LINUX_VERSION_CODE >= KERNEL_VERSION(3, 11, 0)
static struct attribute_group plin_sysfs_dev_attrs_group = {
	.attrs = plin_sysfs_dev_common_attrs,
};

static struct attribute_group plin_sysfs_dev_stats_attrs_group = {
	.name = "statistics",
	.attrs = plin_sysfs_dev_stats_attrs,
};

static const struct attribute_group *plin_sysfs_dev_attrs_groups[] = {
	&plin_sysfs_dev_attrs_group,
	&plin_sysfs_dev_stats_attrs_group,
	NULL,
};
#endif

/*
 * create a device node to export devices properties into /sys/class/plin tree.
 */
void plin_sysfs_create_dev_ex(struct plin_dev *dev, struct device *parent)
{
#ifdef DEBUG_SYSFS_TRACE
	pr_info(DRV_NAME ": %s(%p, %d, %d)\n",
		__func__, dev, drv_major(&plin), dev_minor(dev));
#endif
	/* no need to create anything is sysfs device not created */
	if (!plin.class || !dev_intf(dev)->sysfs_dev) {
		dev->sysfs.dev = NULL;
		return;
	}

#if LINUX_VERSION_CODE < KERNEL_VERSION(3, 11, 0)
	/* Should use WAIT_FOR key in Udev rules... */
	dev->sysfs.dev = device_create_drvdata(plin.class,
				dev_intf(dev)->sysfs_dev,
				MKDEV(drv_major(&plin), dev_minor(dev)),
				dev, DRV_NAME "%u", dev_minor(dev));

	if (!IS_ERR(dev->sysfs.dev)) {
		plin_sysfs_add_attrs(dev->sysfs.dev,
				     plin_sysfs_dev_common_attrs);
		plin_sysfs_add_attrs(dev->sysfs.dev,
				     plin_sysfs_dev_stats_attrs);
#else
	/* since 3.11, it is possible to add attrs when creating the device
	 * node, that is *BEFORE* the UEVENT is being sent to userspace!
	 * Doing this, Udev rules does not need of WAIT_FOR key anymore!
	 */
	dev->sysfs.dev = device_create_with_groups(plin.class,
				dev_intf(dev)->sysfs_dev,
				MKDEV(drv_major(&plin), dev_minor(dev)),
				dev, plin_sysfs_dev_attrs_groups,
				DRV_NAME "%u", dev_minor(dev));

	if (!IS_ERR(dev->sysfs.dev)) {
#endif /* KERNEL_VERSION(3, 11, 0) */

#ifdef DEBUG_SYSFS
		pr_info(DRV_NAME ": %s(%p=\"%s\")\n", __func__, dev,
			dev->sysfs.dev->kobj.name);
#endif
	} else {
		pr_warn(DRV_NAME ": %s(): device_create() failure err=%ld\n",
			__func__, (long )IS_ERR(dev->sysfs.dev));

		dev->sysfs.dev = NULL;
	}
}

/*
 * Destroy sysfs device node
 */
void plin_sysfs_destroy_dev(struct plin_dev *dev)
{
	if (dev->sysfs.dev) {
#ifdef DEBUG_SYSFS_TRACE
		pr_info(DRV_NAME ": %s(%p=\"%s\")\n", __func__, dev,
			dev->sysfs.dev->kobj.name);
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(3, 11, 0)
		plin_sysfs_del_attrs(dev->sysfs.dev,
				     plin_sysfs_dev_stats_attrs);

		plin_sysfs_del_attrs(dev->sysfs.dev,
				     plin_sysfs_dev_common_attrs);
#endif
		__plin_sysfs_destroy_dev(dev->sysfs.dev);
		dev->sysfs.dev = NULL;
	}
}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(6, 4, 0)
#define PLIN_ATTR_RO_CONST	const struct
#else
#define PLIN_ATTR_RO_CONST	struct
#endif

static ssize_t plin_show_version(PLIN_ATTR_RO_CONST class *cls,
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,34)
				 PLIN_ATTR_RO_CONST class_attribute *attr,
#endif
				 char *buf)
{
	return sprintf(buf, "%s\n", DRV_VER_STR);
}

static struct class_attribute plin_attr = {
	.attr = {
		.name = "version",
		.mode = S_IRUGO,
	},
	.show = plin_show_version,
};

/*
 * int plin_sysfs_init(void)
 */
int plin_sysfs_init(void)
{
	int err;

	/* create /sys/class/plin */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(6, 4, 0)
	plin.class = class_create(DRV_NAME);
	if (IS_ERR(plin.class)) {
		err = PTR_ERR(plin.class);
		plin.class = NULL;
		pr_warn(DRV_NAME
			": %s(): class_create() failed (err %d)\n",
			__func__, err);

		return err;
	}
#else
	plin.class = class_create(THIS_MODULE, DRV_NAME);
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,34)
	sysfs_attr_init(&plin_attr.attr);
#endif

	/* add attribute nodes to the class */
	err = class_create_file(plin.class, &plin_attr);
	if (err) {
		pr_warn(DRV_NAME
			": %s(): class_create_file() failed (err %d)\n",
			__func__, err);

		return err;
	}

	plin_foreach_intf(plin_sysfs_do_create_intf, NULL);

	return 0;
}

/*
 * void plin_sysfs_exit(void)
 */
void plin_sysfs_exit(void)
{
	/* remove class nodes */
	class_remove_file(plin.class, &plin_attr);

	/* and destroy the class itself */
	class_destroy(plin.class);

	plin.class = NULL;
}

#else /* PLIN_SYSFS_API */

void plin_sysfs_create_intf(struct plin_intf *intf)
{
	intf->sysfs_dev = NULL;
}

void plin_sysfs_destroy_intf(struct plin_intf *intf)
{
}

void plin_sysfs_create_dev_ex(struct plin_dev *dev, struct device *parent)
{
	dev->sysfs.dev = NULL;
}

void plin_sysfs_destroy_dev(struct plin_dev *dev)
{
}

int plin_sysfs_init(void)
{
	return 0;
}

void plin_sysfs_exit(void)
{
}

#endif /* PLIN_SYSFS_API */
