/* SPDX-License-Identifier: GPL-2.0-only */
/*
 * Copyright (C) 2025 Stephane Grosjean <stephane.grosjean@hms-networks.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the version 2 of the GNU General Public License
 * as published by the Free Software Foundation
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 */
#include "plin_main.h"
#include "plin_usb.h"

#include <linux/usb.h>
#include <linux/slab.h>

#if defined(DEBUG)
#define DEBUG_USB
#elif defined(DEBUG_TRACE)
#define DEBUG_USB_TRACE
#endif

#ifdef DEBUG_USB
#define DEBUG_USB_RX
#define DEBUG_USB_TX
#define DEBUG_USB_TRACE
#endif

/* If defined, USB interface command buffer access is mutualy exclusive.
 * If not defined, several tasks could send commands at the same time if nothing
 * is done elsewhere to prevent it.
 * The chrdev API defines plin_chrdev_lock_ioctl()/plin_chrdev_unlock_ioctl()
 * which goal is to synchronize this access through user tasks. But,
 * WARNING! this interface is not used by sysfs/Udev.
 */
//#define PLIN_USB_LOCK_CMD_BUFFER_WITH_SPINLOCK

/* If defined, then id_user_num/id_user_str are cached by the driver
 * If not defined, then a get access to these values will be forwarded to the
 * device, which can create deadlock in the kernel if access is made from Udev
 * rule!
 * SHOULD be defined, especially if PLIN_USB_LOCK_CMD_BUFFER_WITH_SPINLOCK is
 * defined!
 */
#ifdef PLIN_USB_LOCK_CMD_BUFFER_WITH_SPINLOCK
#define PLIN_USB_USES_CACHE_ID_STR
#else
#define PLIN_USB_USES_CACHE_ID_STR
#endif

/* If defined, each URB buffer is allocated.
 * If not defined, a big buffer is used and cut into several buffers.
 * SHOULD BE defined to run with Raspberry PI!
 */
#define PLIN_USB_URB_USES_OWN_BUFFER

#define DRV_NAME_USB			DRV_NAME "_usb"

#define PEAK_USB_VENDOR_ID		0x0c72	/* The USB device vendor ID */
#define PCAN_USBPRO_PRODID		0x000d
#define PCAN_USBPRO_FD_PRODID		0x0011
#define PLIN_USB_PRODID			0x0016

#define PLIN_BUFLEN_LS		64	/* low/full-speed (usb 1.1) */
#define PLIN_BUFLEN_HS		512	/* high-speed (usb 2.0) */

#define PLIN_CMD_LEN			64
#define PLIN_TXMSG_LEN			64

#define PLIN_RSP_LEN			PLIN_BUFLEN_HS

#define PLIN_CMD_TIMEOUT_DEF		1000
#define PLIN_USB_AUTOBAUD_TIMEOUT_DEF	4000

#define PLIN_RXMSG_ITEM_LEN		32

/*
 * Vendor Request control msgs aren't supported by PLIN-USB.
 */
#define USB_VENDOR_REQ_INFO		0
#define USB_VENDOR_REQ_FCT		2

/* Vendor Request value for XXX_INFO */
#define USB_VENDOR_INFO_FW		1

/* Vendor Request value for XXX_FCT */
#define USB_VENDOR_FCT_DRVLD		5 /* driver is loaded */

/* Vendor Request commands buffer size */
#define USB_VENDOR_REQ_BUF_LEN		64

#ifdef PCAN_USBPRO_PRODID
#if LINUX_VERSION_CODE < KERNEL_VERSION(5, 12, 0)
MODULE_SUPPORTED_DEVICE("PEAK PCAN-USB Pro");
#endif

#define USB_VENDOR_INFO_BL		0

/* USB_VENDOR_INFO_BL vendor request record type (PCAN-USB Pro only) */
struct usb_pro_bl_info {
	u32	ctrl_type;
	u8	ver[4];	/* [0] -> main [1]-> sub [2]-> debug */
	u8	day;
	u8	month;
	u8	year;
	u8	dummy;
	u32	serial_num_high;
	u32	serial_num_low;
	u32	hw_type;
	u32	hw_rev;
}  __attribute__ ((packed));

/* USB_VENDOR_INFO_FW vendor request record type (PCAN-USB Pro only) */
struct usb_pro_fw_info {
	u32	ctrl_type;
	u8	ver[4];	/* [0] -> main [1]-> sub [2]-> debug */
	u8	day;
	u8	month;
	u8	year;
	u8	dummy;
	u32	fw_type;
}  __attribute__ ((packed));

#endif  /* PCAN_USBPRO_PRODID */

#ifdef PCAN_USBPRO_FD_PRODID

#define WORKAROUND_PCAN_USBPRO_FD_TIMESYNC_ISSUE

#if LINUX_VERSION_CODE < KERNEL_VERSION(5, 12, 0)
MODULE_SUPPORTED_DEVICE("PEAK PCAN-USB Pro FD");
#endif

struct usb_fd_fw_info_v1 {
	__le16	size_of;	/* sizeof this */
	__le16	type;		/* type of this structure */
	u8	hw_type;	/* Type of hardware (HW_TYPE_xxx) */
	u8	bl_ver[3];	/* Bootloader version */
	u8	hw_ver;		/* Hardware version (PCB) */
	u8	fw_ver[3];	/* Firmware version */
	__le32	dev_id[2];	/* "device id" per CAN */
	__le32	ser_no;		/* S/N */
	__le32	flags;		/* special functions */
} __attribute__ ((packed));

struct usb_fd_fw_info {
	__le16	size_of;	/* sizeof this */
	__le16	type;		/* type of this structure */
	u8	hw_type;	/* Type of hardware (PCAN_USBFD_MCU_xxx) */
	u8	bl_ver[3];	/* Bootloader version */
	u8	hw_ver;		/* Hardware version (PCB) */
	u8	fw_ver[3];	/* Firmware version */
	__le32  dev_id[2];	/* "device id" per CAN */
	__le32  ser_no;		/* S/N */
	__le32  flags;		/* special functions */

	/* ATSAM MCU only */
	u8	cmd_out_ep;	/* ep for cmd */
	u8	cmd_in_ep;	/* ep for replies */
	u8	data_out_ep[2];	/* ep for CANx TX */
	u8	data_in_ep;	/* ep for CAN RX */
	u8	dummy[3];
} __attribute__ ((packed));

#endif	/* PCAN_USBPRO_FD_PRODID */

#ifdef PLIN_USB_PRODID
#if LINUX_VERSION_CODE < KERNEL_VERSION(5, 12, 0)
MODULE_SUPPORTED_DEVICE("PEAK PLIN-USB");
#endif

#endif	/* PLIN_USB_PRODID */

/* endpoints usage */
enum {
	EP_CMD_OUT,
	EP_RSP_IN,
	EP_DATA_TX,
	EP_DATA_TX_1 = EP_DATA_TX,
	EP_DATA_RX,
	EP_DATA_TX_2,
	EP_UNUSED,
	EP_COUNT
};

struct plin_usb_intf_desc {
	struct plin_intf_desc desc;

	u8 ep_list[EP_COUNT];		/* EP used by the LIN USB interface */
};

#define to_intf_desc(i)			(&(i)->desc)
#define to_usb_intf_desc(d)		((struct plin_usb_intf_desc *)(d))

#define intf_ep_list(i)	to_usb_intf_desc(intf_desc(to_intf(i)))->ep_list

/* list of supported PEAK-System LIN USB interfaces */
static struct plin_usb_intf_desc plin_usb_intfs_list[] = {
#ifdef PCAN_USBPRO_PRODID
	{
		.desc = PLIN_INTF("PCAN-USB Pro", PCAN_USBPRO_PRODID, 2),
		.ep_list = { 0x04, 0x84, 0x05, 0x85, 0x06, 0x86 },
	},
#endif
#ifdef PCAN_USBPRO_FD_PRODID
	{
		.desc = PLIN_INTF("PCAN-USB Pro FD", PCAN_USBPRO_FD_PRODID, 2),
		.ep_list = { 0x04, 0x84, 0x05, 0x85, 0x06, 0x86 },
	},
#endif
#ifdef PLIN_USB_PRODID
	{
		.desc = PLIN_INTF("PLIN-USB", PLIN_USB_PRODID, 1),
		.ep_list = { 0x01, 0x81, 0x02, 0x82, },
	},
#endif
};

/* supported device ids. */
static const struct usb_device_id plin_usb_devids[] = {
#ifdef PCAN_USBPRO_PRODID
	{USB_DEVICE(PEAK_USB_VENDOR_ID, PCAN_USBPRO_PRODID)},
#endif
#ifdef PCAN_USBPRO_FD_PRODID
	{USB_DEVICE(PEAK_USB_VENDOR_ID, PCAN_USBPRO_FD_PRODID)},
#endif
#ifdef PLIN_USB_PRODID
	{USB_DEVICE(PEAK_USB_VENDOR_ID, PLIN_USB_PRODID)},
#endif
	{}
};

MODULE_DEVICE_TABLE(usb, plin_usb_devids);

#define PLIN_USB_IDENTIFY_AT_INIT_DEF	1

static uint usb_identify = PLIN_USB_IDENTIFY_AT_INIT_DEF;
module_param(usb_identify, uint, 0644);
MODULE_PARM_DESC(usb_identify,
		 " identify USB devices when driver starts (def="
		 __stringify(PLIN_USB_IDENTIFY_AT_INIT_DEF) ")");

#define PLIN_USB_HWRESET_AT_INIT_DEF	1

static uint usb_hwreset = PLIN_USB_HWRESET_AT_INIT_DEF;
module_param(usb_hwreset, uint, 0644);
MODULE_PARM_DESC(usb_hwreset,
		 " hw reset USB devices when driver starts (def="
		 __stringify(PLIN_USB_IDENTIFY_AT_INIT_DEF) ")");

#define PLIN_USB_TX_URBS_DEF		4

static uint usb_txurbsmax = PLIN_USB_TX_URBS_DEF;
module_param(usb_txurbsmax, uint, 0644);
MODULE_PARM_DESC(usb_txurbsmax,
		 " number of allocated URBs for Tx path per device (def="
		 __stringify(PLIN_USB_TX_URBS_DEF) ")");

#define PLIN_USB_RX_URBS_MIN		2
#define PLIN_USB_RX_URBS_MAX		4
#define PLIN_USB_RX_URBS_DEF		2

static uint usb_rxurbsmax = PLIN_USB_RX_URBS_DEF;
module_param(usb_rxurbsmax, uint, 0644);
MODULE_PARM_DESC(usb_rxurbsmax,
		 " number of allocated URBs for Rx path per interface (def="
		 __stringify(PLIN_USB_RX_URBS_DEF) ")");

#define PLIN_USB_RX_URBLEN_MIN		512
#define PLIN_USB_RX_URBLEN_MAX		4096
#define PLIN_USB_RX_URBLEN_DEF		PLIN_USB_RX_URBLEN_MAX

static uint usb_rxurblen = PLIN_USB_RX_URBLEN_DEF;
module_param(usb_rxurblen, uint, 0644);
MODULE_PARM_DESC(usb_rxurblen,
		 " length of each allocated URB for Rx path per interface (def="
		 __stringify(PLIN_USB_RX_URBLEN_DEF) ")");

static int unloading_driver = 0;

#ifdef CONFIG_UBSAN_BOUNDS_STRICT
#define PLIN_USB_RX_BUFFER_MAX_SIZE
#define PLIN_USB_TX_BUFFER_MAX_SIZE
#else
#define PLIN_USB_RX_BUFFER_MAX_SIZE		0
#define PLIN_USB_TX_BUFFER_MAX_SIZE		0
#endif

/* LIN USB interface private struct */
struct plin_usb_intf {
	struct plin_intf lin_intf;
	struct usb_interface *usb_intf;
	int cmd_timeout;
	int buflen;
	int bufidx_next;
	u8 cmd_buffer[PLIN_CMD_LEN];
	u8 rsp_buffer[PLIN_RSP_LEN];

#ifdef PLIN_USB_LOCK_CMD_BUFFER_WITH_SPINLOCK
	spinlock_t cmd_lock;
#endif
	struct usb_anchor rx_free_urbs;
	struct usb_anchor rx_submitted_urbs;
	atomic_t rx_submitted;
	int rx_submitted_before_reset;

#ifdef PLIN_USB_URB_USES_OWN_BUFFER
	u8 *rx_buffer[PLIN_USB_RX_BUFFER_MAX_SIZE];
#else
	u8 rx_buffer[PLIN_USB_RX_BUFFER_MAX_SIZE] __attribute__((aligned(8)));
#endif
};

#define to_intf(u)		(&(u)->lin_intf)
#define to_usb_intf(i)		((struct plin_usb_intf *)(i))

/* LIN USB device private struct */
struct plin_usb_dev {
	struct plin_dev	lin_dev;
	unsigned int ep_data_out;

	u8 cmd_buffer[PLIN_CMD_LEN];
	u8 rsp_buffer[PLIN_RSP_LEN];

#ifdef PLIN_USB_USES_CACHE_ID_STR
	struct plin_usb_ident_str ident_str_cache;
#endif
	struct usb_anchor tx_free_urbs;
	struct usb_anchor tx_submitted_urbs;

#ifdef PLIN_USB_URB_USES_OWN_BUFFER
	u8 *tx_buffer[PLIN_USB_TX_BUFFER_MAX_SIZE];
#else
	u8 tx_buffer[PLIN_USB_TX_BUFFER_MAX_SIZE] __attribute__((aligned(8)));
#endif
};

#define to_dev(u)		(&(u)->lin_dev)
#define to_usb_dev(d)		((struct plin_usb_dev *)(d))

static const char *plin_usb_speed_name(enum usb_device_speed usb_speed)
{
	switch (usb_speed) {
	case USB_SPEED_LOW:
		return "low";
	case USB_SPEED_FULL:
		return "full";		/* 1.1 */
	case USB_SPEED_HIGH:
		return "high";		/* 2.0 */
#ifdef USB_SPEED_SUPER
	case USB_SPEED_SUPER:
		return "super";		/* 3.0 */
#ifdef USB_SPEED_SUPER_PLUS
	case USB_SPEED_SUPER_PLUS:
		return "super+";	/* 3.1 */
#endif
#endif
	case USB_SPEED_UNKNOWN:
	default:
		break;
	}

	return "unknown";
}

/*
 * interface level
 */
static int plin_usb_send_req(struct plin_usb_intf *intf, int req_id,
			     int req_value)
{
	struct usb_device *usb_dev = interface_to_usbdev(intf->usb_intf);
	u8 req_type = USB_TYPE_VENDOR | USB_RECIP_OTHER;
	const int req_size = USB_VENDOR_REQ_BUF_LEN;
	unsigned int p;
	int err;

#ifdef DEBUG_USB_TRACE
	pr_info(DRV_NAME
		": %s(req_id=%d, req_value=%d, req_size=%d)\n",
		__func__, req_id, req_value, req_size);
#endif

#if 0
	/* TODO: usb device unregistered? */
	if (!(dev->state & PCAN_USB_STATE_CONNECTED))
		return 0;
#endif

	switch (req_id) {
	case USB_VENDOR_REQ_FCT:
		p = usb_sndctrlpipe(usb_dev, 0);
		break;

	case USB_VENDOR_REQ_INFO:

		/* fall through */
		fallthrough;
	default:
		p = usb_rcvctrlpipe(usb_dev, 0);
		req_type |= USB_DIR_IN;
		memset(intf->cmd_buffer, '\0', req_size);
		break;
	}

	err = usb_control_msg(usb_dev, p, req_id, req_type, req_value, 0,
			      intf->cmd_buffer, req_size,
			      USB_CTRL_GET_TIMEOUT);
	if (err < 0)
		dev_err(&intf->usb_intf->dev,
			"unable to request usb[type=%d value=%d] err=%d\n",
			req_id, req_value, err);

	return err;
}

/*
 * int plin_usb_drv_loaded(struct plin_usb_intf *intf, int loaded)
 *
 * Tell the USB LIN interface that a driver is loaded/removed.
 */
static int plin_usb_drv_loaded(struct plin_usb_intf *intf, int loaded)
{
	struct usb_device *usb_dev = interface_to_usbdev(intf->usb_intf);
	int err;

	switch (le16_to_cpu(usb_dev->descriptor.idProduct)) {
#ifdef PCAN_USBPRO_PRODID
	case PCAN_USBPRO_PRODID:
		break;
#endif
#ifdef PCAN_USBPRO_FD_PRODID
	case PCAN_USBPRO_FD_PRODID:
		break;
#endif
	default:
		/* Vendor Request not supported by this device */
		return 0;
	}

#ifdef DEBUG_USB_TRACE
	pr_info(DRV_NAME ": %s(loaded=%u)\n", __func__, loaded);
#endif

	intf->cmd_buffer[0] = 1;	/* 0=CAN 1=LIN */
	intf->cmd_buffer[1] = !!loaded;

	err = plin_usb_send_req(intf, USB_VENDOR_REQ_FCT, USB_VENDOR_FCT_DRVLD);

	return err >= 0 ? 0 : err;
}

/*
 * int __plin_usb_send_cmd(struct plin_usb_intf *intf, int l, void *cmd,
 * 			  unsigned int ep_cmd_out)
 *
 * Low-level function to send a LIN command to the correspondig EP:
 *
 * - cmd buffer must be DMA -able (stack not allowed).
 * - l SHOULD be <= 64
 */
static int __plin_usb_send_cmd(struct plin_usb_intf *intf, int l, void *cmd,
			      unsigned int ep_cmd_out)
{
	struct usb_device *usb_dev = interface_to_usbdev(intf->usb_intf);
	int err, try = 0;
	u8 *p = cmd;

#ifdef DEBUG_USB_TRACE
	/*
	 * 1	USB_SPEED_LOW	usb 1.1
	 * 2	USB_SPEED_FULL	usb 1.1		packet size=64
	 * 3	USB_SPEED_HIGH	usb 2.0		packet size=512
	 */
	pr_info(DRV_NAME ": %s(if=%d, ep=%xh l=%d): speed=%d\n",
		__func__, intf->usb_intf->cur_altsetting->desc.bInterfaceNumber,
		ep_cmd_out, l, usb_dev->speed);
#endif

	/* "The device driver should send a multiple of 64 bytes even when the
	 * command needs less data space."
	 */
	l = (((l - 1) / 64) + 1) * 64;

#ifdef DEBUG_USB_TRACE
	dump_mem("sending cmd", cmd, l);
#endif

	do {
		int actual_len;

		err = usb_bulk_msg(usb_dev,
				   usb_sndbulkpipe(usb_dev, ep_cmd_out),
				   p, l, &actual_len, intf->cmd_timeout);

		switch (err) {

		case 0:
			if (actual_len > 0)
				break;

			/* fall through */
			fallthrough;

		case -ETIMEDOUT:
			pr_warn(DRV_NAME ": %s #%d: "
				"sending cmd %u l=%d failed err=%d %d/3\n",
				intf_name(to_intf(intf)),
				intf_idx(to_intf(intf)),
				p[0], l, err, try+1);

			if (++try < 3)
				continue;

			/* fall through */
			fallthrough;
		default:
			pr_err(DRV_NAME ": %s #%d: "
			       "sending cmd %u l=%d failed err=%d actual=%d\n",
			       intf_name(to_intf(intf)),
			       intf_idx(to_intf(intf)),
			       p[0], l, err, actual_len);

			return err;
		}

		p += actual_len;
		l -= actual_len;

#ifdef DEBUG_USB
		if (l > 0)
			pr_info(DRV_NAME
				": %s(ep=%xh): loop to write %u bytes\n",
				__func__, ep_cmd_out, l);
#endif
	} while (l > 0);

	return err;
}

static int plin_usb_send_cmd(struct plin_usb_intf *intf, int l, void *cmd)
{
	return __plin_usb_send_cmd(intf, l, cmd,
				   intf_ep_list(intf)[EP_CMD_OUT]);
}

static inline void plin_usb_lock_cmd(struct plin_usb_intf *intf)
{
#ifdef PLIN_USB_LOCK_CMD_BUFFER_WITH_SPINLOCK
	spin_lock(&intf->cmd_lock);
#endif
}

static inline void plin_usb_unlock_cmd(struct plin_usb_intf *intf)
{
#ifdef PLIN_USB_LOCK_CMD_BUFFER_WITH_SPINLOCK
	spin_unlock(&intf->cmd_lock);
#endif
}

/*
 * int plin_usb_read_rsp(struct plin_usb_intf *intf, int l, void *rsp)
 *
 * Low-level function to Wait for a response to a command sent:
 *
 * - rsp buffer must be DMA -able (stack not allowed).
 * - l MUST be >= 64
 */
static int plin_usb_read_rsp(struct plin_usb_intf *intf, int l, void *rsp)
{
	struct usb_device *usb_dev = interface_to_usbdev(intf->usb_intf);
	unsigned int ep_rsp_in = intf_ep_list(intf)[EP_RSP_IN];
	int err, try = 0;
	u8 *p = rsp;

#ifdef DEBUG_USB_TRACE
	pr_info(DRV_NAME ": %s(ep=%xh l=%d): speed=%d\n",
		__func__, ep_rsp_in, l, usb_dev->speed);
#endif

	/* TODO:
	 * submitting a 512 bytes buffer when not in HIGH-SPEED?
	 */
	do {
		int actual_len;

		err = usb_bulk_msg(usb_dev,
				   usb_rcvbulkpipe(usb_dev, ep_rsp_in),
				   p, l, &actual_len,
				   intf->cmd_timeout);

		switch (err) {

		case 0:
			if (actual_len > 0)
				break;

			/* fall through */
			fallthrough;

		case -ETIMEDOUT:
			if (++try < 3)
				continue;

			/* fall through */
			fallthrough;
		default:
			pr_err(DRV_NAME ": %s: reading rsp failed (err %d)\n",
			       intf_name(to_intf(intf)), err);

			break;
		}

		l = actual_len;
		break;

	} while (1);

#ifdef DEBUG_USB_TRACE
	dump_mem("received rsp", rsp, l);
#endif

	return err;
}

static struct plin_usb_cmd *plin_usb_init_cmd_unlock(struct plin_usb_intf *intf,
						     u8 cmd)
{
	intf->cmd_buffer[0] = cmd;

	return (struct plin_usb_cmd *)intf->cmd_buffer;
}

static struct plin_usb_cmd *plin_usb_init_cmd_lock(struct plin_usb_intf *intf,
						   u8 cmd)
{
	plin_usb_lock_cmd(intf);

	return plin_usb_init_cmd_unlock(intf, cmd);
}

/*
 * int plin_usb_get_fw_ver(struct plin_usb_intf *intf)
 */
static int plin_usb_get_fw_ver(struct plin_usb_intf *intf)
{
	struct plin_usb_cmd *cmd = plin_usb_init_cmd_lock(intf,
						     PLIN_USB_CMD_GET_FW_VER);
	struct plin_usb_cmd *rsp;
	int err;

#ifdef DEBUG_USB_TRACE
	pr_info(DRV_NAME ": %s(%s #%d)\n",
		__func__, intf_name(to_intf(intf)), intf_idx(to_intf(intf)));
#endif

	/* send the command */
	err = plin_usb_send_cmd(intf, PLIN_CMD_LEN, cmd);
	if (err) {
		pr_err(DRV_NAME
		       ": %s #%d: failed to request FW info (err %d)\n",
		       intf_name(to_intf(intf)), intf_idx(to_intf(intf)), err);
		goto lbl_unlock;
	}

	/* get the response (use the same buffer) */
	err = plin_usb_read_rsp(intf, PLIN_RSP_LEN, intf->rsp_buffer);
	if (err) {
		pr_err(DRV_NAME
		       ": %s #%d: failed to get FW info (err %d)\n",
		       intf_name(to_intf(intf)), intf_idx(to_intf(intf)), err);
		goto lbl_unlock;
	}

	rsp = (struct plin_usb_cmd *)intf->rsp_buffer;
	to_intf(intf)->fw_ver[0] = rsp->fw_ver.major;
	to_intf(intf)->fw_ver[1] = rsp->fw_ver.minor;
	to_intf(intf)->fw_ver[2] = rsp->fw_ver.sub;

lbl_unlock:

	/* leave critical section */
	plin_usb_unlock_cmd(intf);

	return err;
}


/*
 * static int plin_usb_submit_rx_urb(struct urb *urb)
 */
static int plin_usb_submit_rx_urb(struct urb *urb)
{
	struct plin_usb_intf *intf = urb->context;
	int err;

	usb_anchor_urb(urb, &intf->rx_submitted_urbs);

	err = usb_submit_urb(urb, GFP_ATOMIC);
	if (err) {
		usb_unanchor_urb(urb);
		pr_err(DRV_NAME
		       ": %s: failed to submit another Rx urb (err %d)\n",
		       intf_name(to_intf(intf)), err);
	} else {
#ifdef DEBUG_USB_RX
		pr_info(DRV_NAME ": urb@%p anchored to submitted list\n", urb);
#endif
		atomic_inc(&intf->rx_submitted);
	}

	return err;
}

/*
 * static int plin_usb_submit_rx_urbs(struct plin_usb_intf *intf)
 *
 * Take all urbs from rx_free list to submit them.
 * Note that any urb that can't be submitted is killed.
 */
static int plin_usb_submit_rx_urbs(struct plin_usb_intf *intf)
{
	int submitted = 0;

	while (1) {
		struct urb *urb = usb_get_from_anchor(&intf->rx_free_urbs);
		if (!urb)
			break;

		if (plin_usb_submit_rx_urb(urb))
			usb_kill_urb(urb);
		else
			submitted++;
	}

	return submitted;
}

/*
 * static void plin_usb_kill_rx_urbs(struct plin_usb_intf *intf)
 *
 * Kill all submitted rx urbs. These MUST be moved to the rx_free list.
 */
static void plin_usb_kill_rx_urbs(struct plin_usb_intf *intf)
{
	usb_kill_anchored_urbs(&intf->rx_submitted_urbs);
}

/*
 * device level
 */
static inline int plin_usb_send_cmd_dev(struct plin_dev *dev)
{
	return plin_usb_send_cmd(to_usb_intf(dev_intf(dev)), PLIN_CMD_LEN,
				 to_usb_dev(dev)->cmd_buffer);
}

static inline int plin_usb_read_rsp_dev(struct plin_dev *dev)
{
	return plin_usb_read_rsp(to_usb_intf(dev_intf(dev)), PLIN_RSP_LEN,
				 to_usb_dev(dev)->rsp_buffer);
}

static inline void plin_usb_lock_cmd_dev(struct plin_dev *dev)
{
	plin_usb_lock_cmd(to_usb_intf(dev_intf(dev)));
}

static inline int plin_usb_unlock_cmd_dev(struct plin_dev *dev, int err)
{
	plin_usb_unlock_cmd(to_usb_intf(dev_intf(dev)));

	return err;
}

static struct plin_usb_cmd *plin_usb_init_cmd_dev_nolock(struct plin_dev *_dev,
							 u8 cmd)
{
	struct plin_usb_dev *dev = to_usb_dev(_dev);
	struct plin_usb_cmd *cmd_ptr = (struct plin_usb_cmd *)dev->cmd_buffer;

	cmd_ptr->id = cmd;
	cmd_ptr->bus = dev_idx(_dev);

	return cmd_ptr;
}

static struct plin_usb_cmd *plin_usb_init_cmd_dev_lock(struct plin_dev *dev,
						       u8 cmd)
{
	plin_usb_lock_cmd_dev(dev);

	return plin_usb_init_cmd_dev_nolock(dev, cmd);
}

/*
 * int _plin_usb_init_hw(struct plin_dev *dev, int mode, u16 baudrate)
 */
static int _plin_usb_init_hw(struct plin_dev *dev, int mode, u16 baudrate)
{
	struct plin_usb_cmd *cmd = plin_usb_init_cmd_dev_lock(dev,
						     PLIN_USB_CMD_INIT_HW);
	int err;

#ifdef DEBUG_USB_TRACE
	pr_info(DRV_NAME ": %s(%s, mode=%u, bitrate=%u)\n",
		__func__, dev_full_name(dev), mode, baudrate);
#endif

	cmd->init_hw.baudrate = cpu_to_le16(baudrate);
	cmd->init_hw.mode = (u8 )mode;

	err = plin_usb_send_cmd_dev(dev);
	if (err)
		pr_err(DRV_NAME
		       ": %s: failed to send init cmd (err %d)\n",
		       dev_full_name(dev), err);

	/* cache the settings */
	dev->baudrate = baudrate;
	dev->mode = cmd->init_hw.mode;

	/* leave critical section */
	return plin_usb_unlock_cmd_dev(dev, err);
}

/*
 * int _plin_usb_stop(struct plin_dev *dev)
 */
static int _plin_usb_stop(struct plin_dev *dev)
{
#ifdef DEBUG_USB_TRACE
	pr_info(DRV_NAME ": %s(%s)\n", __func__, dev_full_name(dev));
#endif

	/* "The value must be between 1000 and 20000 also when mode is set
	 * to none."
	 */
	return _plin_usb_init_hw(dev, PLIN_MODE_NONE, dev->baudrate);
}

/*
 * static void plin_usb_kill_tx_urbs(struct plin_usb_dev *dev)
 *
 * Kill all submitted tx urbs. These MUST be moved to the tx_free list.
 */
static void plin_usb_kill_tx_urbs(struct plin_usb_dev *dev)
{
	usb_kill_anchored_urbs(&dev->tx_submitted_urbs);
}

/*
 * static int plin_usb_post_process_init_hw(struct plin_dev *_dev, int mode)
 *
 * Do start/stop providing USB core with Rx/Tx urbs.
 */
static int plin_usb_post_process_init_hw(struct plin_dev *_dev, int mode)
{
	struct plin_usb_dev *dev = to_usb_dev(_dev);
	struct plin_usb_intf *intf = to_usb_intf(dev_intf(_dev));
	int err = 0;

#ifdef DEBUG_USB_TRACE
	pr_info(DRV_NAME ": %s(%s): mode=%d\n", __func__,
		dev_full_name(_dev), mode);
#endif

	_dev->flags &= ~PLIN_DEV_BUS_SLEEP;

	if (mode == PLIN_MODE_NONE) {
		/* abort pending Tx urbs (this cannot be done during
		 * usb_disconnect())
		 */
		plin_usb_kill_tx_urbs(dev);

	/* if submit all Rx free urbs */
	} else {

		/* provide USB core with Rx urbs extracted from the
		 * rx_free list
		 */
		plin_usb_submit_rx_urbs(intf);
		if (!atomic_read(&intf->rx_submitted))
			pr_err(DRV_NAME ": %s: unable to receive any data!\n",
			       dev_intf_name(_dev));
		else if (atomic_read(&intf->rx_submitted) != usb_rxurbsmax)
			pr_warn(DRV_NAME
				": %s: only %d rx urbs (instead of %d) "
				"submitted!\n",
				dev_intf_name(_dev),
				atomic_read(&intf->rx_submitted),
				usb_rxurbsmax);
	}

	return err;
}

/*
 * static int _plin_usb_send_cmd(struct plin_dev *dev, void *b, int l)
 */
static int _plin_usb_send_cmd(struct plin_dev *dev, void *b, int l)
{
	struct plin_usb_cmd *cmd = plin_usb_init_cmd_dev_lock(dev, 0);
	int err;

	if (l > PLIN_CMD_LEN)
		l = PLIN_CMD_LEN;

	memcpy(cmd, b, l);

	err = plin_usb_send_cmd_dev(dev);

	/* leave critical section */
	if (!plin_usb_unlock_cmd_dev(dev, err)) {

		/* post-process command to catch start/stop command,
		 * to start/stop providing USB core with Rx urbs
		 */
		switch (cmd->id) {

		case PLIN_USB_CMD_INIT_HW:
			plin_usb_post_process_init_hw(dev, cmd->init_hw.mode);
			break;
		}
	}

	return err;
}

/*
 * static int _plin_usb_read_rsp(struct plin_dev *dev, void *b, int l)
 *
 * Note: no internal lock mechanism is used for direct access!
 */
static int _plin_usb_read_rsp(struct plin_dev *dev, void *b, int l)
{
	int err = plin_usb_read_rsp_dev(dev);
	if (!err) {

		if (l > PLIN_RSP_LEN)
			l = PLIN_RSP_LEN;

		memcpy(b, to_usb_dev(dev)->rsp_buffer, l);
	}

	return err;
}

/*
 * int plin_usb_identify(struct plin_dev *dev)
 */
static int plin_usb_identify(struct plin_dev *dev)
{
	int err;

	plin_usb_init_cmd_dev_lock(dev, PLIN_USB_CMD_IDENTIFY_BUS);

#ifdef DEBUG_USB_TRACE
	pr_info(DRV_NAME ": %s(%s)\n", __func__, dev_full_name(dev));
#endif

	/* send the command */
	err = plin_usb_send_cmd_dev(dev);
	if (err)
		pr_err(DRV_NAME
		       ": %s: failed to send identify cmd (err %d)\n",
		       dev_full_name(dev), err);

	/* leave critical section */
	return plin_usb_unlock_cmd_dev(dev, err);
}

#if 0
/*
 * int plin_usb_get_id_str(struct plin_dev *_dev, char *buf, int len)
 */
static int plin_usb_get_id_str(struct plin_dev *dev, char *buf, int len)
{
	struct plin_usb_cmd *cmd = plin_usb_init_cmd_dev_lock(dev,
						PLIN_USB_CMD_GET_IDENT_STR);
	int err, l = sizeof(cmd->ident_str);

#ifdef DEBUG_USB_TRACE
	pr_info(DRV_NAME ": %s(%s)\n", __func__, dev_full_name(dev));
#endif

	/* send the command and read the response in dev->rsp_buffer */
	err = plin_usb_send_cmd_dev(dev);
	if (err) {
		pr_err(DRV_NAME
		       ": %s: failed to send cmd %u (err %d)\n",
		       dev_full_name(dev), cmd->id, err);

		return err;
	}

	/* be sure that response buffer is empty */
	memset(to_usb_dev(dev)->rsp_buffer, '\0', l);

	/* get the reply */
	err = plin_usb_read_rsp_dev(dev);
	if (err) {
		pr_err(DRV_NAME
		       ": %s: failed to get rsp to cmd %u (err %d)\n",
		       dev_full_name(dev), cmd->id, err);
	}

	if (l > len)
		l = len;

	memcpy(buf, to_usb_dev(dev)->rsp_buffer, l);

	/* leave critical section */
	return plin_usb_unlock_cmd_dev(dev, l);
}
#endif

/*
 * struct plin_usb_ident_str *plin_usb_get_ident_str_lock(struct plin_dev *dev)
 */
static struct plin_usb_ident_str *
	plin_usb_get_ident_str_lock(struct plin_dev *dev)
{
	struct plin_usb_cmd *cmd = plin_usb_init_cmd_dev_lock(dev,
						PLIN_USB_CMD_GET_IDENT_STR);
	int err, l = sizeof(cmd->ident_str);

#ifdef DEBUG_USB_TRACE
	pr_info(DRV_NAME ": %s(%s)\n", __func__, dev_full_name(dev));
#endif

	/* send the command and read the response in dev->rsp_buffer */
	err = plin_usb_send_cmd_dev(dev);
	if (err) {
		pr_err(DRV_NAME
		       ": %s: failed to send cmd %u (err %d)\n",
		       dev_full_name(dev), cmd->id, err);

		/* leave critical section */
		plin_usb_unlock_cmd_dev(dev, err);

		return NULL;
	}

	/* be sure that response buffer is empty */
	memset(to_usb_dev(dev)->rsp_buffer, '\0', l);

	/* get the reply */
	err = plin_usb_read_rsp_dev(dev);
	if (err) {
		pr_err(DRV_NAME
		       ": %s: failed to get rsp to cmd %u (err %d)\n",
		       dev_full_name(dev), cmd->id, err);

		/* leave critical section */
		plin_usb_unlock_cmd_dev(dev, err);

		return NULL;
	}

	return &((struct plin_usb_cmd *)to_usb_dev(dev)->rsp_buffer)->ident_str;
}

#ifndef PLIN_USB_USES_CACHE_ID_STR
/*
 * int plin_usb_get_id_user_str(struct plin_dev *dev, char *buf, int len)
 */
static int plin_usb_get_id_user_str(struct plin_dev *dev, char *buf, int len)
{
	struct plin_usb_ident_str *ident_str;

#ifdef DEBUG_USB_TRACE
	pr_info(DRV_NAME ": %s(%s)\n", __func__, dev_full_name(dev));
#endif

	ident_str = plin_usb_get_ident_str_lock(dev);
	if (!ident_str)
		return -EIO;

	if (len > PLIN_USB_IDENT_USER_LEN)
		len = PLIN_USB_IDENT_USER_LEN;

	memcpy(buf, ident_str->user_str, len);

	/* leave critical section */
	return plin_usb_unlock_cmd_dev(dev, len);
}

/*
 * int plin_usb_get_id_user_num(struct plin_dev *dev, u32 *user_num)
 */
static int plin_usb_get_id_user_num(struct plin_dev *dev, u32 *user_num)
{
	struct plin_usb_ident_str *ident_str;

#ifdef DEBUG_USB_TRACE
	pr_info(DRV_NAME ": %s(%s)\n", __func__, dev_full_name(dev));
#endif

	ident_str = plin_usb_get_ident_str_lock(dev);
	if (!ident_str)
		return -EIO;

	if (user_num)
		*user_num = le32_to_cpu(ident_str->user_id);

	/* leave critical section */
	return plin_usb_unlock_cmd_dev(dev, 0);
}
#endif

/*
 * static int plin_usb_set_id_str_unlock(struct plin_dev *dev,
 * 				   struct plin_usb_ident_str *ident_str)
 */
static int plin_usb_set_id_str_unlock(struct plin_dev *dev,
				struct plin_usb_ident_str *ident_str)
{
	struct plin_usb_cmd *cmd = plin_usb_init_cmd_dev_nolock(dev,
						PLIN_USB_CMD_SET_IDENT_STR);
	int err;

#ifdef DEBUG_USB_TRACE
	pr_info(DRV_NAME ": %s(%s)\n", __func__, dev_full_name(dev));
#endif

	/* copy full ident string */
	cmd->ident_str = *ident_str;

	/* send the command */
	err = plin_usb_send_cmd_dev(dev);
	if (err) {
		pr_err(DRV_NAME
		       ": %s: failed to set id string (err %d)\n",
		       dev_full_name(dev), err);

#ifdef PLIN_USB_USES_CACHE_ID_STR
	} else {
		/* update the local cache with new value */
		to_usb_dev(dev)->ident_str_cache = *ident_str;
#endif
	}

	/* leave critical section */
	return plin_usb_unlock_cmd_dev(dev, err);
}

/*
 * static int plin_usb_set_id_user_str(struct plin_dev *dev, const char *buf,
 * 				       int len)
 */
static int plin_usb_set_id_user_str(struct plin_dev *dev, const char *buf,
				    int len)
{
	struct plin_usb_ident_str *ident_str;

#ifdef DEBUG_USB_TRACE
	pr_info(DRV_NAME ": %s(%s)\n", __func__, dev_full_name(dev));
#endif

	ident_str = plin_usb_get_ident_str_lock(dev);
	if (!ident_str)
		return -EIO;

	if (len > PLIN_USB_IDENT_USER_LEN)
		len = PLIN_USB_IDENT_USER_LEN;

	memcpy(ident_str->user_str, buf, len);

	return plin_usb_set_id_str_unlock(dev, ident_str);
}

/*
 * static int plin_usb_set_id_user_num(struct plin_dev *dev, u32 user_num)
 */
static int plin_usb_set_id_user_num(struct plin_dev *dev, u32 user_num)
{
	struct plin_usb_ident_str *ident_str;

#ifdef DEBUG_USB_TRACE
	pr_info(DRV_NAME ": %s(%s)\n", __func__, dev_full_name(dev));
#endif

	ident_str = plin_usb_get_ident_str_lock(dev);
	if (!ident_str)
		return -EIO;

	ident_str->user_id = cpu_to_le32(user_num);

	return plin_usb_set_id_str_unlock(dev, ident_str);
}

/*
 * int plin_usb_set_led_state(struct plin_dev *dev, u8 state)
 */
static int plin_usb_set_led_state(struct plin_dev *dev, u8 state)
{
	struct plin_usb_cmd *cmd = plin_usb_init_cmd_dev_lock(dev,
						PLIN_USB_CMD_LED_STATE);
	int err;

#ifdef DEBUG_USB_TRACE
	pr_info(DRV_NAME ": %s(%s)\n", __func__, dev_full_name(dev));
#endif

	cmd->led_state.on_off = !!state;

	/* send the command */
	err = plin_usb_send_cmd_dev(dev);
	if (err) {
		pr_err(DRV_NAME
		       ": %s: failed to set LED state (err %d)\n",
		       dev_full_name(dev), err);
	}

	/* leave critical section */
	return plin_usb_unlock_cmd_dev(dev, err);
}

/*
 * int plin_usb_hw_reset(struct plin_dev *dev)
 */
static int plin_usb_hw_reset(struct plin_dev *dev)
{
	int err;

	plin_usb_init_cmd_dev_lock(dev, PLIN_USB_CMD_RST_HW_CFG);

#ifdef DEBUG_USB_TRACE
	pr_info(DRV_NAME ": %s(%s)\n", __func__, dev_full_name(dev));
#endif

	/* send the command */
	err = plin_usb_send_cmd_dev(dev);
	if (err) {
		pr_err(DRV_NAME
		       ": %s: failed to send hw reset cmd (err %d)\n",
		       dev_full_name(dev), err);
	}

	/* leave critical section */
	return plin_usb_unlock_cmd_dev(dev, err);
}

/*
 * static int plin_usb_is_bus_ok(struct plin_dev *dev, void *arg)
 */
static int plin_usb_is_bus_ok(struct plin_dev *dev, void *arg)
{
	u8 *pb = arg;

	return (dev_idx(dev) == pb[2]);
}

#ifdef PLIN_USB_USES_CACHE_ID_STR
/*
 * int plin_usb_cache_id_str(struct plin_dev *dev)
 *
 * Cache full id_str for Udev access using sysfs (no lock allowed)
 */
static int plin_usb_cache_id_str(struct plin_dev *dev)
{
	struct plin_usb_ident_str *ident_str;

#ifdef DEBUG_USB_TRACE
	pr_info(DRV_NAME ": %s(%s)\n", __func__, dev_full_name(dev));
#endif

	ident_str = plin_usb_get_ident_str_lock(dev);
	if (!ident_str)
		return -EIO;

	to_usb_dev(dev)->ident_str_cache = *ident_str;

	/* leave critical section */
	return plin_usb_unlock_cmd_dev(dev, 0);
}

static int plin_usb_get_cached_id_user_str(struct plin_dev *dev,
					   char *buf, int len)
{
	if (len > PLIN_USB_IDENT_USER_LEN)
		len = PLIN_USB_IDENT_USER_LEN;

	memcpy(buf, to_usb_dev(dev)->ident_str_cache.user_str, len);

	return len;
}

static int plin_usb_get_cached_id_user_num(struct plin_dev *dev, u32 *user_num)
{
	if (user_num)
		*user_num =
			le32_to_cpu(to_usb_dev(dev)->ident_str_cache.user_id);

	return 0;
}
#endif

/*
 * static void plin_usb_rx_callback(struct urb *urb)
 *
 * Rx buffer handler. Buffer contains up to 8 fixed length messages.
 * Each message is defined with a type:
 *
 * 0	LIN frame
 * ----------------------------------------------------------------------------
 * 1	Bus goes to sleep
 * 2	Wake up frame received
 * 3	Autobaud timeout
 * 4	Autobaud succesfull (ctrlr MUST be in NONE mode and frames should exist
 *      on the bus (perioidcal frames written by the master, for example).
 *	d[0]..d[3] contains the number of microseconds to transfer 8 bits.
 *
 *	Example with a rx buffer that contains 01 packet that carries a msg
 *	type = 0x04. Data byte d[0] (=0x42) is the 25th byte of the message:
 *
 *      [20590.614803] plin: dumping received buffer (256 bytes):
 *      [20590.614805] plin: 00 26 01 00[04 00 00 00 00 00 00 00 01 00 00 00
 *      [20590.614806] plin: 00 00 00 00 4d 76 4c 66 00 00 00 00 42 03 00 00
 *      [20590.614807] plin: b0 96 00 00]-- -- -- -- -- -- -- -- -- -- -- --
 *
 *	0x342 = 834 µs -> 8 bits => 8.000.000 / 834 = 9592,32 bps
 * 5	USB overrun frame
 */
static void plin_usb_rx_callback(struct urb *urb)
{
	struct plin_usb_intf *intf = urb->context;
	struct plin_intf *_intf = to_intf(intf);
	u8 *pb, buf_type, buf_idx, msg_count;
	int i;

#ifdef DEBUG_USB_TRACE
	pr_info(DRV_NAME ": %s(%s): urb[status=%d actual_len=%d]\n", __func__,
		intf_name(_intf), urb->status, urb->actual_length);
#endif

	atomic_dec(&intf->rx_submitted);

	switch (urb->status) {

	case 0:
		if (!urb->actual_length)
			goto lbl_submit_rx;

		/* success */
		break;

	default:
		pr_err(DRV_NAME ": %s: Rx urb aborted (err %d)\n",
		       intf_name(_intf), urb->status);

		/* fall through */
		fallthrough;
	case -ENOENT:
	case -EILSEQ:
	case -EPROTO:
	case -ECONNRESET:
	case -ESHUTDOWN:

		/* disconnection (rmmod plin): no need to resubmit the urb:
		 * link it to the rx free urbs list so that it will be
		 * correctly released next */
		goto lbl_free_rx;
	}

	/* decode urb buffer (it contains data) */
	pb = urb->transfer_buffer;

	/* First byte MUST be 0! */
	buf_type = pb[0];
	if (buf_type) {
		if (printk_ratelimit())
			pr_err(DRV_NAME
			       ": WARNING: %s(%s): Loss-of-Sync in Rx buffer: "
			       "got wrong buf_type(%d)! LIN data lost!\n",
			       __func__, intf_name(_intf), buf_type);

		goto lbl_submit_rx;
	}

	buf_idx = pb[1];
	msg_count = pb[2];

#ifdef DEBUG_USB_RX
	pr_info(DRV_NAME ": urb@%p\n", urb);
	dump_mem("received buffer", urb->transfer_buffer, urb->actual_length);
#endif
	if (!msg_count) {
		pr_warn(DRV_NAME
			": WARNING: %s(%s): got null msg_count in Rx buffer "
			"(buf_type=%d buf_idx=%d)\n",
			__func__, intf_name(_intf),
			buf_type, buf_idx);
	}

	if (intf->bufidx_next < 0)
		intf->bufidx_next = buf_idx;

	else if (buf_idx != intf->bufidx_next) {
		if (printk_ratelimit())
			pr_warn(DRV_NAME
				": WARNING: %s(%s): got out-of-order Rx buffer "
				"(buf_type=%d buf_idx=%d, waiting for %d)\n",
				__func__, intf_name(_intf),
				buf_type, buf_idx, intf->bufidx_next);
	}

	/* split urb transfer buffer content into devices rx fifo */
	for (i = 0, pb += 4; i < msg_count; i++) {

		struct plin_dev *_dev;

		/* pb[2] contains the bus number */
		_dev = plin_intf_foreach_dev(_intf, plin_usb_is_bus_ok, pb);
		if (!_dev) {
			pr_warn(DRV_NAME
				": %s: ABNORMAL bus number %d (no dev found)\n",
				intf_name(_intf), pb[2]);

		} else {
			struct plin_msg msg;

			msg.type = (u16 )pb[0];
			msg.id = pb[4] & PLIN_FRM_ID_MASK;
			msg.len = pb[5];
			msg.dir = pb[6];
			msg.cs_type = pb[7];
			msg.flags = (u16 )le32_to_cpu(*(__le32 *)(pb + 12));
			msg.ts_us = le64_to_cpu(*(__le64 *)(pb + 16));
			memcpy(msg.data, pb + 24, PLIN_DAT_LEN);

#ifdef DEBUG_USB_TRACE
			pr_info(DRV_NAME
				": %llu %s < msg[type=%u id=%u len=%d]\n",
				msg.ts_us, dev_full_name(_dev),
				msg.type, msg.id, msg.len);
			dump_mem("received message", &msg, sizeof(msg));
#endif
			plin_rx_msg(_dev, &msg);
		}

		pb += PLIN_RXMSG_ITEM_LEN;
	}

	intf->bufidx_next++;
	intf->bufidx_next &= 0xff;

lbl_submit_rx:

	/* submit the urb once processed. */
	if (!plin_usb_submit_rx_urb(urb))
		return;

lbl_free_rx:

#ifdef DEBUG_USB_RX
	pr_info(DRV_NAME ": urb@%p anchored to free list\n", urb);
#endif
	/* this urb is not submitted again, then it's linked to the free list */
	usb_anchor_urb(urb, &intf->rx_free_urbs);
}

/*
 * static void plin_usb_tx_callback(struct urb *urb)
 */
static void plin_usb_tx_callback(struct urb *urb)
{
	struct plin_usb_dev *dev = urb->context;
	struct plin_dev *_dev = to_dev(dev);

#ifdef DEBUG_USB_TRACE
	pr_info(DRV_NAME ": %s(%s): usb[status=%d]\n", __func__,
		dev_full_name(_dev), urb->status);
#endif

	switch (urb->status) {

	case 0:
		break;

	default:
		pr_err(DRV_NAME ": %s: Tx urb aborted (err %d)\n",
		       dev_full_name(_dev), urb->status);

		/* fall through */
		fallthrough;

	case -EPROTO:
	case -ENOENT:
	case -ECONNRESET:
	case -ESHUTDOWN:
		break;
	}

	/* anchor that urb into the Tx free queue */
	usb_anchor_urb(urb, &dev->tx_free_urbs);

	/* tx_free_urbs no more empty */
	wake_up_interruptible(&_dev->tx_wait);
}

/*
 * static int plin_usb_write(struct plin_dev *dev, struct plin_msg *f)
 *
 * Note: this path is used only in master mode, when the scheduler is not
 *       running. The frame is sent independently from the frames table
 *       settings.
 */
static int plin_usb_write(struct plin_dev *_dev, struct plin_msg *msg)
{
	struct plin_usb_dev *dev = to_usb_dev(_dev);
	struct urb *urb;
	int err;
	u8 *pb;

#ifdef DEBUG_USB_TRACE
	pr_info(DRV_NAME ": %s(%s, id=%xh len=%d)\n", __func__,
		dev_full_name(_dev), msg->id, msg->len);
#endif

	/* take a urb from the tx_free queue and submit it */
	urb = usb_get_from_anchor(&dev->tx_free_urbs);
	if (!urb) {
		pr_warn(DRV_NAME
			": %s: no more free Tx urb to submit for now\n",
			dev_full_name(_dev));
		return -EAGAIN;
	}

	/* 4 dummy bytes + 4 unused bytes */
	pb = urb->transfer_buffer + 4 + 4;

	/* copy frame content */
	*pb++ = plin_frm_get_pid(msg->id);
	*pb++ = msg->len;
	*pb++ = msg->dir;
	*pb++ = msg->cs_type;
	if (msg->cs_type == PLIN_FRM_CST_CUSTOM) {
		*pb = 0;	/* TODO */
	}
	pb += 4;
	memcpy(pb, msg->data, PLIN_DAT_LEN);

#ifdef DEBUG_USB_TX
#ifdef DEBUG_MINOR
	if (dev_minor(_dev) == DEBUG_MINOR)
#endif
	dump_mem("transmitted frame", urb->transfer_buffer,
		 urb->transfer_buffer_length);
#endif

	/* submit the Tx urb now */
	usb_anchor_urb(urb, &dev->tx_submitted_urbs);
	err = usb_submit_urb(urb, GFP_KERNEL);
	if (!err) {
		_dev->stats.tx_msgs++;
		_dev->stats.tx_frames++;
		_dev->stats.tx_bytes += msg->len;
	} else {

		pr_err(DRV_NAME
		       ": %s: failed to submit Tx urb (err %d)\n",
	      	       dev_full_name(_dev), err);

		/* relink that urb to the tx_free queue */
		usb_unanchor_urb(urb);
		usb_anchor_urb(urb, &dev->tx_free_urbs);
	}

	return err;
}

/*
 * static int plin_usb_write_busy(struct plin_dev *_dev)
 */
static int plin_usb_write_busy(struct plin_dev *_dev)
{
	struct plin_usb_dev *dev = to_usb_dev(_dev);

#ifdef DEBUG_USB_TRACE
	pr_info(DRV_NAME ": %s(%s)\n", __func__, dev_full_name(_dev));
#endif

	return list_empty(&dev->tx_free_urbs.urb_list);
}

/*
 * static void plin_usb_cleanup(struct plin_dev *_dev)
 *
 * Last chance to play with USB transfers and URBs.
 * Called before removing USB driver *ONLY*!
 *
 * WARNING: This function SHOULD NOT be called in any other context than
 * closing *ALL* the devices of an interface.
 */
static void plin_usb_cleanup(struct plin_dev *dev)
{
#ifdef DEBUG_USB_TRACE
	pr_info(DRV_NAME ": %s(%s): is_last=%d\n",
		__func__, dev_full_name(dev),
		list_is_last(&dev->link, &dev_intf(dev)->dev_list));
#endif

	/* stop established session */
	_plin_usb_stop(dev);

	/* cleanup the session */
	plin_usb_post_process_init_hw(dev, PLIN_MODE_NONE);

	/* abort pending Rx urbs when last device is closed */
	if (list_is_last(&dev->link, &dev_intf(dev)->dev_list)) {
		struct plin_usb_intf *intf = to_usb_intf(dev_intf(dev));

		/* kill any outsanding rx urbs
		 * Note: urbs are automatically linked to the rx_free list when
		 * aborted
		 */
		plin_usb_kill_rx_urbs(intf);
	}
}

/*
 * static void plin_usb_free_dev(struct plin_dev *_dev)
 *
 * USB device private destructor.
 * Called by plin_free_dev().
 */
static void plin_usb_free_dev(struct plin_dev *_dev)
{
	struct plin_usb_dev *dev = to_usb_dev(_dev);
#ifdef PLIN_USB_URB_USES_OWN_BUFFER
	int i;
#endif

#ifdef DEBUG_USB_TRACE
	pr_info(DRV_NAME ": %s(%s)\n", __func__, dev_full_name(_dev));
#endif

	/* usb_kill_anchored_urbs() can't be called when in disconnect() */
	usb_scuttle_anchored_urbs(&dev->tx_submitted_urbs);
	usb_scuttle_anchored_urbs(&dev->tx_free_urbs);

#ifdef PLIN_USB_URB_USES_OWN_BUFFER
	for (i = 0; i < usb_txurbsmax; i++)
		kfree(dev->tx_buffer[i]);
#endif
}

/*
 * static void plin_usb_free_intf(struct plin_intf *_intf)
 *
 * USB interface private destructor.
 * Called by plin_free_intf().
 */
static void plin_usb_free_intf(struct plin_intf *_intf)
{
	struct plin_usb_intf *intf = to_usb_intf(_intf);
#ifdef PLIN_USB_URB_USES_OWN_BUFFER
	int i;
#endif

#ifdef DEBUG_USB_TRACE
	pr_info(DRV_NAME ": %s(%s)\n", __func__, intf_name(_intf));
#endif

	/* rx_urbs have been killed but not released */
	usb_scuttle_anchored_urbs(&intf->rx_submitted_urbs);
	usb_scuttle_anchored_urbs(&intf->rx_free_urbs);

#ifdef PLIN_USB_URB_USES_OWN_BUFFER
	for (i = 0; i < usb_rxurbsmax; i++)
		kfree(intf->rx_buffer[i]);
#endif
}

/*
 * static int plin_usb_do_kill_tx_urbs(struct plin_dev *_dev, void *arg)
 */
static int plin_usb_do_kill_tx_urbs(struct plin_dev *_dev, void *arg)
{
	/* abort any submitted tx urbs */
	plin_usb_kill_tx_urbs(to_usb_dev(_dev));

	return 0;
}

/*
 * static int plin_usb_pre_reset(struct usb_interface *usb_intf)
 *
 * Called before usb_reset_device():
 *
 * see include/linux/usb.h:
 * "This routine must not return until the driver has no active
 *  URBs for the device, and no more URBs may be submitted until the
 *  post_reset method is called."
 *
 * Note: the pcan driver does call usb_reset_device() when removed.
 */
static int plin_usb_pre_reset(struct usb_interface *usb_intf)
{
	struct plin_usb_intf *intf = usb_get_intfdata(usb_intf);
	if (!intf)
		return 0;

#ifdef DEBUG_USB_TRACE
	pr_info(DRV_NAME ": %s(%s #%u)\n", __func__,
		intf_name(to_intf(intf)), intf_idx(to_intf(intf)));
#endif

	intf->rx_submitted_before_reset = atomic_read(&intf->rx_submitted);

	/* kill any outsanding rx/tx urbs
	 * Note: urb are automatically linked to the x_free list by their
	 * respective callback when aborted.
	 */
	plin_intf_foreach_dev(to_intf(intf), plin_usb_do_kill_tx_urbs, NULL);
	plin_usb_kill_rx_urbs(intf);

	return 0;
}

/*
 * static int plin_usb_post_reset(struct usb_interface *usb_intf)
 *
 * Called after usb_reset_device(). 
 *
 * Note: the pcan driver does call usb_reset_device() when removed.
 */
static int plin_usb_post_reset(struct usb_interface *usb_intf)
{
	struct plin_usb_intf *intf = usb_get_intfdata(usb_intf);
	if (!intf)
		return 0;

#ifdef DEBUG_USB_TRACE
	pr_info(DRV_NAME ": %s(%s #%u)\n", __func__,
		intf_name(to_intf(intf)), intf_idx(to_intf(intf)));
#endif

	/* submit all rx_free urbs (if any dev were opened) */
	if (intf->rx_submitted_before_reset)
		plin_usb_submit_rx_urbs(intf);

	return 0;
}

/*
 * static int plin_usb_suspend(struct usb_interface *usb_intf,
 * 			       pm_message_t message)
 */
static int plin_usb_suspend(struct usb_interface *usb_intf,
			    pm_message_t message)
{
#ifdef DEBUG_USB_TRACE
	struct plin_usb_intf *intf = usb_get_intfdata(usb_intf);

	pr_info(DRV_NAME ": %s(%s #%u)\n", __func__,
		intf_name(to_intf(intf)), intf_idx(to_intf(intf)));
#endif

	return 0;
}

/*
 * static int plin_usb_resume(struct usb_interface *usb_intf)
 */
static int plin_usb_resume(struct usb_interface *usb_intf)
{
#ifdef DEBUG_USB_TRACE
	struct plin_usb_intf *intf = usb_get_intfdata(usb_intf);

	pr_info(DRV_NAME ": %s(%s #%u)\n", __func__,
		intf_name(to_intf(intf)), intf_idx(to_intf(intf)));
#endif

	return 0;
}

/*
 * static int plin_usb_do_dev_info(struct plin_dev *_dev, void *arg)
 */
static int plin_usb_do_dev_info(struct plin_dev *_dev, void *arg)
{
	pr_info(DRV_NAME "%u: %s\n",
		dev_minor(_dev), dev_full_name(_dev));

	return 0;
}

#ifdef WORKAROUND_PCAN_USBPRO_FD_TIMESYNC_ISSUE
/* Up to FW 3.4.x, PCAN-USB Pro FD starts timers synchronization only on
 * reception of the UCAN_ENABLE_OPTION[UCAN_USB_OPTION_CALIBRATION] message
 * on EP#1.
 */
#define PCAN_USB_PRO_FD_CAN_CMDOUT_EP		1

struct __packed ucan_option {
	__le16	opcode_channel;
	__le16	mask;
	__le16	unused;
	__le16	ext_mask;
};

#define UCAN_CMD_SET_EN_OPTION			0x00b
#define UCAN_USB_OPTION_CALIBRATION		0x8000

static void pcan_usb_pro_fd_timesync_issue(struct plin_usb_intf *intf)
{
	struct ucan_option *cmd;
	int err;

	cmd = (struct ucan_option *)plin_usb_init_cmd_lock(intf, 0);

#ifdef DEBUG_USB_TRACE
	pr_info(DRV_NAME ": %s(%s)\n", __func__, intf_name(to_intf(intf)));
#endif

	cmd->opcode_channel = cpu_to_le16(UCAN_CMD_SET_EN_OPTION);
	cmd->mask = 0;
	cmd->unused = 0;
	cmd->ext_mask = cpu_to_le16(UCAN_USB_OPTION_CALIBRATION);

	/* add END-OF-COLLECTION special record */
	memset(cmd+1, 0xff, sizeof(*cmd));

	/* send the command */
	err = __plin_usb_send_cmd(intf, 2 * sizeof(*cmd), cmd,
				  PCAN_USB_PRO_FD_CAN_CMDOUT_EP);
	if (err) {
		pr_err(DRV_NAME
		       ": %s: failed to enable calibration messages (err %d)\n",
		       intf_name(to_intf(intf)), err);
	}

#ifdef DEBUG_USB
	pr_info(DRV_NAME ": LIN timer sync workaround activated");
#endif

	/* leave critical section */
	plin_usb_unlock_cmd(intf);
}
#endif

/*
 * static int plin_usb_probe(struct usb_interface *usb_intf,
 *			     const struct usb_device_id *id)
 *
 * Probe the USB LIN interface and create one LIN dev per controller.
 *
 * Note: PCAN-USB Pro (FD) LIN interface is interface #1 while
 *       PLIN-USB LIN interface is interface #0.
 */
static int plin_usb_probe(struct usb_interface *usb_intf,
			  const struct usb_device_id *id)
{
	struct usb_device *usb_dev = interface_to_usbdev(usb_intf);
	const u16 devid = le16_to_cpu(usb_dev->descriptor.idProduct);
	struct usb_host_interface *if_desc = usb_intf->cur_altsetting;
	struct plin_usb_intf_desc *intf_desc = NULL;
	struct plin_usb_intf *intf;
	struct plin_intf *_intf;
	struct plin_usb_dev *dev;
	struct plin_dev *_dev;
	struct urb *urb;
	int i, j, err, sizeof_intf, sizeof_dev;

#ifdef DEBUG_USB_TRACE
	unsigned int intf_num = if_desc->desc.bInterfaceNumber;

	pr_info(DRV_NAME ": %s(devid=%xh): intf#=%d EP#=%u\n",
		__func__, devid, intf_num, if_desc->desc.bNumEndpoints);
#endif

	/* look for the corresponding USB interface */
	for (i = 0; i < ARRAY_SIZE(plin_usb_intfs_list); i++)  {
		intf_desc = plin_usb_intfs_list + i;
		if (to_intf_desc(intf_desc)->devid == devid)
			break;
	}

	if (i >= ARRAY_SIZE(plin_usb_intfs_list)) {
		dev_err(&usb_intf->dev,
			"this LIN device is not supported by the driver\n");
		return -ENODEV;
	}

	/* All EP proposed by this interface MUST be known EP */
	for (i = 0; i < if_desc->desc.bNumEndpoints; i++) {
		struct usb_endpoint_descriptor *ep = &if_desc->endpoint[i].desc;
		int j;

#ifdef DEBUG_USB_TRACE
		dev_info(&usb_intf->dev, "%02u: ep %02xh (wDataSz=%d)\n",
			 i, ep->bEndpointAddress,
			 le16_to_cpu(ep->wMaxPacketSize));
#endif

		for (j = 0; j < ARRAY_SIZE(intf_desc->ep_list); j++)
			if (ep->bEndpointAddress == intf_desc->ep_list[j])
				break;

		if (j >= ARRAY_SIZE(intf_desc->ep_list)) {
#ifdef DEBUG_USB_TRACE
			dev_info(&usb_intf->dev, "ep %02xh not in LIN range\n",
				 ep->bEndpointAddress);
#endif
			return -ENODEV;
		}
	}

#if 1
	/* useless */
#else
	/* makes alternate settings #0 the current of this interface (LIN) */
	err = usb_set_interface(usb_dev, intf_num, 0);
	if (err < 0) {
		pr_err(DRV_NAME ": usb_set_interface() failed! (err %d)\n",
			err);
		return err;
	}
#endif


	/* alloc a usb device intf first */
	sizeof_intf = sizeof(struct plin_usb_intf);

#ifdef PLIN_USB_URB_USES_OWN_BUFFER
	sizeof_intf += usb_rxurbsmax * sizeof(u8 *);
#else
	/* allocate usb_rxurbsmax * URBs of usb_rxurblen bytes each */
	sizeof_intf += usb_rxurbsmax * usb_rxurblen;
#endif

	_intf = _plin_alloc_intf(sizeof_intf, to_intf_desc(intf_desc));
	if (!_intf) {
		err = -ENOMEM;
		goto lbl_exit;
	}

	/* init intf object BEFORE using it */
	_intf->free_intf = plin_usb_free_intf;

	intf = to_usb_intf(_intf);

#ifdef PLIN_USB_LOCK_CMD_BUFFER_WITH_SPINLOCK
	spin_lock_init(&intf->cmd_lock);
#endif
	intf->usb_intf = usb_intf;
	intf->cmd_timeout = PLIN_CMD_TIMEOUT_DEF;
	intf->buflen = usb_rxurblen;
	intf->bufidx_next = -1;

	init_usb_anchor(&intf->rx_free_urbs);
	init_usb_anchor(&intf->rx_submitted_urbs);
	atomic_set(&intf->rx_submitted, 0);

	/* first, get fw info */
	switch (devid) {
#ifdef PCAN_USBPRO_PRODID
	case PCAN_USBPRO_PRODID:
		/* need 512 bytes buffers */
		err = plin_usb_send_req(intf, USB_VENDOR_REQ_INFO,
					USB_VENDOR_INFO_FW);
		if (err < sizeof(struct usb_pro_fw_info)) {
			pr_err(DRV_NAME ": %s #%d: fw info failure (err=%d)\n",
			       intf_name(_intf), intf_idx(_intf), err);

			goto lbl_free_intf;
		} else {
			struct usb_pro_fw_info *fwi;

			fwi = (struct usb_pro_fw_info *)intf->cmd_buffer;
			for (i = 0; i < ARRAY_SIZE(_intf->fw_ver); i++)
				if (i < ARRAY_SIZE(fwi->ver))
					_intf->fw_ver[i] = fwi->ver[i];
		}
		break;
#endif
#ifdef PCAN_USBPRO_FD_PRODID
	case PCAN_USBPRO_FD_PRODID:
#if 1
		/* need 512 bytes buffers */
		err = plin_usb_send_req(intf, USB_VENDOR_REQ_INFO,
					USB_VENDOR_INFO_FW);
		if (err < sizeof(struct usb_fd_fw_info_v1)) {
			pr_err(DRV_NAME ": %s #%d: fw info failure (err=%d)\n",
			       intf_name(_intf), intf_idx(_intf), err);

			goto lbl_free_intf;
		} else {
			struct usb_fd_fw_info_v1 *fwi;

			fwi = (struct usb_fd_fw_info_v1 *)intf->cmd_buffer;
			for (i = 0; i < ARRAY_SIZE(_intf->fw_ver); i++)
				if (i < ARRAY_SIZE(fwi->fw_ver))
					_intf->fw_ver[i] = fwi->fw_ver[i];
		}
		break;
#else
		/* looks like those device return 256 bytes (instead of 512)
		 * buffer only
		 */
		/* fall through */
		fallthrough;
#endif
#endif
#ifdef PLIN_USB_PRODID
	case PLIN_USB_PRODID:
		err = plin_usb_get_fw_ver(intf);
		if (err)
			goto lbl_free_intf;
		break;
#endif
	}

	/* Rx part */
	for (j = 0; j < usb_rxurbsmax; j++) {
		u8 *urb_buffer;

		urb = usb_alloc_urb(0, GFP_KERNEL);
		if (!urb)
			break;

#ifdef PLIN_USB_URB_USES_OWN_BUFFER
		intf->rx_buffer[j] = kzalloc(intf->buflen, GFP_KERNEL|GFP_DMA);
		if (!intf->rx_buffer[j]) {
			usb_free_urb(urb);
			break;
		}
		urb_buffer = intf->rx_buffer[j];
#else
		urb_buffer = intf->rx_buffer + j * intf->buflen;
#endif
		usb_fill_bulk_urb(urb, usb_dev, 
			usb_rcvbulkpipe(usb_dev,intf_ep_list(intf)[EP_DATA_RX]),
			urb_buffer,
			intf->buflen,
			plin_usb_rx_callback, intf);

		/* link it to the Rx urbs free list */
		usb_anchor_urb(urb, &intf->rx_free_urbs);

#ifdef DEBUG_USB_RX
		pr_info(DRV_NAME ": urb@%p anchored to free list\n", urb);
#endif
	}

	if (j < usb_rxurbsmax) {
		pr_warn(DRV_NAME ": %u Rx urbs allocated instead of %u\n",
			j, usb_rxurbsmax);
	}

	/* link that interface to the driver list */
	plin_link_intf(_intf);

	/* create a LIN dev for each controller */
	sizeof_dev = sizeof(struct plin_usb_dev);

#ifdef PLIN_USB_URB_USES_OWN_BUFFER
	sizeof_dev += usb_txurbsmax * sizeof(u8 *);
#else
	sizeof_dev += usb_txurbsmax * PLIN_TXMSG_LEN;
#endif

	for (i = 0; i < intf_ctrl_count(_intf); i++) {

		_dev = _plin_alloc_dev(sizeof_dev);
		if (!_dev) {
			err = -ENOMEM;
			goto lbl_unlink_intf;
		}

		_dev->free_dev = plin_usb_free_dev;

		_dev->send_direct_cmd = _plin_usb_send_cmd;
		_dev->read_direct_rsp = _plin_usb_read_rsp;

		_dev->write = plin_usb_write;
		_dev->write_busy = plin_usb_write_busy;

		_dev->cleanup = plin_usb_cleanup;

		_dev->identify = plin_usb_identify;
#ifdef PLIN_USB_USES_CACHE_ID_STR
		_dev->get_id_str_nolock = plin_usb_get_cached_id_user_str;
		_dev->get_id_num_nolock = plin_usb_get_cached_id_user_num;

#elif !defined(PLIN_USB_LOCK_CMD_BUFFER_WITH_SPINLOCK)
		/* WARNING! theses accesses SHOULD not use any spinlocks! */
		_dev->get_id_str_nolock = plin_usb_get_id_user_str;
		_dev->get_id_num_nolock = plin_usb_get_id_user_num;
#else
		/* donot propose any direct access in these conditions */
#endif
		_dev->set_id_str = plin_usb_set_id_user_str;
		_dev->set_id_num = plin_usb_set_id_user_num;
		_dev->set_led_state = plin_usb_set_led_state;

		dev = to_usb_dev(_dev);

		init_usb_anchor(&dev->tx_free_urbs);
		init_usb_anchor(&dev->tx_submitted_urbs);

		/* Tx part */
		switch (i) {
		case 0:
			dev->ep_data_out = intf_ep_list(intf)[EP_DATA_TX_1];
			break;
		case 1:
			dev->ep_data_out = intf_ep_list(intf)[EP_DATA_TX_2];
			break;
		}

		for (j = 0; j < usb_txurbsmax; j++) {
			u8 *urb_buffer;

			urb = usb_alloc_urb(0, GFP_KERNEL);
			if (!urb)
				break;

#ifdef PLIN_USB_URB_USES_OWN_BUFFER
			dev->tx_buffer[j] = kzalloc(PLIN_TXMSG_LEN,
						    GFP_KERNEL|GFP_DMA);
			if (!dev->tx_buffer[j]) {
				usb_free_urb(urb);
				break;
			}
			urb_buffer = dev->tx_buffer[j];
#else
			urb_buffer = dev->tx_buffer + j * PLIN_TXMSG_LEN,
#endif

			usb_fill_bulk_urb(urb, usb_dev, 
				usb_sndbulkpipe(usb_dev, dev->ep_data_out),
				urb_buffer,
				PLIN_TXMSG_LEN,
				plin_usb_tx_callback, dev);

			/* link it to the Tx urbs free list */
			usb_anchor_urb(urb, &dev->tx_free_urbs);
		}

		if (j < usb_txurbsmax) {
			pr_warn(DRV_NAME
				": %u Tx urbs allocated instead of %u\n",
				j, usb_txurbsmax);
		}

		/* link that device to the current interface */
		plin_link_dev(_intf, _dev);

#ifdef PLIN_USB_USES_CACHE_ID_STR
		/* cache id_num and id_str for Udev access */
		plin_usb_cache_id_str(_dev);
#endif
		/* hw reset the device just like it was plugged in */
		if (usb_hwreset)
			plin_usb_hw_reset(_dev);

		if (usb_identify)
			plin_usb_identify(_dev);
	}

	/* back linkage */
	usb_set_intfdata(usb_intf, intf);

	dev_info(&usb_intf->dev, "%s #%u fw=%u.%u.%u: %uxLIN (%s speed):\n",
		 intf_name(_intf), intf_idx(_intf),
		 _intf->fw_ver[0],
		 _intf->fw_ver[1],
		 _intf->fw_ver[2],
		 intf_ctrl_count(_intf),
		 plin_usb_speed_name(usb_dev->speed));

	plin_intf_foreach_dev(_intf, plin_usb_do_dev_info, intf);

	/* tell the LIN USB interface the driver is loaded */
	plin_usb_drv_loaded(intf, 1);

#ifdef WORKAROUND_PCAN_USBPRO_FD_TIMESYNC_ISSUE
#define FW_VERSION(a, b, c)	KERNEL_VERSION(a, b, c)
	if ((devid == PCAN_USBPRO_FD_PRODID) &&
	    (FW_VERSION(_intf->fw_ver[0],
			_intf->fw_ver[1],
			_intf->fw_ver[2]) < FW_VERSION(3, 5, 0)))
		pcan_usb_pro_fd_timesync_issue(intf);
#endif

	/* Udev can run */
	plin_sysfs_create_intf(_intf);

	return 0;

lbl_unlink_intf:

	plin_unlink_intf(_intf);

lbl_free_intf:

	/* free the LIN interface as well as all linked LIN devices */
	plin_free_intf(_intf);

lbl_exit:
	return err;
}

static void plin_usb_disconnect(struct usb_interface *usb_intf)
{
	struct plin_intf *_intf;
	struct plin_usb_intf *intf = usb_get_intfdata(usb_intf);
	if (!intf)
		return;

	_intf = to_intf(intf);

#ifdef DEBUG_USB_TRACE
	pr_info(DRV_NAME ": %s(%s #%u)\n", __func__,
		intf_name(_intf), intf_idx(_intf));
#endif

	if (!unloading_driver)
		dev_info(&usb_intf->dev, "%s #%u disconnected\n",
			 intf_name(_intf), intf_idx(_intf));

	/* tell the LIN USB interface the driver is being removed */
	plin_usb_drv_loaded(intf, 0);

	/* unlink interface from the driver list */
	plin_unlink_intf(to_intf(intf));

	/* free memory used by the interface and its LIN devices */
	plin_free_intf(_intf);

	/* prevent from reentrance */
	usb_set_intfdata(usb_intf, NULL);
}

static struct usb_driver plin_usb_driver = {
	.name = DRV_NAME,
	.probe = plin_usb_probe,
	.disconnect = plin_usb_disconnect,
	.id_table = plin_usb_devids,
	.pre_reset = plin_usb_pre_reset,
	.post_reset = plin_usb_post_reset,
	.suspend = plin_usb_suspend,
	.resume = plin_usb_resume,
};

/*
 * plin driver sub module interface
 */
int plin_usb_init(void)
{
	int err;

#ifdef DEBUG_USB_TRACE
	pr_info(DRV_NAME ": %s()\n", __func__);
#endif

	if (usb_rxurbsmax > PLIN_USB_RX_URBS_MAX)
		usb_rxurbsmax = PLIN_USB_RX_URBS_MAX;
	else if (usb_rxurbsmax < PLIN_USB_RX_URBS_MIN)
		usb_rxurbsmax = PLIN_USB_RX_URBS_MIN;

	if (usb_rxurblen > PLIN_USB_RX_URBLEN_MAX)
		usb_rxurblen = PLIN_USB_RX_URBLEN_MAX;
	else if (usb_rxurblen < PLIN_USB_RX_URBLEN_MIN)
		usb_rxurblen = PLIN_USB_RX_URBLEN_MIN;

	err = usb_register(&plin_usb_driver);
	if (err)
		pr_err(DRV_NAME ": usb_register failed (err %d)\n", err);

	return err;
}

void plin_usb_exit(void)
{
#ifdef DEBUG_USB_TRACE
	pr_info(DRV_NAME ": %s()\n", __func__);
#endif

	unloading_driver = 1;
	usb_deregister(&plin_usb_driver);
}
