/* SPDX-License-Identifier: LGPL-2.1-only */
/*
 * lin.c - a program to configure a LIN device.
 *
 * Copyright (C) 2015-2025  PEAK System-Technik GmbH <www.peak-system.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 * Contact: <linux.peak@hms-networks.com>
 * Author:  Stephane Grosjean <stephane.grosjean@hms-networks.com>
 */
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>

/* chardev interface */
#include <fcntl.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <errno.h>

#include <string.h>
#include <ctype.h>

#include "plin.h"
#include "plin_version.h"

#ifndef ARRAY_SIZE
#define ARRAY_SIZE(x)		(sizeof(x) / sizeof((x)[0]))
#endif

#define STR(s)			#s
#define __stringify(x)		STR(x)

//#define DEBUG_TRACE

#define VERSION_MAJOR		DRV_VER_MAJOR
#define VERSION_MINOR		DRV_VER_MINOR
#define VERSION_SUBMINOR	DRV_VER_SUBMINOR

struct token {
	char abbrv;
	char *str;
};

static struct token mode_name[] = {
	{ 'n', "none" },
	{ 'S', "slave" },
	{ 'M', "master" }
};

static struct token dir_name[] = {
	{ 'd', "disabled" },
	{ 'p', "publisher" },
	{ 's', "subscriber" },
	{ 'S', "subscriber (auto len)" }
};

static struct token cs_name[] = {
	{ 'u', "custom" },
	{ 'c', "classic" },
	{ 'e', "enhanced" },
	{ 'a', "auto" }
};

static struct token slot_type_name[] = {
	{ 'u', "unconditional frame" },
	{ 'e', "event frame" },
	{ 's', "sporadic frame" },
	{ 'M', "master request" },
	{ 'S', "slave response" }
};

struct context;
struct cmd {
	char *name;
	int (*handler)(struct context *ctx, int argc, char *argv[]);
	int (*usage)(struct context *ctx, char *fmt, ...);
	char *abbrev;
	char *help_msg;
	char *abstract;		/* should not be larger than 40 char */
};

struct context {
	int lin_fd;
	struct cmd *history[8];
	int history_lvl;
	int cmd;
	int cmd_ioctl_loop_min, cmd_ioctl_loop_max, cmd_ioctl_loop;
	int cmd_ioctl_user_data;
	int (*cmd_ioctl_pre)(struct context *ctx);
	char *dev_name;
	int (*rsp_handler)(struct context *ctx, int c);

	union {
		struct plin_usb_init_hw init_hw;
		struct plin_usb_frm_entry frm_entry;
		struct plin_usb_auto_baud auto_baud;
		struct plin_usb_get_baudrate get_baudrate;
		struct plin_usb_id_filter id_filter;
		struct plin_usb_get_mode get_mode;
		struct plin_usb_ident_str ident_str;
		struct plin_usb_fw_ver fw_ver;
		struct plin_usb_keep_alive keep_alive;
		struct plin_usb_add_schd_slot add_schd_slot;
		struct plin_usb_del_schd del_schd;
		struct plin_usb_get_slot_count get_slot_count;
		struct plin_usb_get_schd_slot get_schd_slot;
		struct plin_usb_start_schd start_schd;
		struct plin_usb_resume_schd resume_schd;
		struct plin_usb_suspend_schd suspend_schd;
		struct plin_usb_get_status get_status;
		struct plin_usb_update_data update_data;
		struct plin_usb_rsp_remap rsp_remap;
		struct plin_usb_led_state leds_state;
		struct plin_usb_set_schd_brkpt schd_brkpt;
#ifndef NO_PRIVATE
		unsigned char plin_private_data[1024];
#endif
	} u;
};

#define DECLARE_CMD(_name)	\
static int _name##_handler(struct context *ctx, int argc, char *argv[]); \
static int _name##_usage(struct context *ctx, char *fmt, ...);

#define DECLARE_CMD_NOHELP(_name)	\
static int _name##_handler(struct context *ctx, int argc, char *argv[]);

#define __CMD_FULL(_name, _handler, _usage, _abbrv, _help, _abst) \
	{ __stringify(_name), _handler, _usage, _abbrv, _help, _abst }

#define __CMD_FU(_name, _func, _abbrv, _help) \
	__CMD_FULL(_name, _func##_handler, _func##_usage, _abbrv, _help, NULL)

#define __CMD_F(_name, _func, _abbrv, _abst) \
	__CMD_FULL(_name, _func##_handler, NULL, _abbrv, NULL, _abst)

#define __CMD(_name, _abbrv, _help) \
	__CMD_FU(_name, _name, _abbrv, _help)

#define __CMD_FUA(_name, _func, _abbrv, _help, _abst) \
	__CMD_FULL(_name, _func##_handler, _func##_usage, _abbrv, _help, _abst)

#define __CMD_FA(_name, _func, _abbrv, _help, _abst) \
	__CMD_FULL(_name, _func##_handler, NULL, _abbrv, _help, _abst)

#define __CMD_A(_name, _abbrv, _help, _abst) \
	__CMD_FUA(_name, _name, _abbrv, _help, _abst)

DECLARE_CMD(set);
DECLARE_CMD(get);
DECLARE_CMD(start);
DECLARE_CMD(disconnect);
DECLARE_CMD(wake_up);
DECLARE_CMD(suspend);
DECLARE_CMD(resume);
DECLARE_CMD(add);
DECLARE_CMD(del);
DECLARE_CMD(identify);
DECLARE_CMD(reset);
DECLARE_CMD(help);
DECLARE_CMD(version);

static struct cmd cmds[] = {
	__CMD_A(set, "-s",
"Allows to give a value to variables and objets that define how the given LIN device works",
"set frames entries, ID filters, LEDs..." ),
	__CMD_A(get, "-g",
"Allows to read the current values of variables and other objects that define the operation of the given LIN device",
"get values set in the device"),
	__CMD_A(add, "-a",
"Add a row to some internal table of the given LIN device",
"add row in frames and scheduler tables"),
	__CMD_A(del, "-d",
"Delete a specified row from some internal table of the given LIN device",
"delete row in frames/scheduler tables"),
	__CMD_A(start, "go",
"Start some internal process from the given LIN device",
"start master/slave mode, scheduler..." ),
	__CMD_A(disconnect, "stop",
"Disconnect the given LIN device from the LIN bus",
"disconnect the LIN device" ),
	__CMD_FUA(wake-up, wake_up, NULL,
"Write a wake up pulse onto the LIN bus to go out from SLEEP state (slave only)",
"wake-up the LIN bus (slave mode)" ),
	__CMD_A(suspend, "pause",
"Suspend any started internal process from the given LIN device",
"suspend scheduler, keep-alive..."),
	__CMD_A(resume, "restart",
"Resume running internal process from the given LIN device",
"resume scheduler, keep-alive..."),
	__CMD_A(identify, "id",
"Identify the given LIN device by flashing its LED",
"flash the LED of the given LIN device"),
	__CMD_A(reset, "rst",
"Either reset the entire LIN device or its Tx queue only",
"hard/soft reset the device"),
	__CMD_A(help, "-h",
"Display contextual help", "display this help"),
	__CMD_A(version, "-v",
"Display the version of the software", "display the version"),

	{ NULL, }
};

static struct cmd *search_cmd_by_name(char *name, struct cmd cmds[])
{
	struct cmd *cmd;

	for (cmd = cmds; cmd->name; cmd++)
		if (!strcmp(cmd->name, name))
			return cmd;
		else if (cmd->abbrev && !strcmp(cmd->abbrev, name))
			return cmd;

	return NULL;
}

static int cmd_help(struct context *ctx, char *fmt, va_list ap)
{
	if (fmt) {
		fprintf(stderr, "Error: ");
		vfprintf(stderr, fmt, ap);
	} else {
		fprintf(stderr, "%s.\n",
			ctx->history[ctx->history_lvl-1]->help_msg);
	}

	return 1;
}

static int cmd_usage(struct context *ctx, char *fmt, ...)
{
	int i;
	va_list ap;

	va_start(ap, fmt);
	fprintf(stderr,"\nUSAGE: ");
	for (i = 0; i < ctx->history_lvl-1; i++)
		fprintf(stderr,"%s ", ctx->history[i]->name);

	fputs(ctx->history[i]->name, stderr);
	if (ctx->history[i]->abbrev)
		fprintf(stderr,"|%s", ctx->history[i]->abbrev);
	putc(' ', stderr);

	vfprintf(stderr, fmt, ap);
	va_end(ap);

	return 1;
}

#define TABSIZE		8
#define SCREEN_WIDTH	80

static int subcmds_usage(struct context *ctx, char *prompt, struct cmd cmds[])
{
	struct cmd *c;
	int i;

	fprintf(stderr, "\n%s:\n", prompt);
	for (c = cmds; c->name; c++) {
		int l = TABSIZE;

		putc('\t', stderr);
		for (i = 0; i < ctx->history_lvl; i++)
			l += fprintf(stderr, "%s ", ctx->history[i]->name);
		l += fprintf(stderr, "%s", c->name);
		if (c->abbrev)
			l += fprintf(stderr, "|%s", c->abbrev);

		if (c->abstract) {
			/* add as many '\t' as need to reach SCREEN_WIDTH/2 */
			for (; l < SCREEN_WIDTH/2; l += TABSIZE) 
				putc('\t', stderr);
			fprintf(stderr, "%s", c->abstract);
		}
		putc('\n', stderr);
	}

	fprintf(stderr, "\nSEE ALSO:\n\t");
	for (i = 0; i < ctx->history_lvl; i++)
		fprintf(stderr, "%s ", ctx->history[i]->name);
	fprintf(stderr, "%s help\n", prompt);

	return 1;
}

static int opts_usage(struct context *ctx, char *prompt, struct cmd cmds[])
{
	struct cmd *c;
	int i;

	fprintf(stderr, "\n%s:\n", prompt);
	for (c = cmds; c->name; c++) {
		int l = TABSIZE;
		
		l += fprintf(stderr, "\t%s", c->name) - 1;
		if (c->abbrev)
			l += fprintf(stderr, "|%s", c->abbrev);

		if (c->abstract) {
			for (; l < SCREEN_WIDTH/2; l += TABSIZE)
				putc('\t', stderr);
			fprintf(stderr, "%s", c->abstract);
		}

		putc('\n', stderr);
	}

	return 1;
}

/*
 * int cmd_dev_usage(struct context *ctx, char *fmt, ...)
 *
 * Default usage handler for a "classic" command that needs (only) the
 * node of the device.
 */
static int cmd_dev_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "/dev/plinX\n");
}

static int cmd_handler(struct context *ctx, struct cmd sub_cmds[],
		       int argc, char *argv[])
{
	struct cmd *cmd, *sub_cmd;

#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif
	cmd = ctx->history[ctx->history_lvl-1];
	if (argc < 2)
		return cmd->usage(ctx, NULL);

	sub_cmd = search_cmd_by_name(argv[1], sub_cmds);
	if (!sub_cmd)
		return cmd->usage(ctx, "%s: \"%s\" unknown function\n",
				  cmd->name, argv[1]);

	ctx->history[ctx->history_lvl++] = sub_cmd;

	/* shortcut to "SUBCMD help" */
	if (argc > 2)
		if (!strcmp(argv[2], "help") || !strcmp(argv[2], "-h"))
			if (sub_cmd->usage)
				return sub_cmd->usage(ctx, NULL);

	return sub_cmd->handler(ctx, argc-1, argv+1);
}

static int usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	cmd_usage(ctx, "COMMAND [OPTIONS] [/dev/plinX]\n");
	return subcmds_usage(ctx, "COMMAND", cmds);
}

/*
 * strtohunit(argv, "kM");
 * strtohunit(argv, "ms");
 */
static int strtohunit(const char *str, void *pv, const char *units)
{
	char *endptr;
	__u32 m = 1;
	__le16 v16;

	__u32 v = strtoul(str, &endptr, 0);
	if (*endptr) {
		const char *pu;

		if (!units)
			return 1;

		/* might not be invalid if found char is a unit */
		for (pu = units; *pu; pu++) {
			m *= 1000;
			if (*endptr == *pu)
				break;
		}

		if (!*pu)
			return 2;
	}

	/* need to use a tmp v16 and memcpy() because pv might be unaligned */
	v16 = htole16(v * m);
	memcpy(pv, &v16, 2);

	return 0;
}

static int strtobs(const char *str, int n, u8 *pb)
{
	int i;

	if (!pb)
		return 0;

	memset(pb, '\0', n);
	for (i = 0; *str && (i < n); i++) {

		char *endptr;

		__u32 v	= strtoul(str, &endptr, 0);
		switch (*endptr) {
		case ' ':
		case ',':
		case ':':
		case '-':
			endptr++;
		case '\0':
			str = endptr;
			break;
		default:
			*pb = 0xff;	/* error */
			return i;
		}

		*pb++ = (u8 )v;
	}

	return i;
}

static int islongopt(const char *argv, const char *opt)
{
	return !strncmp(argv+2, opt, strlen(opt));
}

static int do_pre_frm_entry(struct context *ctx)
{
#ifdef DEBUG_TRACE
	printf("%s(loop=%d)\n", __func__, ctx->cmd_ioctl_loop);
#endif

	ctx->u.frm_entry.id = (u8 )ctx->cmd_ioctl_loop;
	return 0;
}

static u8 get_frm_default_len(u8 id)
{
	if (id < 0x20)
		return 2;
	if (id < 0x30)
		return 4;

	return 8;
}

/*
 * $ lin set 
 */
DECLARE_CMD(set_sub_fe);
DECLARE_CMD(set_pub_fe);
DECLARE_CMD(set_fe_data);
DECLARE_CMD(set_id_filter);
DECLARE_CMD(set_id_string);
DECLARE_CMD(set_id_num);
DECLARE_CMD(set_schd_brkpt);
DECLARE_CMD(set_rsp_remap);
DECLARE_CMD(set_leds);
#if 0//ndef NO_PRIVATE
DECLARE_CMD(set_private);
#endif
static struct cmd set_cmds[] = {
	__CMD_FUA(sub-frm-entry, set_sub_fe, "sfe",
"Configure a subscribe frame entry in the LIN device frame table",
"define a subscribe frame entry"),
	__CMD_FUA(pub-frm-entry, set_pub_fe, "pfe",
"Configure a publish frame entry in the LIN device frame table",
"define a publish frame entry"),
	__CMD_FUA(frm-entry-data, set_fe_data, "fed",
"Update the reponse data of a publish frame entry of the LIN device frame table",
"update data of a publish frame entry"),
	__CMD_FUA(id-filter, set_id_filter, "if",
"Set an USB filter on LIN frame IDs to forward to applications",
"filter IDs given to applications" ),
	__CMD_FUA(ident-string, set_id_string, "is",
"Define a 24-bytes max user string to save in the device memory",
"save a user string into device memory"),
	__CMD_FUA(ident-num, set_id_num, "in",
"Define a user 32 bit value to save in the device memory",
"save a user private number into device memory"),
	__CMD_FUA(schd-brkpt, set_schd_brkpt, "bp",
"Define a breakpoint for a specific schedule in the device",
"define a breakpoint in a scheduler"),
	__CMD_FUA(rsp-remap, set_rsp_remap, NULL,
"Set the response remap used for event frames in slave mode",
"set response remap"),
	__CMD_FUA(leds, set_leds, NULL,
"Switch ALL the LEDs of the given device on/off",
"switch a device LEDs on/off"),
#if 0//ndef NO_PRIVATE
	__CMD_FUA(private, set_private, "sp",
"Set ioctl private command",
"reserved usage"),
#endif
	{ NULL, }
};

static int set_handler(struct context *ctx, int argc, char *argv[])
{
	return cmd_handler(ctx, set_cmds, argc, argv);
}

static int set_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	cmd_usage(ctx, "FUNCTION [OPTIONS] /dev/plinX\n");
	return subcmds_usage(ctx, "FUNCTION", set_cmds);
}

/*
 * $ lin set sub-frm-entry
 */
DECLARE_CMD_NOHELP(set_fe_cs_1);
DECLARE_CMD_NOHELP(set_fe_cs_2);
DECLARE_CMD_NOHELP(set_sfe_auto_len);
DECLARE_CMD_NOHELP(set_sfe_cs_auto);
DECLARE_CMD_NOHELP(set_sfe_len);
static struct cmd sub_fe_opts[] = {
	__CMD_F(--cs-classic, set_fe_cs_1, "-c",
"classic checksum (default LIN 1.3)"),
	__CMD_F(--cs-enhanced, set_fe_cs_2, "-e",
"enhanced checksum (LIN 2.0+)"),
	__CMD_F(--cs-auto, set_sfe_cs_auto, "-a",
"automatic checksum (subscriber only)"),
	__CMD_F(--len, set_sfe_len, "-l",
"length of frame [1..8]"),
	__CMD_F(--auto-len, set_sfe_auto_len, "-A",
"subscriber using auto length"),

	{ NULL, }
};

static int set_sfe_auto_len_handler(struct context *ctx, int argc, char *argv[])
{
	ctx->u.frm_entry.direction = PLIN_FRM_DIR_SUBSCRIBER_AUTO_LEN;
	return 0;
}

static int set_sfe_cs_auto_handler(struct context *ctx, int argc, char *argv[])
{
	ctx->u.frm_entry.checksum = PLIN_FRM_CST_AUTO;
	return 0;
}

static int set_fe_cs_1_handler(struct context *ctx, int argc, char *argv[])
{
	ctx->u.frm_entry.checksum = PLIN_FRM_CST_CLASSIC;
	return 0;
}

static int set_fe_cs_2_handler(struct context *ctx, int argc, char *argv[])
{
	ctx->u.frm_entry.checksum = PLIN_FRM_CST_ENHANCED;
	return 0;
}

static int set_sfe_len_handler(struct context *ctx, int argc, char *argv[])
{
	/* next arg MUST be the frame length */
	if (argc < 2)
		return -set_sub_fe_usage(ctx, "missing frame length\n");

	if ((strtobs(argv[1], 1, &ctx->u.frm_entry.len) != 1) ||
		(ctx->u.frm_entry.len > PLIN_DAT_LEN) ||
		(ctx->u.frm_entry.len < 1))
		return -set_sub_fe_usage(ctx, "invalid frame length [1..%u]\n",
					PLIN_DAT_LEN);

	/* Tell we have processed next cmdline argument */
	return 1;
}

static int set_sub_fe_handler(struct context *ctx, int argc, char *argv[])
{
	struct cmd *opt;
	int i;

#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif
	if (argc == 1)
		return set_sub_fe_usage(ctx, NULL);

	else if (argc < 3)
		return set_sub_fe_usage(ctx, "not enough arguments\n");

	memset(&ctx->u.frm_entry, '\0', sizeof(ctx->u.frm_entry));

	ctx->u.frm_entry.direction = PLIN_FRM_DIR_SUBSCRIBER;
	ctx->u.frm_entry.checksum = PLIN_FRM_CST_CLASSIC;

	/* Frame ID (or "all") */
	if (!strcmp(argv[1], "all")) {
		ctx->cmd_ioctl_loop_min = PLIN_FRM_ID_MIN;
		ctx->cmd_ioctl_loop_max = PLIN_FRM_ID_MAX;
		ctx->cmd_ioctl_pre = do_pre_frm_entry;
	} else if ((strtobs(argv[1], 1, &ctx->u.frm_entry.id) != 1) ||
				(ctx->u.frm_entry.id > PLIN_FRM_ID_MAX))
		return set_sub_fe_usage(ctx, "ID out of range [%u..%u]\n",
					   PLIN_FRM_ID_MIN,
					   PLIN_FRM_ID_MAX);

	/* everything between ID and /dev might be an option */
	for (i = 2; i < argc-1; ) {
		int s;
		struct cmd *opt = search_cmd_by_name(argv[i], sub_fe_opts);
		if (!opt)
			return set_sub_fe_usage(ctx,
					"\"%s\" invalid option\n", argv[i]);

		/* argc doesn't take last arg in account
		 * (since it MUST be /dev/plinX)
		 */
		s = opt->handler(ctx, argc-1-i, argv+i);
		if (s < 0)
			return -s;

		i += 1 + s;
	}

	/* if length not specified, use default one according to ID range: */
	if (!ctx->u.frm_entry.len)
		ctx->u.frm_entry.len = get_frm_default_len(ctx->u.frm_entry.id);

	ctx->cmd = PLIOSETFRMENTRY;
	ctx->dev_name = argv[argc-1];

	return 0;
}

static int set_sub_fe_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	cmd_usage(ctx, "ID|all [OPTIONS] /dev/plinX\n");
	return opts_usage(ctx, "OPTIONS", sub_fe_opts);
}

/*
 * $ lin set pub-frm-entry
 */
DECLARE_CMD_NOHELP(set_pfe_rsp_en);
DECLARE_CMD_NOHELP(set_pfe_sng_shot);
DECLARE_CMD_NOHELP(set_pfe_ign_data);
DECLARE_CMD_NOHELP(set_pfe_data);
DECLARE_CMD_NOHELP(set_pfe_len);
static struct cmd pub_fe_opts[] = {
	__CMD_F(--cs-classic, set_fe_cs_1, "-c",
"classic checksum (default LIN 1.3)"),
	__CMD_F(--cs-enhanced, set_fe_cs_2, "-e",
"enhanced checksum (LIN 2.0+)"),
	__CMD_F(--rsp-enable, set_pfe_rsp_en, "-r",
"response enable (default)"),
	__CMD_F(--sng-shot, set_pfe_sng_shot, "-s",
"single shot"),
	__CMD_F(--ign-data, set_pfe_ign_data, "-i",
"ignore initial data"),
	__CMD_F(--data, set_pfe_data, "-d",
"next arg is the data bytes string"),
	__CMD_F(--len, set_pfe_len, "-l",
"next arg is the frame length [1..8]"),

	{ NULL, }
};

/*
 * Note: when a publish frame is created with rsp-enable, then its data are
 * sent too.
 */
static int set_pfe_rsp_en_handler(struct context *ctx, int argc, char *argv[])
{
	ctx->u.frm_entry.flags |= htole16(PLIN_FRM_FLG_RSP_ENABLE);
	return 0;
}

static int set_pfe_sng_shot_handler(struct context *ctx, int argc, char *argv[])
{
	ctx->u.frm_entry.flags |= htole16(PLIN_FRM_FLG_SINGLE_SHOT);
	return 0;
}

static int set_pfe_ign_data_handler(struct context *ctx, int argc, char *argv[])
{
	ctx->u.frm_entry.flags |= htole16(PLIN_FRM_FLG_IGNORE_DATA);
	return 0;
}

static int set_pfe_data_handler(struct context *ctx, int argc, char *argv[])
{
	int l;

#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif

	/* next arg MUST be the data bytes string */
	if (argc < 2)
		return -set_pub_fe_usage(ctx, "missing data bytes\n");

	l = strtobs(argv[1], PLIN_DAT_LEN, ctx->u.frm_entry.d);
	ctx->u.frm_entry.len = l;

	/* Tell we have processed next cmdline argument */
	return 1;
}

static int set_pfe_len_handler(struct context *ctx, int argc, char *argv[])
{
	/* next arg MUST be the frame length */
	if (argc < 2)
		return -set_pub_fe_usage(ctx, "missing frame length\n");

	if ((strtobs(argv[1], 1, &ctx->u.frm_entry.len) != 1) ||
		(ctx->u.frm_entry.len > PLIN_DAT_LEN) ||
		(ctx->u.frm_entry.len < 1))
		return -set_pub_fe_usage(ctx, "invalid frame length [1..%u]\n",
					PLIN_DAT_LEN);

	/* Tell we have processed next cmdline argument */
	return 1;
}

static int set_pub_fe_handler(struct context *ctx, int argc, char *argv[])
{
	int i;

#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif

	if (argc == 1)
		return set_pub_fe_usage(ctx, NULL);

	else if (argc < 3)
		return set_pub_fe_usage(ctx, "not enough arguments\n");

	memset(&ctx->u.frm_entry, '\0', sizeof(ctx->u.frm_entry));

	ctx->u.frm_entry.direction = PLIN_FRM_DIR_PUBLISHER;
	ctx->u.frm_entry.checksum = PLIN_FRM_CST_CLASSIC;

	/* default flag is NOT to ignore initial data but enable response so
	 * that data bytes will be copied on the wire
	 */
	ctx->u.frm_entry.flags = htole16(PLIN_FRM_FLG_RSP_ENABLE);

	/* Frame ID (or "all") */
	if (!strcmp(argv[1], "all")) {
		ctx->cmd_ioctl_loop_min = PLIN_FRM_ID_MIN;
		ctx->cmd_ioctl_loop_max = PLIN_FRM_ID_MAX;
		ctx->cmd_ioctl_pre = do_pre_frm_entry;
	} else if ((strtobs(argv[1], 1, &ctx->u.frm_entry.id) != 1) ||
				(ctx->u.frm_entry.id > PLIN_FRM_ID_MAX))
		return set_pub_fe_usage(ctx, "ID out of range [%u..%u]\n",
					   PLIN_FRM_ID_MIN,
					   PLIN_FRM_ID_MAX);

	/* everything between ID and /dev might be an option */
	for (i = 2; i < argc-1; ) {
		int s;
		struct cmd *opt = search_cmd_by_name(argv[i], pub_fe_opts);
		if (!opt)
			return set_pub_fe_usage(ctx,
					"\"%s\" invalid option\n", argv[i]);

		/* argc doesn't take last arg in account
		 * (since it MUST be /dev/plinX)
		 */
		s = opt->handler(ctx, argc-1-i, argv+i);
		if (s < 0)
			return -s;

		i += 1 + s;
	}

	/* if length not specified, use default one according to ID range: */
	if (!ctx->u.frm_entry.len)
		ctx->u.frm_entry.len = get_frm_default_len(ctx->u.frm_entry.id);

	ctx->cmd = PLIOSETFRMENTRY;
	ctx->dev_name = argv[argc-1];

	return 0;
}

static int set_pub_fe_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	cmd_usage(ctx, "ID|all [OPTIONS] /dev/plinX\n");
	return opts_usage(ctx, "OPTIONS", pub_fe_opts);
}

/*
 * $ set frm-entry-data
 *
 * Change data of the pubished frame entry:
 *
 * $ lin add pub-frm-entry  0x11 --rsp-enable -d "1 2" /dev/plin0
 * $ lin add unc-schd-slot 0 1s 0x11 /dev/plin0
 * $ lin start schedule 0 /dev/plin0
 * ...
 * $ lin set frm-entry-data 0x11 -d "3 4" /dev/plin0
 */
DECLARE_CMD_NOHELP(set_fed_from);
DECLARE_CMD_NOHELP(set_fed_data);
static struct cmd set_fed_opts[] = {
	__CMD_F(--from, set_fed_from, "-f",
"define which byte is the first [0..7] (0 default)"),
	__CMD_F(--data, set_fed_data, "-d",
"define new data bytes to save"),

	{ NULL, }
};

static int set_fed_from_handler(struct context *ctx, int argc, char *argv[])
{
	/* next arg MUST be the data byte index */
	if (argc < 2)
		return -set_fe_data_usage(ctx,
					  "missing byte position [0..%u]\n",
					  PLIN_DAT_LEN-1);

	if ((strtobs(argv[1], 1, &ctx->u.update_data.idx) != 1) ||
				(ctx->u.update_data.idx > PLIN_DAT_LEN-1))
		return -set_fe_data_usage(ctx,
					  "invalid byte position [0..%u]\n",
					  PLIN_DAT_LEN-1);

	/* Tell we have processed next cmdline argument */
	return 1;
}

static int set_fed_data_handler(struct context *ctx, int argc, char *argv[])
{
	int l;

#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif

	/* next arg MUST be the data bytes string */
	if (argc < 2)
		return -set_fe_data_usage(ctx, "missing data bytes\n");

	if (ctx->u.update_data.idx >= PLIN_DAT_LEN)
		return -set_fe_data_usage(ctx, "wrong data byte index %d\n",
					  ctx->u.update_data.idx);

#if 1
	l = strtobs(argv[1], PLIN_DAT_LEN, ctx->u.update_data.d);
#else
	l = strtobs(argv[1], PLIN_DAT_LEN - ctx->u.update_data.idx,
		    ctx->u.update_data.d + ctx->u.update_data.idx);
#endif

	ctx->u.update_data.len = l;

	/* Tell we have processed next cmdline argument */
	return 1;
}

static int set_fe_data_handler(struct context *ctx, int argc, char *argv[])
{
	struct cmd *opt;
	int i;

#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif

	if (argc == 1)
		return set_fe_data_usage(ctx, NULL);

	else if (argc < 3)
		return set_fe_data_usage(ctx, "not enough arguments\n");

	/* from and idx default are 0 */
	memset(&ctx->u.update_data, '\0', sizeof(ctx->u.update_data));

	/* Frame ID (or "all") */
	if (!strcmp(argv[1], "all")) {
		ctx->cmd_ioctl_loop_min = PLIN_FRM_ID_MIN;
		ctx->cmd_ioctl_loop_max = PLIN_FRM_ID_MAX;
		ctx->cmd_ioctl_pre = do_pre_frm_entry;

	} else if ((strtobs(argv[1], 1, &ctx->u.update_data.id) != 1) ||
				(ctx->u.update_data.id > PLIN_FRM_ID_MAX))
		return set_fe_data_usage(ctx, "ID out of range [%u..%u]\n",
					   PLIN_FRM_ID_MIN,
					   PLIN_FRM_ID_MAX);

	/* everything between ID and /dev might be an option */
	for (i = 2; i < argc-1; ) {
		int s;
		struct cmd *opt = search_cmd_by_name(argv[i], set_fed_opts);
		if (!opt)
			return set_fe_data_usage(ctx,
					"\"%s\" invalid option\n", argv[i]);

		/* argc doesn't take last arg in account
		 * (since it MUST be /dev/plinX)
		 */
		s = opt->handler(ctx, argc-1-i, argv+i);
		if (s < 0)
			return s;

		i += 1 + s;
	}

	ctx->cmd = PLIOCHGBYTEARRAY;
	ctx->dev_name = argv[argc-1];

	return 0;
}

static int set_fe_data_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	cmd_usage(ctx, "ID [OPTIONS] /dev/plinX\n");
	return opts_usage(ctx, "OPTIONS", set_fed_opts);
}

/*
 * $ lin set id-filter
 */
static int set_id_filter_handler(struct context *ctx, int argc, char *argv[])
{
#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif
	if (argc == 1)
		return set_id_filter_usage(ctx, NULL);
	else if (argc != 3)
		return set_id_filter_usage(ctx, "wrong number of arguments\n");

	if (!strcmp(argv[1], "all-opened") ||
	    !strcmp(argv[1], "all-on") ||
	    !strcmp(argv[1], "o"))
		memset(ctx->u.id_filter.id_mask, 0xff,
		       sizeof(ctx->u.id_filter.id_mask));
	else if (!strcmp(argv[1], "all-closed") ||
		 !strcmp(argv[1], "all-off") ||
		 !strcmp(argv[1], "c"))
		memset(ctx->u.id_filter.id_mask, 0x00,
		       sizeof(ctx->u.id_filter.id_mask));

	else
		strtobs(argv[1], 8, ctx->u.id_filter.id_mask);

	ctx->cmd = PLIOSETIDFILTER;
	ctx->dev_name = argv[argc-1];

	return 0;
}

static int set_id_filter_usage(struct context *ctx, char *fmt, ...)
{
#if 0
	fprintf(stderr,
"Define the USB ID filter mask. This mask defines which ID is forwarded(1) to application and which is not(0). Each bit corresponds to an ID, most-left bit corresponds to ID0, while right-hand one corresponds to ID63.\n");
#endif
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx,
			 "xx:xx:xx:xx:xx:xx:xx:xx|all-opened|o|all-closed|c "
			 "/dev/plinX\n");
}

/*
 * $ lin set ident-string
 */
static int set_id_string_handler(struct context *ctx, int argc, char *argv[])
{
	int l;

#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif
	if (argc == 1)
		return set_id_string_usage(ctx, NULL);
	else if (argc != 3)
		return set_id_string_usage(ctx, "wrong number of arguments\n");

	l = strlen(argv[1]);
	if (l > sizeof(ctx->u.ident_str.user_str))
		l = sizeof(ctx->u.ident_str.user_str);

	memcpy(ctx->u.ident_str.user_str, argv[1], l);

	ctx->cmd = PLIOSETUSERSTR;
	ctx->dev_name = argv[argc-1];

	return 0;
}

static int set_id_string_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "\"user_string\" /dev/plinX\n");
}

/*
 * $ lin set ident-num
 */
static int set_id_num_handler(struct context *ctx, int argc, char *argv[])
{
	unsigned long int v;
	char *endptr;

#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif
	if (argc == 1)
		return set_id_num_usage(ctx, NULL);
	else if (argc != 3)
		return set_id_num_usage(ctx, "wrong number of arguments\n");

	v = strtoul(argv[1], &endptr, 0);
	if (*endptr)
		return set_id_num_usage(ctx,
				"\"%s\" invalid user value\n",
				 argv[1]);
#ifdef DEBUG_TRACE
	printf("%s() user_id=%lu\n", __func__, v);
#endif

	ctx->u.ident_str.user_id = htole32(v);

	ctx->cmd = PLIOSETUSERID;
	ctx->dev_name = argv[argc-1];

	return 0;
}

static int set_id_num_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "USER-VALUE /dev/plinX\n");
}

/*
 * $ lin set schd-brkpt
 */
static int set_schd_brkpt_rsp(struct context *ctx, int c)
{
	printf("%08x\n", ctx->u.schd_brkpt.handle);
	return 0;
}

static int set_schd_brkpt_handler(struct context *ctx, int argc, char *argv[])
{
	int l;

#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif
	if (argc == 1)
		return set_schd_brkpt_usage(ctx, NULL);
	else if (argc != 3)
		return set_schd_brkpt_usage(ctx, "wrong number of arguments\n");

	if ((strtobs(argv[1], 1, &ctx->u.schd_brkpt.brkpt) != 1) ||
		(ctx->u.schd_brkpt.brkpt > 1) )
		return set_schd_brkpt_usage(ctx, "BP out of range [0,1]\n");

	ctx->cmd = PLIOSETSCHDBP;
	ctx->dev_name = argv[argc-1];
	ctx->rsp_handler = set_schd_brkpt_rsp;

	return 0;
}

static int set_schd_brkpt_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "BP[0|1] /dev/plinX\n");
}

/*
 * $ lin set rsp-remap
 */
static int set_rsp_remap[PLIN_USB_RSP_REMAP_ID_LEN];
static int set_rsp_remap_rsp(struct context *ctx, int c)
{
	int i;

	/* modify the rsp map where it should be */
	for (i = 0; i < PLIN_USB_RSP_REMAP_ID_LEN; i++)
		if (set_rsp_remap[i] >= 0)
			ctx->u.rsp_remap.id[i] = (u8 )set_rsp_remap[i];

	/* change to set the new rsp map */
	ctx->u.rsp_remap.set_get = PLIN_USB_RSP_REMAP_SET;

	/* request another loop */
	ctx->cmd_ioctl_loop_max++;

	/* no response to handle */
	ctx->rsp_handler = NULL;

	return 0;
}

static int set_rsp_remap_handler(struct context *ctx, int argc, char *argv[])
{
	int i;

#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif

	/* first, get the current rsp remap */
	if (argc == 1)
		return set_rsp_remap_usage(ctx, NULL);
	else if (argc < 2)
		return set_rsp_remap_usage(ctx, "wrong number of arguments\n");

	for (i = 0; i < PLIN_USB_RSP_REMAP_ID_LEN; i++)
		set_rsp_remap[i] = -1;

	/* every arg found before /dev/plinX must be "--idN=XXX" */
	for (i = 1; i < argc-1; i++) {
		int id, v;
		int s = sscanf(argv[i], "--id%u=%i", &id, &v);
		if (s != 2)
			return set_rsp_remap_usage(ctx,
					"\"%s\" arg should be \"--idN=XXX\"\n",
					argv[i]);

		if (id < PLIN_FRM_ID_MIN || id > PLIN_FRM_ID_MAX)
			return set_rsp_remap_usage(ctx,
					"\"%s\" invalid ID number\n",
					argv[i]);
		if (v < PLIN_FRM_ID_MIN || v > PLIN_FRM_ID_MAX)
			return set_rsp_remap_usage(ctx,
					"\"%s\" invalid ID value\n",
					argv[i]);
		set_rsp_remap[id] = v;
	}

	memset(&ctx->u.rsp_remap, '\0', sizeof(ctx->u.rsp_remap));

	ctx->u.rsp_remap.set_get = PLIN_USB_RSP_REMAP_GET;
	ctx->cmd = PLIOSETGETRSPMAP;
	ctx->rsp_handler = set_rsp_remap_rsp;
	ctx->dev_name = argv[argc-1];

	return 0;
}

static int set_rsp_remap_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "[--idN=XXX --idM=YYY ...] /dev/plinX\n");
}

/*
 * $ lin set leds
 */
static int set_leds_handler(struct context *ctx, int argc, char *argv[])
{
	unsigned long int v;
	char *endptr;

#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif
	if (argc == 1)
		return set_leds_usage(ctx, NULL);
	else if (argc != 3)
		return set_leds_usage(ctx, "wrong number of arguments\n");

	v = strtoul(argv[1], &endptr, 0);
	if (*endptr) {
		if (!strcasecmp(argv[1], "on"))
			ctx->u.leds_state.on_off = 1;
		else if (!strcasecmp(argv[1], "off"))
			ctx->u.leds_state.on_off = 0;
		else
			return set_leds_usage(ctx,
					      "Unknown value \"%s\" for leds\n",
					      argv[1]);
	} else {
		ctx->u.leds_state.on_off = (u8 )v;
	}

	ctx->cmd = PLIOSETLEDSTATE;
	ctx->dev_name = argv[argc-1];

	return 0;
}

static int set_leds_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "on|off /dev/plinX\n");
}

/*
 * $ lin get 
 */
DECLARE_CMD(get_frm_entry);
DECLARE_CMD(get_sfe);
DECLARE_CMD(get_pfe);
DECLARE_CMD(get_id_filter);
DECLARE_CMD(get_id_string);
DECLARE_CMD(get_id_num);
DECLARE_CMD(get_fw_ver);
DECLARE_CMD(get_mode);
DECLARE_CMD(get_baudrate);
DECLARE_CMD(get_status);
DECLARE_CMD(get_slot_count);
DECLARE_CMD(get_schd_slot);
DECLARE_CMD(get_rsp_remap);
#ifndef NO_PRIVATE
DECLARE_CMD(get_private);
#endif
static struct cmd get_cmds[] = {
	__CMD_FUA(frm-entry, get_frm_entry, "fe",
"Get one/all frame(s) from the device internal frames table",
"display frames from the device table"),
	__CMD_FUA(sub-frm-entry, get_sfe, "sfe",
"Get subscribe frames only from the device internal frames table",
"display only subscriber frames"),
	__CMD_FUA(pub-frm-entry, get_pfe, "pfe",
"Get publish frames only from the device internal frames table",
"display only publisher frames"),
	__CMD_FUA(id-filter, get_id_filter, "if",
"Get the USB IDs filter currently used by the given device",
"get the current IDs filter mask"),
	__CMD_FUA(ident-string, get_id_string, "is",
"Get the user defined 24 bytes string stored in the given device memory",
"get the device user defined ident string"),
	__CMD_FUA(ident-num, get_id_num, "in",
"Get the user defined 32 bit value stored in the given device memory",
"get the device user defined 32 bit value"),
	__CMD_FUA(fw-ver, get_fw_ver, "fw",
"Get the version of the firmware running in the given device",
"display device FW version"),
	__CMD_FUA(mode, get_mode, NULL,
"Get the current mode in which the given device operates",
"get current device mode"),
	__CMD_FUA(baudrate, get_baudrate, "br",
"Get the current baudrate at which the given device operates",
"get the current device baudrate"),
	__CMD_FUA(status, get_status, "st",
"Display the current status of the given LIN device",
"display the device status"),
	__CMD_FUA(slot-count, get_slot_count, "sc",
"Get the number of slots for a specific schedule of the given LIN device",
"get the slot count of a schedule entry"),
	__CMD_FUA(schd-slot, get_schd_slot, "sl",
"Get a single slot from a specific schedule of the given LIN device",
"get a slot from a schedule entry"),
	__CMD_FUA(rsp-remap, get_rsp_remap, "rr",
"Get the response remap used for event frames in slave mode",
"display the response remap (slave mode)"),
#ifndef NO_PRIVATE
	__CMD_FUA(private, get_private, "gp",
"Get ioctl private command",
"reserved usage"),
#endif

	{ NULL, }
};

static int get_handler(struct context *ctx, int argc, char *argv[])
{
#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif
	return cmd_handler(ctx, get_cmds, argc, argv);
}

static int get_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	cmd_usage(ctx, "FUNCTION [OPTIONS] /dev/plinX\n");
	return subcmds_usage(ctx, "FUNCTION", get_cmds);
}

/*
 * $ lin get frm-entry
 */
DECLARE_CMD_NOHELP(get_frm_entry_set);
DECLARE_CMD_NOHELP(get_frm_entry_table);
static struct cmd get_frm_entry_opts[] = {
	__CMD_F(--set-fmt, get_frm_entry_set, "-s",
		"display frame in \"lin set\" cmd format"),
	__CMD_F(--table-fmt, get_frm_entry_table, "-t",
		"display frame in a table fmt (default)"),
	{ NULL, }
};

static int get_frm_entry_set_rsp(struct context *ctx, int c)
{
	unsigned short frm_flags = le16toh(ctx->u.frm_entry.flags);
	int frm_auto_len = 0;

	/* ignore disabled frames */
	if (ctx->u.frm_entry.direction == PLIN_FRM_DIR_DISABLED)
		return 0;

	switch (ctx->u.frm_entry.direction) {
	case PLIN_FRM_DIR_PUBLISHER:
		printf("pub-frm-entry ");
		break;
	case PLIN_FRM_DIR_SUBSCRIBER_AUTO_LEN:
		frm_auto_len++;

		/* fall through */
	case PLIN_FRM_DIR_SUBSCRIBER:
		printf("sub-frm-entry ");
		break;

	default:
		fprintf(stderr, "Direction %d not handled in this format\n",
			ctx->u.frm_entry.direction);
		return 1;
	}

	printf("0x%02X ", ctx->u.frm_entry.id);

	switch (ctx->u.frm_entry.checksum) {
	case PLIN_FRM_CST_CLASSIC:
		printf("--cs-classic ");
		break;

	case PLIN_FRM_CST_ENHANCED:
		printf("--cs-enhanced ");
		break;

	case PLIN_FRM_CST_AUTO:
		printf("--cs-auto ");
		break;

	default:
		fprintf(stderr, "Checksum type %d not handled in this format\n",
			ctx->u.frm_entry.checksum);
		return 2;
	}

	if (frm_flags & PLIN_FRM_FLG_RSP_ENABLE)
		printf("--rsp-enable ");

	if (frm_flags & PLIN_FRM_FLG_SINGLE_SHOT)
		printf("--sng-shot ");

	if (frm_flags & PLIN_FRM_FLG_IGNORE_DATA)
		printf("--ign-data ");

	if (frm_auto_len)
		printf("--auto-len ");

	printf("--len %u\n", ctx->u.frm_entry.len);

	return 0;
}

static int get_frm_entry_table_rsp(struct context *ctx, int c)
{
	int i;

	if (!ctx->cmd_ioctl_user_data)
		printf("ID (hex)  l D C FLGS DATA\n---------+-+-+-+----+-----------------------\n");

	ctx->cmd_ioctl_user_data++;

	/* if running "all" then ignore disabled frames */
	if ((ctx->cmd_ioctl_loop_max > ctx->cmd_ioctl_loop_min) &&
		(ctx->u.frm_entry.direction == PLIN_FRM_DIR_DISABLED))
		return 0;

	printf("%02u  (%02X)  %u %c %c %04x",
	       ctx->u.frm_entry.id, ctx->u.frm_entry.id,
	       ctx->u.frm_entry.len,
	       dir_name[ctx->u.frm_entry.direction].abbrv,
	       cs_name[ctx->u.frm_entry.checksum].abbrv,
	       le16toh(ctx->u.frm_entry.flags));

	for (i = 0; i < ctx->u.frm_entry.len; i++)
		printf(" %02x", ctx->u.frm_entry.d[i]);

	printf("\n");

	return 0;
}

static int get_frm_entry_set_handler(struct context *ctx, int argc,
				     char *argv[])
{
	ctx->rsp_handler = get_frm_entry_set_rsp;
	return 0;
}

static int get_frm_entry_table_handler(struct context *ctx, int argc,
				       char *argv[])
{
	ctx->rsp_handler = get_frm_entry_table_rsp;
	return 0;
}

static int get_frm_entry_handler(struct context *ctx, int argc, char *argv[])
{
	struct cmd *opt;

#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif

	if (argc == 1)
		return get_frm_entry_usage(ctx, NULL);

	ctx->cmd = PLIOGETFRMENTRY;
	ctx->rsp_handler = get_frm_entry_table_rsp;

	switch (argc) {

	case 4:
		opt = search_cmd_by_name(argv[2], get_frm_entry_opts);
		if (!opt)
			return get_frm_entry_usage(ctx,
					"\"%s\" invalid option\n", argv[2]);

		opt->handler(ctx, argc-2, argv+2);

		/* fall through */
	case 3:
		break;

	default:
		return get_frm_entry_usage(ctx, "wrong number of arguments\n");
	}

	memset(&ctx->u.frm_entry, '\0', sizeof(ctx->u.frm_entry));

	/* Frame ID (or "all") */
	if (!strcmp(argv[1], "all")) {
		ctx->cmd_ioctl_loop_min = PLIN_FRM_ID_MIN;
		ctx->cmd_ioctl_loop_max = PLIN_FRM_ID_MAX;
		ctx->cmd_ioctl_pre = do_pre_frm_entry;
	} else if ((strtobs(argv[1], 1, &ctx->u.frm_entry.id) != 1) ||
				(ctx->u.frm_entry.id > PLIN_FRM_ID_MAX))
		return get_frm_entry_usage(ctx, "ID out of range [%u..%u]\n",
					   PLIN_FRM_ID_MIN,
					   PLIN_FRM_ID_MAX);

	ctx->dev_name = argv[argc-1];

	return 0;
}

static int get_frm_entry_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	cmd_usage(ctx, "ID|all [OPTIONS] /dev/plinX\n");
	return opts_usage(ctx, "OPTIONS", get_frm_entry_opts);
}

/*
 * $ lin get sub-frm-entry
 */
static int get_sfe_rsp(struct context *ctx, int c)
{
	switch (ctx->u.frm_entry.direction) {
	case PLIN_FRM_DIR_SUBSCRIBER:
	case PLIN_FRM_DIR_SUBSCRIBER_AUTO_LEN:
		return get_frm_entry_table_rsp(ctx, c);
		break;
	}

	return 0;
}

static int get_sfe_handler(struct context *ctx, int argc, char *argv[])
{
#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif

	if (argc == 1)
		return get_sfe_usage(ctx, NULL);
	else if (argc != 2)
		return get_sfe_usage(ctx, "wrong number of arguments\n");

	ctx->cmd_ioctl_loop_min = PLIN_FRM_ID_MIN;
	ctx->cmd_ioctl_loop_max = PLIN_FRM_ID_MAX;
	ctx->cmd_ioctl_pre = do_pre_frm_entry;

	ctx->cmd = PLIOGETFRMENTRY;
	ctx->rsp_handler = get_sfe_rsp;
	ctx->dev_name = argv[argc-1];

	return 0;
}

static int get_sfe_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "/dev/plinX\n");
}

/*
 * $ lin get pub-frm-entry
 */
static int get_pfe_rsp(struct context *ctx, int c)
{
	if (ctx->u.frm_entry.direction == PLIN_FRM_DIR_PUBLISHER)
		return get_frm_entry_table_rsp(ctx, c);

	return 0;
}

static int get_pfe_handler(struct context *ctx, int argc, char *argv[])
{
#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif

	if (argc == 1)
		return get_pfe_usage(ctx, NULL);
	else if (argc != 2)
		return get_pfe_usage(ctx, "wrong number of arguments\n");

	ctx->cmd_ioctl_loop_min = PLIN_FRM_ID_MIN;
	ctx->cmd_ioctl_loop_max = PLIN_FRM_ID_MAX;
	ctx->cmd_ioctl_pre = do_pre_frm_entry;

	ctx->cmd = PLIOGETFRMENTRY;
	ctx->rsp_handler = get_pfe_rsp;
	ctx->dev_name = argv[argc-1];

	return 0;
}

static int get_pfe_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "/dev/plinX\n");
}

/*
 * $ lin get id-filter
 */
static int get_id_filter_rsp(struct context *ctx, int c)
{
	int i;
	u8 *pb = (u8 *)ctx->u.id_filter.id_mask;

	for (i = 0; i < 8; i++, pb++)
		printf("0x%02x%c", *pb, (i < 7) ? ':' : '\n');

	return 0;
}

static int get_id_filter_handler(struct context *ctx, int argc, char *argv[])
{
#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif
	if (argc == 1)
		return get_fw_ver_usage(ctx, NULL);
	else if (argc != 2)
		return get_fw_ver_usage(ctx, "wrong number of arguments\n");

	ctx->cmd = PLIOGETIDFILTER;
	ctx->rsp_handler = get_id_filter_rsp;
	ctx->dev_name = argv[argc-1];

	return 0;
}

static int get_id_filter_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "/dev/plinX\n");
}

/*
 * $ lin get fw-ver
 */
static int get_fw_ver_rsp(struct context *ctx, int c)
{
	printf("%u.%u.%u\n",
		ctx->u.fw_ver.major,
		ctx->u.fw_ver.minor,
		ctx->u.fw_ver.sub);
	return 0;
}

static int get_fw_ver_handler(struct context *ctx, int argc, char *argv[])
{
#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif
	if (argc == 1)
		return get_fw_ver_usage(ctx, NULL);
	else if (argc != 2)
		return get_fw_ver_usage(ctx, "wrong number of arguments\n");

	ctx->cmd = PLIOGETFWVER;
	ctx->rsp_handler = get_fw_ver_rsp;
	ctx->dev_name = argv[argc-1];

	return 0;
}

static int get_fw_ver_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "/dev/plinX\n");
}

/*
 * $ lin get ident-string
 */
static int get_id_string_rsp(struct context *ctx, int c)
{
	int i;

	for (i = 0; i < sizeof(ctx->u.ident_str.user_str); i++) {

		/* consider C strings */
		if (!ctx->u.ident_str.user_str[i] ||
				ctx->u.ident_str.user_str[i] == 0xff)
			break;

		putchar(isprint(ctx->u.ident_str.user_str[i]) ?
			ctx->u.ident_str.user_str[i] : '.');
	}

	putchar('\n');

	return 0;
}

static int get_id_string_handler(struct context *ctx, int argc, char *argv[])
{
#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif
	if (argc == 1)
		return get_id_string_usage(ctx, NULL);
	else if (argc != 2)
		return get_id_string_usage(ctx, "wrong number of arguments\n");

	ctx->cmd = PLIOGETIDSTR;
	ctx->rsp_handler = get_id_string_rsp;
	ctx->dev_name = argv[argc-1];

	return 0;
}

static int get_id_string_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "/dev/plinX\n");
}

/*
 * $ lin get ident-num
 */
static int get_id_num_rsp(struct context *ctx, int c)
{
	const u32 user_id = le32toh(ctx->u.ident_str.user_id);

	if (user_id != 0xffffffff)
		printf("%u", user_id);

	putchar('\n');

	return 0;
}

static int get_id_num_handler(struct context *ctx, int argc, char *argv[])
{
#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif
	if (argc == 1)
		return get_id_num_usage(ctx, NULL);
	else if (argc != 2)
		return get_id_num_usage(ctx, "wrong number of arguments\n");

	ctx->cmd = PLIOGETUSERID;
	ctx->rsp_handler = get_id_num_rsp;
	ctx->dev_name = argv[argc-1];

	return 0;
}

static int get_id_num_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "/dev/plinX\n");
}

/*
 * $ lin get mode
 */
DECLARE_CMD_NOHELP(get_mode_str);
DECLARE_CMD_NOHELP(get_mode_dec);
static struct cmd get_mode_opts[] = {
	__CMD_F(--string, get_mode_str, "-s", "display mode in ascii"),
	__CMD_F(--dec, get_mode_dec, "-d", "display mode in decimal (default)"),

	{ NULL, }
};

static int get_mode_str_handler(struct context *ctx, int argc, char *argv[])
{
	ctx->cmd_ioctl_user_data = 1;
	return 0;
}

static int get_mode_dec_handler(struct context *ctx, int argc, char *argv[])
{
	ctx->cmd_ioctl_user_data = 0;
	return 0;
}

static int get_mode_rsp(struct context *ctx, int c)
{
	if (ctx->cmd_ioctl_user_data == 1 &&
			(ctx->u.get_mode.mode < ARRAY_SIZE(mode_name)))
		printf("%s\n", mode_name[ctx->u.get_mode.mode].str);
	else
		printf("%u\n", ctx->u.get_mode.mode);

	return 0;
}

static int get_mode_handler(struct context *ctx, int argc, char *argv[])
{
	struct cmd *opt;

#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif

	ctx->cmd_ioctl_user_data = 0;
	switch (argc) {
	case 3:
		opt = search_cmd_by_name(argv[1], get_mode_opts);
		if (!opt)
			return get_mode_usage(ctx,
					"\"%s\" invalid option\n", argv[1]);

		opt->handler(ctx, argc-2, argv+2);

		/* fall through */
	case 2:
		break;
	case 1:
		return get_mode_usage(ctx, NULL);

	default:
		return get_mode_usage(ctx, "wrong number of arguments\n");
	}

	ctx->cmd = PLIOGETMODE;
	ctx->rsp_handler = get_mode_rsp;
	ctx->dev_name = argv[argc-1];

	return 0;
}

static int get_mode_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	cmd_usage(ctx, "[OPTIONS] /dev/plinX\n");
	return opts_usage(ctx, "OPTIONS", get_mode_opts);
}

/*
 * $ lin get baudrate
 */
static int get_baudrate_rsp(struct context *ctx, int c)
{
	printf("%u\n", le16toh(ctx->u.get_baudrate.baudrate));
}

static int get_baudrate_handler(struct context *ctx, int argc, char *argv[])
{
#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif
	if (argc == 1)
		return get_mode_usage(ctx, NULL);
	else if (argc != 2)
		return get_mode_usage(ctx, "wrong number of arguments\n");

	ctx->cmd = PLIOGETBAUDRATE;
	ctx->rsp_handler = get_baudrate_rsp;
	ctx->dev_name = argv[argc-1];

	return 0;
}

static int get_baudrate_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "/dev/plinX\n");
}

/*
 * $ lin get status
 */
static int get_status_rsp(struct context *ctx, int c)
{
	printf("mode=%s (%u)\n"
	       "tx_qfree=%u\n"
	       "schd_pool_free=%u\n"
	       "baudrate=%u\n"
	       "usb_rx_ovrn=%u\n"
	       "usb_filter=%016llxh\n"
	       "bus_state=%u\n",
	       mode_name[ctx->u.get_status.mode].str,
	       ctx->u.get_status.mode,
	       ctx->u.get_status.tx_qfree,
	       le16toh(ctx->u.get_status.schd_poolfree),
	       le16toh(ctx->u.get_status.baudrate),
	       le16toh(ctx->u.get_status.usb_rx_ovr),
	       (long long )le64toh(ctx->u.get_status.usb_filter),
	       ctx->u.get_status.bus_state);

	return 0;
}

static int get_status_handler(struct context *ctx, int argc, char *argv[])
{
#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif
	if (argc == 1)
		return get_status_usage(ctx, NULL);
	else if (argc != 2)
		return get_status_usage(ctx, "wrong number of arguments\n");

	ctx->cmd = PLIOGETSTATUS;
	ctx->rsp_handler = get_status_rsp;
	ctx->dev_name = argv[argc-1];

	return 0;
}

static int get_status_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "/dev/plinX\n");
}

/*
 * $ lin get slot-count
 */
static int do_pre_schedule(struct context *ctx)
{
	ctx->u.get_slot_count.schedule = (u8 )ctx->cmd_ioctl_loop;
	return 0;
}

static int get_slot_count_rsp(struct context *ctx, int c)
{
	if (ctx->cmd_ioctl_loop_min < ctx->cmd_ioctl_loop_max) {
		if (!c)
			printf("SCHD count\n----------\n");

		printf("%4u ", ctx->u.get_slot_count.schedule);
	}
	
	printf("%5u\n", le16toh(ctx->u.get_slot_count.count));
	return 0;
}

static int get_slot_count_handler(struct context *ctx, int argc, char *argv[])
{
#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif
	if (argc == 1)
		return get_slot_count_usage(ctx, NULL);
	else if (argc != 3)
		return get_slot_count_usage(ctx, "wrong number of arguments\n");

	/* SCHEDULE (or "all") */
	if (!strcmp(argv[1], "all")) {
		ctx->cmd_ioctl_loop_min = PLIN_SCH_IDX_MIN;
		ctx->cmd_ioctl_loop_max = PLIN_SCH_IDX_MAX;
		ctx->cmd_ioctl_pre = do_pre_schedule;
	} else if ((strtobs(argv[1], 1, &ctx->u.get_slot_count.schedule) != 1)
		|| (ctx->u.get_slot_count.schedule > PLIN_SCH_IDX_MAX))
		return get_slot_count_usage(ctx,
					   "SCHEDULE out of range [%u..%u]\n",
					   PLIN_SCH_IDX_MIN,
					   PLIN_SCH_IDX_MAX);

	ctx->cmd = PLIOGETSLOTSCNT;
	ctx->rsp_handler = get_slot_count_rsp;
	ctx->dev_name = argv[argc-1];

	return 0;
}

static int get_slot_count_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "SCHEDULE[0..7]|all /dev/plinX\n");
}

/*
 * $ get schd-slot
 */
static int get_schd_slot_rsp(struct context *ctx, int c)
{
	int i, l;

	if (!c) {
		if (ctx->cmd_ioctl_loop_min < ctx->cmd_ioctl_loop_max)
			printf("SCHD ");
 		printf("SLOT T   c/r delay   handle I0 I1 I2 I3 I4 I5 I6 I7\n");
		if (ctx->cmd_ioctl_loop_min < ctx->cmd_ioctl_loop_max)
			printf("-----");
		printf("---------------------------------------------------\n");
	}

	if (ctx->u.get_schd_slot.err) {
		if (ctx->cmd_ioctl_loop_min < ctx->cmd_ioctl_loop_max)
			printf("%4u ", ctx->u.get_schd_slot.schedule);
		printf("%4u Error: no schedule present\n",
			ctx->u.get_schd_slot.slot_idx);
		return 0;
	}

	if (ctx->cmd_ioctl_loop_min < ctx->cmd_ioctl_loop_max)
		printf("%4u ", ctx->u.get_schd_slot.schedule);

	printf("%4u %c %5u %5u %08x ",
		ctx->u.get_schd_slot.slot_idx,
		slot_type_name[ctx->u.get_schd_slot.type].abbrv,
		ctx->u.get_schd_slot.count_resolve,
		le16toh(ctx->u.get_schd_slot.delay),
		ctx->u.get_schd_slot.handle);

	/* IDO ; ID1 - ID7 only for sporadic frames */
	l = (ctx->u.get_schd_slot.type == PLIN_USB_SLOT_SPORADIC) ?
				PLIN_USB_SLOT_NB_MAX : 1;
	for (i = 0; i < l; i++)
		printf("%02x ", ctx->u.get_schd_slot.id[i]);

	putchar('\n');
	return 0;
}

/*
 * static int do_pre_schd_slot(struct context *ctx)
 *
 * This function should loop on all slots for one/all schedules.
 * Therefore, it controls the loop index by itself.
 */
static int do_pre_schd_slot(struct context *ctx)
{
#ifdef DEBUG_TRACE
	printf("%s(loop=%d) user_data=%d schd=%d slot=%d\n",
		__func__, ctx->cmd_ioctl_loop, ctx->cmd_ioctl_user_data,
		ctx->u.get_schd_slot.schedule, ctx->u.get_schd_slot.slot_idx);
#endif

	/* first call: get the slot count for this schedule */
	if (!ctx->cmd_ioctl_user_data) {

		/* hook to read the current slot count for the given SCHEDULE */
		struct plin_usb_get_slot_count gsc = {
			.schedule = ctx->u.get_schd_slot.schedule,
		};

		int err = ioctl(ctx->lin_fd, PLIOGETSLOTSCNT, &gsc);
		if (err) {
			fprintf(stderr,
				"Failed to get schedule #%u slot count "
				"(errno %d)\n",
				ctx->cmd_ioctl_loop, errno);

			/* go on next SCHEDULE (or ends) */
			return 1;
		}

		ctx->cmd_ioctl_user_data = le16toh(gsc.count);
		ctx->u.get_schd_slot.slot_idx = 0;

	/* this schedule defines slots: go on next */
	} else {
		ctx->u.get_schd_slot.slot_idx++;
	}

	/* if no (more) slot for that schedule, go on next (or ends) */
	if (ctx->u.get_schd_slot.slot_idx >= ctx->cmd_ioctl_user_data) {
		ctx->cmd_ioctl_user_data = 0;
		ctx->u.get_schd_slot.schedule++;
		return 1;
	}

	/* stay on the same SCHEDULE and do the ioctl() */
	ctx->cmd_ioctl_loop--;

	return 0;
}

static int get_schd_slot_handler(struct context *ctx, int argc, char *argv[])
{
#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif
	if (argc == 1)
		return get_schd_slot_usage(ctx, NULL);
	else if (argc != 3)
		return get_schd_slot_usage(ctx, "wrong number of arguments\n");

	/* SCHEDULE (or "all") */
	if (!strcmp(argv[1], "all")) {
		ctx->cmd_ioctl_loop_min = PLIN_SCH_IDX_MIN;
		ctx->cmd_ioctl_loop_max = PLIN_SCH_IDX_MAX;

		/* be sure to start from first SCHEDULE */
		ctx->u.get_schd_slot.schedule = 0;

	} else if ((strtobs(argv[1], 1, &ctx->u.get_schd_slot.schedule) != 1)
		|| (ctx->u.get_schd_slot.schedule > PLIN_SCH_IDX_MAX))
		return get_schd_slot_usage(ctx,
					   "SCHEDULE out of range [%u..%u]\n",
					   PLIN_SCH_IDX_MIN,
					   PLIN_SCH_IDX_MAX);

	/* user_data will contain the slot count for the/each SCHEDULE */
	ctx->cmd_ioctl_user_data = 0;

	/* setup pre-ioctl to control loop on slots, even for a single
	 * SCHEDULE, to control the loop over the slots
	 */
	ctx->cmd_ioctl_pre = do_pre_schd_slot;

	ctx->cmd = PLIOGETSCHDSLOT;
	ctx->rsp_handler = get_schd_slot_rsp;
	ctx->dev_name = argv[argc-1];

	return 0;
}

static int get_schd_slot_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "SCHEDULE[0..7]|all /dev/plinX\n");
}

/*
 * $ lin get rsp-remap
 *
 * (slave mode only)
 */
static int get_rsp_remap_rsp(struct context *ctx, int c)
{
	int i, j;

	printf("ID  ");
	for (i = 0; i < 8; i++)
		printf("%2u ", i);
	putchar('\n');
	printf("---+--+--+--+--+--+--+--+--\n");
	for (i = 0; i < 8; i++) {
		printf("%2d  ", i*8);
		for (j = 0; j < 8; j++)
			printf("%02x ", ctx->u.rsp_remap.id[i*8+j]);
		putchar('\n');
	}

	return 0;
}

static int get_rsp_remap_handler(struct context *ctx, int argc, char *argv[])
{
#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif
	if (argc == 1)
		return get_rsp_remap_usage(ctx, NULL);
	else if (argc != 2)
		return get_rsp_remap_usage(ctx, "wrong number of arguments\n");

	memset(&ctx->u.rsp_remap, '\0', sizeof(ctx->u.rsp_remap));

	ctx->u.rsp_remap.set_get = PLIN_USB_RSP_REMAP_GET;
	ctx->cmd = PLIOSETGETRSPMAP;
	ctx->rsp_handler = get_rsp_remap_rsp;
	ctx->dev_name = argv[argc-1];

	return 0;
}

static int get_rsp_remap_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "/dev/plinX\n");
}

#ifndef NO_PRIVATE
/*
 * void dump_mem(const char *prompt, void *p, int l)
 */
static void dump_mem(const char *prompt, void *p, int l)
{
	int i;

	printf("dumping %s from %p (%u bytes):\n",
		(prompt) ? prompt : "memory", p, l);

	for (i = 0; i < l; i++) {
		if (!(i & 0xf))
			printf("%04d ", i);
		printf("%02x ", *((unsigned char *)p + i));
		if ((i & 0xf) == 15)
			putchar('\n');
		else if ((i & 0x7) == 7)
			printf("- ");
	}
	if (i & 0xf)
		putchar('\n');
}

/*
 * $ lin get private
 */
static int get_private_rsp(struct context *ctx, int c)
{
	dump_mem("private command result",
		 ctx->u.plin_private_data, ctx->cmd_ioctl_user_data);

	return 0;
}

static int get_private_handler(struct context *ctx, int argc, char *argv[])
{
	char *endptr;
	unsigned long cmd, size, loop;

#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif
	switch (argc) {
	case 5:
		loop = strtoul(argv[3], &endptr, 0);
		if (*endptr)
			return get_private_usage(ctx,
					"\"%s\" invalid loop value\n",
					 argv[3]);

		ctx->cmd_ioctl_loop_min = 1;
		ctx->cmd_ioctl_loop_max = loop;

		/* fall through */
	case 4:
		size = strtoul(argv[2], &endptr, 0);
		if (*endptr)
			return get_private_usage(ctx,
					"\"%s\" invalid command size\n",
					 argv[2]);

		if (size > sizeof(ctx->u.plin_private_data))
			return get_private_usage(ctx,
					"SIZE out of range [0..%lu]\n",
					sizeof(ctx->u.plin_private_data));

		cmd = strtoul(argv[1], &endptr, 0);
		if (*endptr)
			return get_private_usage(ctx,
					"\"%s\" invalid command number\n",
					 argv[1]);

		break;
	case 1:
		return get_private_usage(ctx, NULL);
	default:
		return get_private_usage(ctx, "wrong number of arguments\n");
	}

	ctx->cmd_ioctl_user_data = size;

	ctx->cmd = _IOC(_IOC_READ, 'p', cmd, size);
	ctx->rsp_handler = get_private_rsp;
	ctx->dev_name = argv[argc-1];

	return 0;
}

static int get_private_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "CMD SIZE /dev/plinX\n");
}
#endif

/*
 * $ lin start
 */
DECLARE_CMD(start_master);
DECLARE_CMD(start_slave);
DECLARE_CMD(start_auto_baud);
DECLARE_CMD(start_keep_alive);
DECLARE_CMD(start_schd);
static struct cmd start_cmds[] = {
	__CMD_FUA(master, start_master, "m",
"Initialize the serial interface and set the given device in master mode",
"set master mode at the given baudrate" ),
	__CMD_FUA(slave, start_slave, "s",
"Initialize the serial interface and set the given device in slave mode",
"set slave mode at the given baudrate" ),
	__CMD_FUA(keep-alive, start_keep_alive, "ka",
"Start writing a keep alive frame to prevent from bus sleep. This frame is halted when the scheduler is running",
"start the keap-alive frame process"),
	__CMD_FUA(auto-baud, start_auto_baud, "ab",
"Start baudrate detection process (bus must not be initialized)",
"start baudrate automatic detection"),
	__CMD_FUA(schedule, start_schd, "schd",
"Start the scheduler at the first slot of the given schedule (master only)",
"start the scheduler from a schedule"),

	{ NULL, }
};

static int start_handler(struct context *ctx, int argc, char *argv[])
{
	return cmd_handler(ctx, start_cmds, argc, argv);
}

static int start_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	cmd_usage(ctx, "FUNCTION [OPTIONS] /dev/plinX\n");
	return subcmds_usage(ctx, "FUNCTION", start_cmds);
}

/*
 * $ lin start master
 */
static int start_master_handler(struct context *ctx, int argc, char *argv[])
{
	int l;

#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif
	if (argc == 1)
		return start_master_usage(ctx, NULL);
	else if (argc != 3)
		return start_master_usage(ctx, "wrong number of arguments\n");

	if (strtohunit(argv[1], &ctx->u.init_hw.baudrate, "k"))
		return start_master_usage(ctx, "invalid baudrate value\n");

	ctx->cmd = PLIOHWINIT;
	ctx->u.init_hw.mode = PLIN_MODE_MASTER;
	
	ctx->dev_name = argv[argc-1];

	return 0;
}

static int start_master_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "BAUDRATE[1000..20000] /dev/plinX\n");
}

/*
 * $ lin start slave
 */
static int start_slave_handler(struct context *ctx, int argc, char *argv[])
{
	int l;

#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif
	if (argc == 1)
		return start_slave_usage(ctx, NULL);
	else if (argc != 3)
		return start_slave_usage(ctx, "wrong number of arguments\n");

	if (strtohunit(argv[1], &ctx->u.init_hw.baudrate, "k"))
		return start_slave_usage(ctx, "invalid baudrate value\n");

	ctx->cmd = PLIOHWINIT;
	ctx->u.init_hw.mode = PLIN_MODE_SLAVE;
	ctx->dev_name = argv[argc-1];

	return 0;
}

static int start_slave_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "BAUDRATE[1000..20000] /dev/plinX\n");
}

/*
 * $ lin start keep-alive
 */
static int start_keep_alive_rsp(struct context *ctx, int c)
{
	//printf("%u\n", le16toh(ctx->u.keep_alive.err));
	return le16toh(ctx->u.keep_alive.err);
}

static int start_keep_alive_handler(struct context *ctx, int argc, char *argv[])
{
#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif

	ctx->u.keep_alive.period_ms = htole16(1000);

	switch (argc) {
	case 4:
		if (strtohunit(argv[2], &ctx->u.keep_alive.period_ms, "s"))
			return start_keep_alive_usage(ctx,
						"invalid period value\n");

		/* fall through */
	case 3:
		if ((strtobs(argv[1], 1, &ctx->u.keep_alive.id) != 1) ||
			(ctx->u.keep_alive.id > PLIN_FRM_ID_MAX))
			return start_keep_alive_usage(ctx,
						"invalid frame ID [%u..%u]\n",
						PLIN_FRM_ID_MIN,
						PLIN_FRM_ID_MAX);
		ctx->dev_name = argv[argc-1];
		break;

	case 1:
		return start_keep_alive_usage(ctx, NULL);

	default:
		return start_keep_alive_usage(ctx, "wrong number of arguments\n");
	}

	ctx->cmd = PLIOSTARTHB;
	ctx->rsp_handler = start_keep_alive_rsp;

	return 0;
}

static int start_keep_alive_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "ID[0..63] [PAUSE_MS] /dev/plinX\n");
}

/*
 * $ lin start auto-baud
 */
static int start_auto_baud_rsp(struct context *ctx, int c)
{
	printf("%u\n", le16toh(ctx->u.auto_baud.err));
	return 0;
}

static int start_auto_baud_handler(struct context *ctx, int argc, char *argv[])
{
#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif

	memset(&ctx->u.auto_baud, '\0', sizeof(ctx->u.auto_baud));

	ctx->u.auto_baud.timeout = htole16(4000);

	switch (argc) {
	case 3:
		if (strtohunit(argv[1], &ctx->u.auto_baud.timeout, "s"))
			return start_auto_baud_usage(ctx,
						"invalid timeout value\n");
		/* fall through */
	case 2:
		break;

	case 1:
		return start_auto_baud_usage(ctx, NULL);
		
	default:
		return start_auto_baud_usage(ctx,
					     "wrong number of arguments\n");
	}

	ctx->cmd = PLIOSTARTAUTOBAUD;
	ctx->rsp_handler = start_auto_baud_rsp;
	ctx->dev_name = argv[argc-1];

	return 0;
}

static int start_auto_baud_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "[TIMEOUT_MS] /dev/plinX\n");
}

/*
 * $ lin start schedule
 */
static int do_display_schd_status_rsp(struct context *ctx, int c)
{
	if (ctx->cmd_ioctl_loop_min < ctx->cmd_ioctl_loop_max) {
		if (!c)
			printf("SCHD status\n-----------\n");

		printf("%4u ", ctx->u.del_schd.schedule);
	}

	if (ctx->u.del_schd.err)
		printf("%u\n", ctx->u.del_schd.err);

	return ctx->u.del_schd.err;
}

static int start_schd_handler(struct context *ctx, int argc, char *argv[])
{
#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif
	if (argc == 1)
		return start_schd_usage(ctx, NULL);
	else if (argc != 3)
		return start_schd_usage(ctx, "wrong number of arguments\n");

	/* SCHEDULE (or "all") */
	if (!strcmp(argv[1], "all")) {
		ctx->cmd_ioctl_loop_min = PLIN_SCH_IDX_MIN;
		ctx->cmd_ioctl_loop_max = PLIN_SCH_IDX_MAX;
		ctx->cmd_ioctl_pre = do_pre_schedule;
	} else if ((strtobs(argv[1], 1, &ctx->u.start_schd.schedule) != 1)
		|| (ctx->u.start_schd.schedule > PLIN_SCH_IDX_MAX))
		return start_schd_usage(ctx,
					"SCHEDULE out of range [%u..%u]\n",
					PLIN_SCH_IDX_MIN,
					PLIN_SCH_IDX_MAX);

	ctx->cmd = PLIOSTARTSCHD;
	ctx->dev_name = argv[argc-1];
	ctx->rsp_handler = do_display_schd_status_rsp;

	return 0;
}

static int start_schd_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "SCHEDULE[0..7]|all /dev/plinX\n");
}

/*
 * $ lin disconnect
 */
static int disconnect_handler(struct context *ctx, int argc, char *argv[])
{
#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif

	if (argc == 1)
		return disconnect_usage(ctx, NULL);
	else if (argc != 2)
		return disconnect_usage(ctx, "wrong number of arguments\n");

	ctx->u.init_hw.mode = PLIN_MODE_NONE;
	ctx->cmd = PLIOHWINIT;
	ctx->dev_name = argv[argc-1];

	return 0;
}

static int disconnect_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "/dev/plinX\n");
}

/*
 * $ lin wake-up
 */
static int wake_up_handler(struct context *ctx, int argc, char *argv[])
{
#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif

	if (argc == 1)
		return wake_up_usage(ctx, NULL);
	else if (argc != 2)
		return wake_up_usage(ctx, "wrong number of arguments\n");

	ctx->cmd = PLIOXMTWAKEUP;
	ctx->dev_name = argv[argc-1];

	return 0;
}

static int wake_up_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "/dev/plinX\n");
}

/*
 * $ lin suspend
 */
DECLARE_CMD(suspend_keep_alive);
DECLARE_CMD(suspend_schd);
static struct cmd suspend_cmds[] = {
	__CMD_FUA(keep-alive, suspend_keep_alive, "ka",
"Suspend writing keep alive frames",
"suspend keep-alive process"),
	__CMD_FUA(scheduler, suspend_schd, "schd",
"Suspend the scheduler. The position is internally saved so that the scheduler can be resumed at that point (see \"resume\" command)",
"suspend the scheduler"),
	{ NULL, }
};

static int suspend_handler(struct context *ctx, int argc, char *argv[])
{
	return cmd_handler(ctx, suspend_cmds, argc, argv);
}

static int suspend_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	cmd_usage(ctx, "FUNCTION [OPTIONS] /dev/plinX\n");
	return subcmds_usage(ctx, "FUNCTION", suspend_cmds);
}

/*
 * $ lin suspend keep-alive
 */
static int suspend_keep_alive_handler(struct context *ctx, int argc,
				      char *argv[])
{
#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif

	if (argc == 1)
		return suspend_keep_alive_usage(ctx, NULL);
	else if (argc != 2)
		return suspend_keep_alive_usage(ctx,
						"wrong number of arguments\n");

	ctx->cmd = PLIOPAUSEHB;
	ctx->dev_name = argv[argc-1];

	return 0;
}

static int suspend_keep_alive_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "/dev/plinX\n");
}

/*
 * $ lin suspend schedule
 */
static int suspend_schd_rsp(struct context *ctx, int c)
{
	if (ctx->u.suspend_schd.err)
		fprintf(stderr, "Error: scheduler not running\n");
	else
		printf("%u:%08x\n",
			ctx->u.suspend_schd.schedule,
			ctx->u.suspend_schd.handle);

	return 0;
}

static int suspend_schd_handler(struct context *ctx, int argc, char *argv[])
{
#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif
	if (argc == 1)
		return suspend_schd_usage(ctx, NULL);
	else if (argc != 2)
		return suspend_schd_usage(ctx, "wrong number of arguments\n");

	ctx->cmd = PLIOPAUSESCHD;
	ctx->dev_name = argv[argc-1];
	ctx->rsp_handler = suspend_schd_rsp;

	return 0;
}

static int suspend_schd_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "/dev/plinX\n");
}

/*
 * $ lin resume
 */
DECLARE_CMD(resume_keep_alive);
DECLARE_CMD(resume_schd);
static struct cmd resume_cmds[] = {
	__CMD_FUA(keep-alive, resume_keep_alive, "ka",
"Resume writing keep alive frames",
"resume writing keep alive frames"),
	__CMD_FUA(scheduler, resume_schd, "schd",
"Resume the scheduler at the point it has been suspended",
"resume the scheduler"),

	{ NULL, }
};

static int resume_handler(struct context *ctx, int argc, char *argv[])
{
	return cmd_handler(ctx, resume_cmds, argc, argv);
}

static int resume_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	cmd_usage(ctx, "FUNCTION [OPTIONS] /dev/plinX\n");
	return subcmds_usage(ctx, "FUNCTION", resume_cmds);
}

/*
 * $ lin resume keep-alive
 */
static int resume_keep_alive_handler(struct context *ctx, int argc,
				      char *argv[])
{
#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif

	if (argc == 1)
		return resume_keep_alive_usage(ctx, NULL);
	else if (argc != 2)
		return resume_keep_alive_usage(ctx,
						"wrong number of arguments\n");

	ctx->cmd = PLIORESUMEHB;
	ctx->dev_name = argv[argc-1];

	return 0;
}

static int resume_keep_alive_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "/dev/plinX\n");
}

/*
 * $ lin resume schedule
 */
static int resume_schd_rsp(struct context *ctx, int c)
{
	printf("%u\n", ctx->u.resume_schd.err);
	if (ctx->u.resume_schd.err)
		fprintf(stderr,
			"Error: not in master mode or no schedule started\n");
	return 0;
}

static int resume_schd_handler(struct context *ctx, int argc, char *argv[])
{
#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif
	if (argc == 1)
		return resume_schd_usage(ctx, NULL);
	else if (argc != 2)
		return resume_schd_usage(ctx, "wrong number of arguments\n");

	ctx->cmd = PLIORESUMESCHD;
	ctx->dev_name = argv[argc-1];
	ctx->rsp_handler = resume_schd_rsp;

	return 0;
}

static int resume_schd_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "/dev/plinX\n");
}

/*
 * $ lin add
 */
DECLARE_CMD(add_uss);
DECLARE_CMD(add_ess);
DECLARE_CMD(add_sss);
DECLARE_CMD(add_rss);
DECLARE_CMD(add_ass);
static struct cmd add_cmds[] = {
	__CMD_FUA(sub-frm-entry, set_sub_fe, "sfe",
"Configure a subscribe frame entry in the LIN device frames table.\n"
"This command does the same than the corresponding \"set\" command",
"define a subscribe frame entry"),
	__CMD_FUA(pub-frm-entry, set_pub_fe, "pfe",
"Configure a publish frame entry in the LIN device frames table.\n"
"This command does the same than the corresponding \"set\" command",
"define a publish frame entry"),
	__CMD_FUA(unc-schd-slot, add_uss, "sig",
"Add an unconditional slot (ID is sent) to a schedule",
"add an unconditionial slot to schedule"),
	__CMD_FUA(evt-schd-slot, add_ess, "ess",
"Add an event slot (ID is sent) to a schedule",
"add an event slot to a schedule"),
	__CMD_FUA(spo-schd-slot, add_sss, "sss",
"Add a sporadic slot (ID0 has the highest priority) to a schedule (master mode only)",
"add a sporadic slot to a schedule"),
	__CMD_FUA(req-schd-slot, add_rss, "rss",
"Add a master request slot (ID=60 is used) to a schedule",
"add a master request slot to a schedule"),
	__CMD_FUA(rsp-schd-slot, add_ass, "ass",
"Add a slave response slot (ID=61 is used) to a schedule",
"add a slave response slot to a schedule"),

	{ NULL, }
};

static int add_handler(struct context *ctx, int argc, char *argv[])
{
	return cmd_handler(ctx, add_cmds, argc, argv);
}

static int add_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	cmd_usage(ctx, "FUNCTION [OPTIONS] /dev/plinX\n");
	return subcmds_usage(ctx, "FUNCTION", add_cmds);
}

static int add_schd_slot_rsp(struct context *ctx, int c)
{
	if (!ctx->u.add_schd_slot.err)
		printf("%08x\n", ctx->u.add_schd_slot.handle);
	else
		fprintf(stderr, "Error %u\n", ctx->u.add_schd_slot.err);

	return 0;
}

/*
 * $ lin add unc-schd-slot
 *
 * Unconditional frame.
 *
 * These always carry signals and their identifiers are in the range 0 to 59
 * (0x00 to 0x3b). All subscribers of the unconditional frame shall receive thei
 * frame and make it available to the application (assuming no errors were
 * detected).
 *
 * (see https://en.wikipedia.org/wiki/Local_Interconnect_Network)
 */
static int add_uss_handler(struct context *ctx, int argc, char *argv[])
{
#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif

	if (argc == 1)
		return add_uss_usage(ctx, NULL);
	else if (argc != 5)
		return add_uss_usage(ctx, "wrong number of arguments\n");

	/* SCHEDULE */
	if ((strtobs(argv[1], 1, &ctx->u.add_schd_slot.schedule) != 1)
		|| (ctx->u.add_schd_slot.schedule > PLIN_SCH_IDX_MAX))
		return add_uss_usage(ctx, "SCHEDULE out of range [%u..%u]\n",
				     PLIN_SCH_IDX_MIN,
				     PLIN_SCH_IDX_MAX);

	/* DELAY_MS (to next entry/slot) */
	if (strtohunit(argv[2], &ctx->u.add_schd_slot.delay, "s"))
		return add_uss_usage(ctx, "invalid delay value\n");

	/* ID */
	if ((strtobs(argv[3], 1, &ctx->u.add_schd_slot.id[0]) != 1)
		|| (ctx->u.add_schd_slot.id[0] > PLIN_FRM_ID_MAX))
		return add_uss_usage(ctx, "ID out of range [%u..%u]\n",
				     PLIN_FRM_ID_UNC_MIN,
				     PLIN_FRM_ID_UNC_MAX);

	ctx->u.add_schd_slot.type = PLIN_USB_SLOT_UNCOND;
	ctx->u.add_schd_slot.count_resolve = 0;

	ctx->cmd = PLIOADDSCHDSLOT;
	ctx->dev_name = argv[argc-1];
	ctx->rsp_handler = add_schd_slot_rsp;

	return 0;
}

static int add_uss_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "SCHEDULE[0..7] DELAY_MS ID /dev/plinX\n");
}

/*
 * $ lin add evt-schd-slot
 *
 * Event-triggered frame
 *
 * The purpose of this is to increase the responsiveness of the LIN clusterxi
 * without assigning too much of the bus bandwidth to the polling of multiple
 * slave nodes with seldom occurring events. The first data byte of the carried
 * unconditional frame shall be equal to a protected identifier assigned to an
 * event-triggered frame. A slave shall reply with an associated unconditional
 * frame only if its data value has changed. If none of the slave tasks
 * responds to the header the rest of the frame slot is silent and the header is
 * ignored. If more than one slave task responds to the header in the same frame
 * slot a collision will occur, and the master has to resolve the collision by
 * requesting all associated unconditional frames before requesting the
 * event-triggered frame again.
 *
 * (see https://en.wikipedia.org/wiki/Local_Interconnect_Network)
 */
static int add_ess_handler(struct context *ctx, int argc, char *argv[])
{
#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif

	if (argc == 1)
		return add_ess_usage(ctx, NULL);
	else if (argc != 6)
		return add_ess_usage(ctx, "wrong number of arguments\n");

	/* SCHEDULE */
	if ((strtobs(argv[1], 1, &ctx->u.add_schd_slot.schedule) != 1)
		|| (ctx->u.add_schd_slot.schedule > PLIN_SCH_IDX_MAX))
		return add_ess_usage(ctx, "SCHEDULE out of range [%u..%u]\n",
				     PLIN_SCH_IDX_MIN,
				     PLIN_SCH_IDX_MAX);

	/* DELAY_MS (to next entry/slot) */
	if (strtohunit(argv[2], &ctx->u.add_schd_slot.delay, "s"))
		return add_ess_usage(ctx, "invalid delay value\n");

	/* ID */
	if ((strtobs(argv[3], 1, &ctx->u.add_schd_slot.id[0]) != 1)
		|| (ctx->u.add_schd_slot.id[0] > PLIN_FRM_ID_MAX))
		return add_ess_usage(ctx, "ID out of range [%u..%u]\n",
				     PLIN_FRM_ID_UNC_MIN,
				     PLIN_FRM_ID_UNC_MAX);

	/* RESOLVE schedule number on event slot (0..7) */
	if ((strtobs(argv[4], 1, &ctx->u.add_schd_slot.count_resolve) != 1)
		|| (ctx->u.add_schd_slot.id[0] > PLIN_FRM_ID_MAX))
		return add_ess_usage(ctx, "RESOLVE out of range [%u..%u]\n",
				     PLIN_SCH_IDX_MIN,
				     PLIN_SCH_IDX_MAX);

	ctx->u.add_schd_slot.type = PLIN_USB_SLOT_EVENT;

	ctx->cmd = PLIOADDSCHDSLOT;
	ctx->dev_name = argv[argc-1];
	ctx->rsp_handler = add_schd_slot_rsp;

	return 0;
}

static int add_ess_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "SCHEDULE[0..7] DELAY_MS ID RESOLVE /dev/plinX\n");
}

/*
 * $ lin add spo-schd-slot
 *
 * Sporadic frame.
 *
 * This frame is transmitted by the master as required, so a collision cannot
 * occur. The header of a sporadic frame shall only be sent in its associated
 * frame slot when the master task knows that a signal carried in the frame has
 * been updated. The publisher of the sporadic frame shall always provide the
 * response to the header.
 *
 * (see https://en.wikipedia.org/wiki/Local_Interconnect_Network)
 */
static int add_sss_handler(struct context *ctx, int argc, char *argv[])
{
	int i, n;

#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif

	if (argc == 1)
		return add_sss_usage(ctx, NULL);
	else if (argc != 5)
		return add_sss_usage(ctx, "wrong number of arguments\n");

	/* SCHEDULE */
	if ((strtobs(argv[1], 1, &ctx->u.add_schd_slot.schedule) != 1)
		|| (ctx->u.add_schd_slot.schedule > PLIN_SCH_IDX_MAX))
		return add_sss_usage(ctx, "SCHEDULE out of range [%u..%u]\n",
				     PLIN_SCH_IDX_MIN,
				     PLIN_SCH_IDX_MAX);

	/* DELAY_MS (to next entry/slot) */
	if (strtohunit(argv[2], &ctx->u.add_schd_slot.delay, "s"))
		return add_sss_usage(ctx, "invalid delay value\n");

	/* NUMBER_IDs during sporadic slot (1..8) */
	n = strtobs(argv[3], PLIN_USB_SLOT_NB_MAX, ctx->u.add_schd_slot.id);
	if (n < PLIN_USB_SLOT_NB_MIN)
		return add_sss_usage(ctx, "bad COUNT of IDs [%u..%u]\n",
				     PLIN_USB_SLOT_NB_MIN,
				     PLIN_USB_SLOT_NB_MAX);

	/* check ID values */
	for (i = 0; i < n; i++)
		if (ctx->u.add_schd_slot.id[i] > PLIN_FRM_ID_UNC_MAX)
			return add_sss_usage(ctx,
					     "ID%u out of range [%u..%u]\n",
					     ctx->u.add_schd_slot.id[i],
					     PLIN_FRM_ID_UNC_MIN,
					     PLIN_FRM_ID_UNC_MAX);

	ctx->u.add_schd_slot.count_resolve = (u8 )n;
	ctx->u.add_schd_slot.type = PLIN_USB_SLOT_SPORADIC;

	ctx->cmd = PLIOADDSCHDSLOT;
	ctx->dev_name = argv[argc-1];
	ctx->rsp_handler = add_schd_slot_rsp;

	return 0;
}

static int add_sss_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "SCHEDULE[0..7] DELAY_MS \"ID0 ID1...\" /dev/plinX\n");
}

/*
 * $ lin add req-schd-slot
 *
 * Diagnostic frame.
 *
 * These always carry diagnostic or configuration data and they always contain
 * eight data bytes. The identifier is either 60 (0x3C), called master request
 * frame, or 61(0x3D), called slave response frame. Before generating the header
 * of a diagnostic frame, the master task asks its diagnostic module if it
 * shall be sent or if the bus shall be silent. The slave tasks publish and
 * subscribe to the response according to their diagnostic module.
 *
 * (see https://en.wikipedia.org/wiki/Local_Interconnect_Network)
 */
static int add_diag_ss_handler(struct context *ctx, u8 type, 
			       int argc, char *argv[])
{
	/* get from history the cmd we come from */
	struct cmd *cmd = ctx->history[ctx->history_lvl-1];

	if (argc == 1)
		return cmd->usage(ctx, NULL);
	else if (argc != 4)
		return cmd->usage(ctx, "wrong number of arguments\n");

	/* SCHEDULE */
	if ((strtobs(argv[1], 1, &ctx->u.add_schd_slot.schedule) != 1)
		|| (ctx->u.add_schd_slot.schedule > PLIN_SCH_IDX_MAX))
		return cmd->usage(ctx, "SCHEDULE out of range [%u..%u]\n",
				     PLIN_SCH_IDX_MIN,
				     PLIN_SCH_IDX_MAX);

	/* DELAY_MS (to next entry/slot) */
	if (strtohunit(argv[2], &ctx->u.add_schd_slot.delay, "s"))
		return cmd->usage(ctx, "invalid delay value\n");

	ctx->u.add_schd_slot.type = type;

	/* debug only: IDs used in that two cases are implicit */
	ctx->u.add_schd_slot.id[0] = (type == PLIN_USB_SLOT_MASTER_REQ) ?
		PLIN_FRM_ID_DIAG_MASTER_REQ :
		PLIN_FRM_ID_DIAG_SLAVE_RSP;

	ctx->cmd = PLIOADDSCHDSLOT;
	ctx->dev_name = argv[argc-1];
	ctx->rsp_handler = add_schd_slot_rsp;

	return 0;
}

static int add_rss_handler(struct context *ctx, int argc, char *argv[])
{
	int n;

#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif

	return add_diag_ss_handler(ctx, PLIN_USB_SLOT_MASTER_REQ, argc, argv);
}

static int add_rss_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "SCHEDULE[0..7] DELAY_MS /dev/plinX\n");
}

/*
 * $ lin add rsp-schd-slot
 */
static int add_ass_handler(struct context *ctx, int argc, char *argv[])
{
	int n;

#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif

	return add_diag_ss_handler(ctx, PLIN_USB_SLOT_SLAVE_RSP, argc, argv);
}

static int add_ass_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "SCHEDULE[0..7] DELAY_MS /dev/plinX\n");
}

/*
 * $ lin del
 */
DECLARE_CMD(del_frm_entry);
DECLARE_CMD(del_schd);
static struct cmd del_cmds[] = {
	__CMD_FUA(frm-entry, del_frm_entry, "fe",
"Delete (disable) a frame entry from the LIN device frames table",
"disable a given frame entry"),
	__CMD_FUA(schedule, del_schd, "schd",
"Delete a schedule and its slots from the scheduler schedules table",
"delete a given schedule and its slots"),

	{ NULL, }
};

static int del_handler(struct context *ctx, int argc, char *argv[])
{
	return cmd_handler(ctx, del_cmds, argc, argv);
}

static int del_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	cmd_usage(ctx, "FUNCTION [OPTIONS] /dev/plinX\n");
	return subcmds_usage(ctx, "FUNCTION", del_cmds);
}

/*
 * $ lin del frm-entry
 */
static int del_frm_entry_handler(struct context *ctx, int argc, char *argv[])
{
#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif

	if (argc == 1)
		return del_frm_entry_usage(ctx, NULL);
	else if (argc != 3)
		return del_frm_entry_usage(ctx, "wrong number of arguments\n");

	memset(&ctx->u.frm_entry, '\0', sizeof(ctx->u.frm_entry));

	/* Frame ID (or "all") */
	if (!strcmp(argv[1], "all")) {
		ctx->cmd_ioctl_loop_min = PLIN_FRM_ID_MIN;
		ctx->cmd_ioctl_loop_max = PLIN_FRM_ID_MAX;
		ctx->cmd_ioctl_pre = do_pre_frm_entry;
	} else if ((strtobs(argv[1], 1, &ctx->u.frm_entry.id) != 1) ||
				(ctx->u.frm_entry.id > PLIN_FRM_ID_MAX))
		return del_frm_entry_usage(ctx, "ID out of range [%u..%u]\n",
					   PLIN_FRM_ID_MIN,
					   PLIN_FRM_ID_MAX);

	/* all fields must have a valid values */
	ctx->u.frm_entry.len = get_frm_default_len(ctx->u.frm_entry.id);
	ctx->u.frm_entry.checksum = PLIN_FRM_CST_CLASSIC;

	ctx->u.frm_entry.direction = PLIN_FRM_DIR_DISABLED;
	ctx->cmd = PLIOSETFRMENTRY;
	ctx->dev_name = argv[argc-1];

	return 0;
}

static int del_frm_entry_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "ID|all /dev/plinX\n");
}

/*
 * $ lin del schedule
 */
static int del_schd_handler(struct context *ctx, int argc, char *argv[])
{
#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif

	if (argc == 1)
		return del_schd_usage(ctx, NULL);
	else if (argc != 3)
		return del_schd_usage(ctx, "wrong number of arguments\n");

	/* SCHEDULE (or "all") */
	if (!strcmp(argv[1], "all")) {
		ctx->cmd_ioctl_loop_min = PLIN_SCH_IDX_MIN;
		ctx->cmd_ioctl_loop_max = PLIN_SCH_IDX_MAX;
		ctx->cmd_ioctl_pre = do_pre_schedule;
	} else if ((strtobs(argv[1], 1, &ctx->u.del_schd.schedule) != 1)
		|| (ctx->u.del_schd.schedule > PLIN_SCH_IDX_MAX))
		return del_schd_usage(ctx, "SCHEDULE out of range [%u..%u]\n",
				      PLIN_SCH_IDX_MIN,
				      PLIN_SCH_IDX_MAX);

	ctx->cmd = PLIODELSCHD;
	ctx->dev_name = argv[argc-1];
	ctx->rsp_handler = do_display_schd_status_rsp;

	return 0;
}

static int del_schd_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "SCHEDULE[0..7]|all /dev/plinX\n");
}

/*
 * $ lin identify
 */
static int identify_handler(struct context *ctx, int argc, char *argv[])
{
#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif
	if (argc == 1)
		return identify_usage(ctx, NULL);
	else if (argc != 2)
		return identify_usage(ctx, "wrong number of arguments\n");

	ctx->cmd = PLIOIDENTIFY;
	ctx->dev_name = argv[argc-1];

	return 0;
}

static int identify_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "/dev/plinX\n");
}

/*
 * $ lin reset
 */
DECLARE_CMD_NOHELP(reset_tx_queue);
DECLARE_CMD_NOHELP(reset_hw);
static struct cmd reset_opts[] = {
	__CMD_F(--tx-queue, reset_tx_queue, "-t",
"reset Tx queue and USB overrun counter"),
	__CMD_F(--hw, reset_hw, "-w",
"hardware reset (default)"),

	{ NULL, }
};

static int reset_tx_queue_handler(struct context *ctx, int argc, char *argv[])
{
	ctx->cmd = PLIORSTUSBTX;
	return 0;
}

static int reset_hw_handler(struct context *ctx, int argc, char *argv[])
{
	ctx->cmd = PLIORSTHW;
	return 0;
}

static int reset_handler(struct context *ctx, int argc, char *argv[])
{
	struct cmd *opt;

#ifdef DEBUG_TRACE
	printf("%s(argc=%d, argv[0]=\"%s\")\n", __func__, argc, argv[0]);
#endif
	ctx->cmd = PLIORSTHW;

	switch (argc) {

	case 3:
		opt = search_cmd_by_name(argv[1], reset_opts);
		if (!opt)
			return reset_usage(ctx,
					"\"%s\" invalid option\n", argv[1]);

		opt->handler(ctx, argc-2, argv+2);

		/* fall through */
	case 2:
		break;

	case 1:
		return reset_usage(ctx, NULL);

	default:
		return reset_usage(ctx, "wrong number of arguments\n");
	}

	ctx->dev_name = argv[argc-1];

	return 0;
}

static int reset_usage(struct context *ctx, char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	cmd_usage(ctx, "[OPTIONS] /dev/plinX\n");
	return opts_usage(ctx, "OPTIONS", reset_opts);
}

/*
 * $ lin help
 */
static int help_handler(struct context *ctx, int argc, char *argv[])
{
	ctx->history_lvl--;
	return usage(ctx, NULL);
}

/*
 * $ lin help help
 */
static int help_usage(struct context *ctx ,char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "\n");
}

/*
 * $ lin version
 */
static int version_handler(struct context *ctx, int argc, char *argv[])
{
	printf("%u.%u.%u\n", VERSION_MAJOR, VERSION_MINOR, VERSION_SUBMINOR);
	return 0;
}

static int version_usage(struct context *ctx ,char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	cmd_help(ctx, fmt, ap);
	va_end(ap);

	return cmd_usage(ctx, "\n");
}

/*
 * int main(int argc, char *argv[])
 */
int main(int argc, char *argv[])
{
	struct cmd *cmd;
	struct cmd cmd0 = {
		.name = "lin" /* argv[0] */,
		.help_msg = "This program configures the LIN device which node is given on command line",
	};
	struct context ctx = {
		.lin_fd = -1,
		.history_lvl = 0,
		.cmd_ioctl_loop_min = 0,
		.cmd_ioctl_loop_max = 0,
	};
	int i, err;

	ctx.history[ctx.history_lvl++] = &cmd0;

	if (argc < 2)
		return usage(&ctx, NULL);

	cmd = search_cmd_by_name(argv[1], cmds);
	if (!cmd)
		return usage(&ctx, "\"%s\" unknown command", argv[1]);

	ctx.history[ctx.history_lvl++] = cmd;

	/* shortcut to "lin COMMAND help" */
	if (argc > 2)
		if (!strcmp(argv[2], "help") || !strcmp(argv[2], "-h"))
			if (cmd->usage)
				return cmd->usage(&ctx, NULL);

	err = cmd->handler(&ctx, argc-1, argv+1);
	if (err)
		return err;

	if (!ctx.dev_name)
		return 0;

	ctx.lin_fd = open(ctx.dev_name, O_RDWR);
	if (ctx.lin_fd < 0) {
		fprintf(stderr, "Failed to open \"%s\" (errno=%d)\n",
			ctx.dev_name, errno);
		return 2;
	}

	/* if set,  send the cmd to the LIN device */
	if (ctx.cmd) {
		int hc = 0;

		/* repeat the cmd if requested */
		for (ctx.cmd_ioctl_loop = ctx.cmd_ioctl_loop_min;
				ctx.cmd_ioctl_loop <= ctx.cmd_ioctl_loop_max;
							ctx.cmd_ioctl_loop++) {

			if (ctx.cmd_ioctl_pre) {
				err = ctx.cmd_ioctl_pre(&ctx);
				if (err < 0)
					break;
				if (err > 0)
					continue;
			}

			err = ioctl(ctx.lin_fd, ctx.cmd, &ctx.u);
			if (err) {
				fprintf(stderr, "ioctl(%u) failure: errno=%d\n",
					ctx.cmd, errno);
				break;
			}

			if (ctx.rsp_handler) {
				err = ctx.rsp_handler(&ctx, hc++);
				if (err)
					break;
			}
		}
	}

	/* close the device */
	close(ctx.lin_fd);

	return err;
}
