/* SPDX-License-Identifier: LGPL-2.1-only */
/*
 * linread.c - a small program to read LIN frames from a LIN device.
 *
 * Copyright (C) 2015-2025  PEAK System-Technik GmbH <www.peak-system.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 * Contact: <linux.peak@hms-networks.com>
 * Author:  Stephane Grosjean <stephane.grosjean@hms-networks.com>
 */
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>

/* chardev interface */
#include <fcntl.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <errno.h>

#include <string.h>
#include <ctype.h>

#include <signal.h>

#include "plin.h"

#define VERSION_MAJOR		1
#define VERSION_MINOR		1
#define VERSION_SUBMINOR	0

struct token {
	char abbrv;
	char *str;
};

static struct token dir_name[] = {
	{ 'd', "disabled" },
	{ 'p', "publisher" },
	{ 's', "subscriber" },
	{ 'S', "subscriber (auto len)" }
};

static struct token cs_name[] = {
	{ 'u', "custom" },
	{ 'c', "classic" },
	{ 'e', "enhanced" },
	{ 'a', "auto" }
};

static const char *prgname;

static const char *count_str = "count";
static const char *eol_str = "eol";
static const char *fmt_str = "fmt";
static const char *frames_only_str = "frames-only";
static const char *help_str = "help";
static const char *verbose_str = "verbose";
static const char *version_str = "version";

static int ctrl_c = 0;
static int show_frames_only = 0;

static void usage(char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);

	if (fmt && *fmt) {
		fprintf(stderr, "ERROR: ");
		vfprintf(stderr, fmt, ap);
	} else {
		fprintf(stderr,
"This program displays messages read from the LIN device\n");
	}

	fprintf(stderr,"\nUSAGE: %s -h|-v\n", prgname);
	fprintf(stderr,"       %s [OPTIONS] /dev/plinX\n", prgname);
	fprintf(stderr, "\nOPTIONS:\n");

	fprintf(stderr,
"   -%c | --%s=x\t\tset EOL char value (0xa|0xd)\n", *eol_str, eol_str);
	fprintf(stderr,
"   -%c | --%s=FMT\tset digits output format (x|X|d|u)\n", *fmt_str, fmt_str);
	fprintf(stderr,
"   -%c | --%s\tdisplay only LIN frames received\n", toupper(*frames_only_str), frames_only_str);
	fprintf(stderr,
"   -%c | --%s=N\tread N msgs then quit\n", 'n', count_str);
	fprintf(stderr,
"   -%c | --%s\t\tdisplay this help\n", *help_str, help_str);
	fprintf(stderr,
"   -%c | --%s\tdisplay the version of the software\n", *version_str, version_str);
	fprintf(stderr,
"   -%c | --%s[=x]\tverbose mode (level x)\n", toupper(*verbose_str), verbose_str);
	fprintf(stderr, "\n");

	va_end(ap);

	exit(1);
}

/*
 * strtounit(argv, "kM");
 * strtounit(argv, "ms");
 */
static unsigned long strtounit(const char *str, const char *units)
{
	char *endptr;
	__u32 m = 1;

	__u32 v = strtoul(str, &endptr, 0);
	if (*endptr) {
		if (units) {
			const char *pu;

			/* might not be invalid if found char is a unit */
			for (pu = units; *pu; pu++) {
				m *= 1000;
				if (*endptr == *pu)
					break;
			}
			if (!*pu)
				usage("Unknown unit character in numeric "
				      "value on command line");

		} else {
			char tmp[512];
			snprintf(tmp, sizeof(tmp),
				"Invalid character in numeric value \"%s\" on "
				"command line", str);
			usage(tmp);
		}
	}

	return v * m;
}

static int strtobs(const char *str, int n, u8 *pb)
{
	int i;

	if (!pb)
		return 0;

	*pb = 0x00;
	for (i = 0; *str && (i < n); i++) {

		char *endptr;

		__u32 v	= strtoul(str, &endptr, 0);
		switch (*endptr) {
		case ',':
		case ':':
		case '-':
		case ' ':
			endptr++;
		case '\0':
			str = endptr;
			break;
		default:
			*pb = 0xff;	/* error */
			return i;
		}

		*pb++ = (u8 )v;
	}

	return i;
}

static int islongopt(const char *argv, const char *opt)
{
	return !strncmp(argv+2, opt, strlen(opt));
}

static char *get_fmt(const char *fmt, char c)
{
	static char tmp_fmt[64];
	snprintf(tmp_fmt, sizeof(tmp_fmt), "%s%c ", fmt, c);
	return tmp_fmt;
}

static void catch_ctrl_c(int s)
{
	ctrl_c = 1;
	printf("User interrupt!\n");
}

int main(int argc, char *argv[])
{
	int lin_dev;
	char *dev_name = NULL;
	int i, cmd = 0, err = 0, count = 0, rx_count = 0;
	char eol = '\n', fmt = 'x';
	char data_fmt[] = "%02";
	struct plin_msg msg;
	enum {
		VERBOSE_QUIET,
		VERBOSE_MIN,
		VERBOSE_NORMAL,
		VERBOSE_HIGH
	} verbose = VERBOSE_MIN;
	struct sigaction sa_act;

	prgname = argv[0];

	for (i = 1; i < argc; i++) {
		char opt, *eqptr;

		switch (argv[i][0]) {
		case '-':
			opt = argv[i][1];
			if (opt == '-') {
				if (islongopt(argv[i], help_str)
				 || islongopt(argv[i], eol_str)
				 || islongopt(argv[i], fmt_str)
				 || islongopt(argv[i], version_str)
				)
					opt = argv[i][2];

				else if (islongopt(argv[i], verbose_str)
				      || islongopt(argv[i], frames_only_str)
				     )
					opt = toupper(argv[i][2]);

				else if (islongopt(argv[i], count_str))
					opt = 'n';
				else
					usage("Invalid long option '--%s'",
					      argv[i]+2);
			}

			eqptr = strchr(argv[i], '=');

			switch (opt) {

			case 'e':
				if (!eqptr)
					usage("%s option needs a value\n",
						eol_str);

				strtobs(eqptr+1, 1, &eol);
				break;

			case 'f':
				if (!eqptr)
					usage("%s option needs a char value\n",
						fmt_str);

				switch (eqptr[1]) {
				case 'd':
				case 'u':
					data_fmt[2] = '3';

					/* fall through */
				case 'x':
				case 'X':
					fmt = eqptr[1];
					break;
				default:
					usage("Invalid value of %s option ",
						fmt_str);
				}
				break;
			case 'F':
				show_frames_only = 1;
				break;

			case 'h':
				usage(NULL);
				break;

			case 'n':
				if (!eqptr)
					usage("%s option needs a value\n",
						count_str);

				count = strtounit(eqptr+1, NULL); 
				break;

			case 'v':
				printf("%u.%u.%u\n",
					VERSION_MAJOR,
					VERSION_MINOR,
					VERSION_SUBMINOR);
				exit(0);
				break;

			case 'V':
				if (eqptr) {
					verbose = strtounit(eqptr+1, NULL);
					if (verbose > VERBOSE_HIGH)
						verbose = VERBOSE_HIGH;
				} else
					verbose = VERBOSE_NORMAL;
				break;

			default:
				usage("Invalid option '-%c'", opt);
				break;
			}
			break;

		default:
			dev_name = argv[i];
			break;
		}
	}

	if (!dev_name)
		usage("%s needs a device name\n", prgname);

	/* open the device */
	lin_dev = open(dev_name, O_RDONLY);
	if (lin_dev < 0)
		usage("Failed to open \"%s\" (errno=%d)\n",
		      dev_name, errno);

	if (verbose > VERBOSE_QUIET) {
		printf("     count    timestamp d ID ");
		if (verbose > VERBOSE_NORMAL)
			printf("CS flags ");
		if (verbose > VERBOSE_MIN)
			printf("len ");
		printf("data\n");
		printf("------------------------------------------------------------------------------");
		if (eol != 0x0a)
			putchar('\n');
	}

	/* clear SA_RESTART flag so that read() won't restart once
	 * (our) SIGINT signal handler will have been called. */
	sigaction(SIGINT, NULL, &sa_act);
	sa_act.sa_flags &= ~SA_RESTART;
	sa_act.sa_handler = catch_ctrl_c;
	sigaction(SIGINT, &sa_act, NULL);

	/* read frames from the LIN device while not interrupted */
	while (1) {

		err = read(lin_dev, &msg, sizeof(msg));
		if (err < 0) {
			if (!ctrl_c)
				fprintf(stderr, "read(%d) failure: errno=%d\n",
					lin_dev, errno);
			break;
		}

		rx_count++;

		if (verbose == VERBOSE_QUIET)
			goto lbl_continue;

		if (show_frames_only && (msg.type != PLIN_MSG_FRAME))
			goto lbl_continue;

		printf("%c%10u %12llu ", eol, rx_count, msg.ts_us);
		/* printf(get_fmt("%1", fmt), msg.dir); */
		printf("%c ", (msg.dir <= PLIN_FRM_DIR_SUBSCRIBER_AUTO_LEN) ?
			dir_name[msg.dir].abbrv : '-');

		switch (msg.type) {
		case PLIN_MSG_FRAME:
			printf(get_fmt("%02", fmt), msg.id);

			if (verbose > VERBOSE_NORMAL) {
				/*printf(get_fmt(" %c", fmt),*/
				printf(" %c ",
					(msg.cs_type <= PLIN_FRM_CST_AUTO) ?
					cs_name[msg.cs_type].abbrv : '-');
				printf(" %04x ", msg.flags);
			}

			if (verbose > VERBOSE_MIN)
				printf(get_fmt("%3", fmt), msg.len);
			if (msg.flags) {
				if (msg.flags & PLIN_FRM_ERR_INC_SYNC)
					printf("Inconsistent sync; ");
				if (msg.flags & PLIN_FRM_ERR_PARITY0)
					printf("Parity bit 0; ");
				if (msg.flags & PLIN_FRM_ERR_PARITY1)
					printf("Parity bit 1; ");
				if (msg.flags & PLIN_FRM_ERR_SLV_NOT_RSP)
					printf("Slave not responding; ");
				if (msg.flags & PLIN_FRM_ERR_TIMEOUT)
					printf("Time out; ");
				if (msg.flags & PLIN_FRM_ERR_BAD_CS)
					printf("Bad checksum; ");
				if (msg.flags & PLIN_FRM_ERR_BUS_SHORT_GND)
					printf("Bus short to GND; ");
				if (msg.flags & PLIN_FRM_ERR_BUS_SHORT_VBAT)
					printf("Bus short to VBat; ");
				if (msg.flags & PLIN_FRM_ERR_RESERVED)
					printf("Reserved; ");
				if (msg.flags & PLIN_FRM_ERR_OTHER_RSP)
					printf("Other response; ");
			} else {
				for (i = 0; i < msg.len; i++)
					printf(get_fmt(data_fmt, fmt),
						msg.data[i]);
			}
			break;

		case PLIN_MSG_SLEEP:
			printf("SLEEP");
			break;

		case PLIN_MSG_WAKEUP:
			printf("WAKEUP");
			break;

		case PLIN_MSG_AUTOBAUD_TO:
			printf("AUTOBAUD_TO");
			break;
	
		case PLIN_MSG_AUTOBAUD_OK:
			printf("AUTOBAUD_OK");

			if (verbose > VERBOSE_MIN) {
				printf(": %02x%02x%02x%02xh",
					msg.data[3], msg.data[2],
					msg.data[1], msg.data[0]);
				if (verbose > VERBOSE_NORMAL) {
					u32 tmp32;

					memcpy(&tmp32, msg.data, 4);
					printf(" (%u bps)",
						8000000 / le32toh(tmp32));
				}
			}
			break;

		case PLIN_MSG_OVERRUN:
			printf("OVERRUN");
			break;

		default:
			printf("Unsupported msg type %d (size=%d)\n",
			       msg.type, err);
			break;
		}

		/* needed when eol!=0x0a */
		fflush(stdout);

lbl_continue:
		if (count && (rx_count >= count))
			break;
	}

	putchar('\n');

	/* close the device */
	close(lin_dev);

	return err;
}
