/* SPDX-License-Identifier: LGPL-2.1-only */
/*
 * linwrite.c - a small program to write a LIN frame through a LIN device.
 *
 * Copyright (C) 2015-2025  PEAK System-Technik GmbH <www.peak-system.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 * Contact: <linux.peak@hms-networks.com>
 * Author:  Stephane Grosjean <stephane.grosjean@hms-networks.com>
 */
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>

/* chardev interface */
#include <fcntl.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <errno.h>

#include <string.h>
#include <ctype.h>

#include "plin.h"

#define VERSION_MAJOR		1
#define VERSION_MINOR		1
#define VERSION_SUBMINOR	0

static const char *prgname;

static const char *bytes_str = "bytes";
static const char *cstype_str = "cstype";
static const char *dir_str = "dir";
static const char *help_str = "help";
static const char *id_str = "id";
static const char *len_str = "len";
static const char *num_str = "num";
static const char *pause_str = "pause";
static const char *version_str = "version";

#define LIN_FRM_ID_DEF		0x001
#define LIN_FRM_DIR_DEF		PLIN_FRM_DIR_PUBLISHER
#define LIN_FRM_LEN_DEF		8
#define LIN_FRM_CST_DEF		PLIN_FRM_CST_CLASSIC

static void usage(char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);

	if (fmt && *fmt) {
		fprintf(stderr, "ERROR: ");
		vfprintf(stderr, fmt, ap);
	} else {
		fprintf(stderr,
"This program writes a %u data bytes Publisher (default) LIN frame with\n"
"ID=0x%x on the LIN device given on command line, with classic checksum type\n"
"(default), during master mode when the scheduler is not running.\n",
			LIN_FRM_LEN_DEF, LIN_FRM_ID_DEF);
	}

	fprintf(stderr,"\nUSAGE: %s -h|-v\n", prgname);
	fprintf(stderr,"       %s [OPTIONS] /dev/plinX\n", prgname);
	fprintf(stderr, "\nOPTIONS:\n");

	fprintf(stderr,
"   -%c | --%s[=\"x y z ...\"]\tchange default data bytes\n", *bytes_str, bytes_str);
	fprintf(stderr,
"   -%c | --%s[=U|C|E|A]\tchange CS type: cUstom|Classic|Enhanced|Auto\n", *cstype_str, cstype_str);
	fprintf(stderr,
"   -%c | --%s[=D|P|S|A]\t\tchange dir: Disabled|Publisher|Subscriber|\n"
"\t\t\t\tsubscriber Auto len\n", *dir_str, dir_str);
	fprintf(stderr,
"   -%c | --%s\t\t\tdisplay this help\n", *help_str, help_str);
	fprintf(stderr,
"   -%c | --%s[=x]\t\tchange default LIN frame Id.\n", *id_str, id_str);
	fprintf(stderr,
"   -%c | --%s[=x]\t\tchange default LIN frame data length\n", *len_str, len_str);
	fprintf(stderr,
"   -%c | --%s[=s]\t\tdefine the number of times the frame is sent to\n"
"\t\t\t\tthe driver\n", *num_str, num_str);
	fprintf(stderr,
"   -%c | --%s[=s]\t\tchange default pause time after transmitting the\n"
"\t\t\t\tframe to the USB driver (in s.)\n", *pause_str, pause_str);
	fprintf(stderr,
"   -%c | --%s\t\tdisplay the version of the software\n", *version_str, version_str);
	fprintf(stderr, "\n");

	va_end(ap);

	exit(1);
}

static int strtobs(const char *str, int n, u8 *pb)
{
	int i;

	if (!pb)
		return 0;

	*pb = 0x00;
	for (i = 0; *str && (i < n); i++) {

		char *endptr;

		__u32 v	= strtoul(str, &endptr, 0);
		switch (*endptr) {
		case ',':
		case ':':
		case '-':
		case ' ':
			endptr++;
		case '\0':
			str = endptr;
			break;
		default:
			*pb = 0xff;	/* error */
			return i;
		}

		*pb++ = (u8 )v;
	}

	return i;
}

static int islongopt(const char *argv, const char *opt)
{
	return !strncmp(argv+2, opt, strlen(opt));
}

int main(int argc, char *argv[])
{
	int lin_dev;
	char *dev_name = NULL;
	int i, cmd = 0, err = 0;
	void *arg = NULL;
	u8 pause_after_write = 1;
	u8 write_number = 1;
	struct plin_msg msg = {
		.id = LIN_FRM_ID_DEF,
		.len = LIN_FRM_LEN_DEF,
		.cs_type = LIN_FRM_CST_DEF,
		.dir = LIN_FRM_DIR_DEF,
		.data = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07 },
	};

	prgname = argv[0];

	for (i = 1; i < argc; i++) {
		char opt, *eqptr;

		switch (argv[i][0]) {
		case '-':
			opt = argv[i][1];
			if (opt == '-') {
				if (islongopt(argv[i], bytes_str)
				 || islongopt(argv[i], cstype_str)
				 || islongopt(argv[i], dir_str)
				 || islongopt(argv[i], help_str)
				 || islongopt(argv[i], id_str)
				 || islongopt(argv[i], len_str)
				 || islongopt(argv[i], num_str)
				 || islongopt(argv[i], pause_str)
				 || islongopt(argv[i], version_str)
				)
					opt = argv[i][2];
				else
					usage("Invalid long option '--%s'",
					      argv[i]+2);
			}

			eqptr = strchr(argv[i], '=');

			switch (opt) {

			case 'b':
				if (!eqptr)
					usage("%s option needs a value\n",
					      bytes_str);
				
				msg.len = strtobs(eqptr+1, 8, msg.data);
				break;

			case 'c':
				if (!eqptr) {
					/* -c is a shortcut for -c=c */
					msg.cs_type = PLIN_FRM_CST_CLASSIC;
					break;
				}
				
				switch (eqptr[1]) {
				case 'u':
				case 'U':
					msg.cs_type = PLIN_FRM_CST_CUSTOM;
					break;
				case 'c':
				case 'C':
					msg.cs_type = PLIN_FRM_CST_CLASSIC;
					break;
				case 'e':
				case 'E':
					msg.cs_type = PLIN_FRM_CST_ENHANCED;
					break;
				case 'a':
				case 'A':
					msg.cs_type = PLIN_FRM_CST_AUTO;
					break;

				default:
					strtobs(eqptr+1, 1, &msg.cs_type);
					break;
				}
				break;

			case 'd':
				if (!eqptr)
					usage("%s option needs a value\n",
					      dir_str);
				
				switch (eqptr[1]) {
				case 'd':
				case 'D':
					msg.dir = PLIN_FRM_DIR_DISABLED;
					break;
				case 'p':
				case 'P':
					msg.dir = PLIN_FRM_DIR_PUBLISHER;
					break;
				case 's':
				case 'S':
					msg.dir = PLIN_FRM_DIR_SUBSCRIBER;
					break;
				case 'a':
				case 'A':
					msg.dir = PLIN_FRM_DIR_SUBSCRIBER_AUTO_LEN;
					break;

				default:
					strtobs(eqptr+1, 1, &msg.dir);
					break;
				}
				break;

			case 'e':
				/* -e is a shortcut for -c=e */
				msg.cs_type = PLIN_FRM_CST_ENHANCED;
				break;

			case 'h':
				usage(NULL);
				break;

			case 'i':
				if (!eqptr)
					usage("%s option needs a value\n",
					      bytes_str);
				
				strtobs(eqptr+1, 1, &msg.id);
				break;

			case 'l':
				if (!eqptr)
					usage("%s option needs a value\n",
					      bytes_str);
				
				strtobs(eqptr+1, 1, &msg.len);
				break;

			case 'n':
				if (!eqptr)
					usage("%s option needs a value\n",
					      num_str);

				strtobs(eqptr+1, 1, &write_number);
				if (!write_number)
					write_number = 1;
				break;

			case 'p':
				if (!eqptr)
					usage("%s option needs a value\n",
					      pause_str);

				strtobs(eqptr+1, 1, &pause_after_write);
				break;

			case 'v':
				printf("%u.%u.%u\n",
					VERSION_MAJOR,
					VERSION_MINOR,
					VERSION_SUBMINOR);
				exit(0);
				break;
			default:
				usage("Invalid option '-%c'", opt);
				break;
			}
			break;

		default:
			dev_name = argv[i];
			break;
		}
	}

	if (!dev_name)
		usage("%s needs a device name\n", prgname);

	/* open the device */
	lin_dev = open(dev_name, O_WRONLY);
	if (lin_dev < 0)
		usage("Failed to open \"%s\" (errno=%d)\n",
		      dev_name, errno);

	/* repeat writing the frame to the LIN device */
	for (i = 0; i < write_number; i++) {
		err = write(lin_dev, &msg, sizeof(msg));
		if (errno) {
			fprintf(stderr, "write(%d) failure: errno=%d\n",
				lin_dev, errno);
			break;
		}
		sleep(pause_after_write);
	}

	/* close the device */
	close(lin_dev);

	return err;
}
