#!/bin/bash
#
# Start Master mode on given LIN device.
#
# USAGE:
#
# $ ./start_master.sh /dev/plinX
#
BAUDRATE=9600

#HB_ID=0x1F
PERIODIC_ID=0x22

# either HB or scheduled frame period
PERIOD_MS=1000
SCHD=0

MYSELF=`basename $0`
LIN=lin
LINREAD=linread

CUT=cut
ECHO=/bin/echo
GREP=grep
#SUDO=sudo

fatal()
{
	if [ -z "$err" -o $err -ne 0 ]; then
		if [ -z "$*" ]; then
			$ECHO "Failed (err $err)"
		else
			$ECHO $*
		fi
		exit 1
	fi
	err=""
}

[ "${1}x" == "x" ] && fatal "Usage: $MYSELF /dev/plinX"

dev=$1

# 0 - reset device
$ECHO "- resetting $dev..."
$SUDO $LIN reset $dev
err=$? && fatal

# 1 - setup Master mode
$ECHO "- starting $dev in MASTER mode at $BAUDRATE bps"
$SUDO $LIN start master $BAUDRATE $dev
err=$? && fatal "Failed to turn $dev in master mode (err $err)"

# 2 - get the status of the LIN node
$ECHO "- checking if mode and baudrate are ok"
dev_baudrate=`$SUDO $LIN get status $dev | $GREP "^baudrate=" | $CUT -d= -f2`
[ "${dev_baudrate}x" != "${BAUDRATE}x" ] && fatal "Error: unable to setup $dev with baudrate=$BAUDRATE"

# 3 - in order to see LIN frames, set (for example) USB id
#     filter to "all open" (default is: "all closed")
$ECHO "- setting ID filter to 'receive all'"
#$SUDO $LIN set id-filter 0xff:0xff:0xff:0xff:0xff:0xff:0xff:0xff $dev
$SUDO $LIN set id-filter all-opened $dev
err=$? && fatal

# 4 - set HeartBeat with Id=$HB_ID
if [ -n "$HB_ID" ]; then
	$ECHO "- stop scheduling (for the keep-alive mechanism to run)"
	$SUDO $LIN suspend scheduler $dev 2> /dev/null
	err=$? && fatal

	$ECHO "- starting keep-alive mechanism with sending ID=$HB_ID every $PERIOD_MS ms."
	$SUDO $LIN start keep-alive $HB_ID $PERIOD_MS $dev
	err=$? && fatal "Failed to start keep-alive mechanism (err $err)"
	#echo "Keep-alive frame $HB_ID written every $PERIOD_MS ms. ..."

# or define our own periodic frame
elif [ -n "$PERIODIC_ID" ]; then
	$ECHO "- create frame $PERIODIC_ID"
	#$SUDO $LIN set sub-frm-entry $PERIODIC_ID --cs-enhanced --auto-len $dev
	$SUDO $LIN set sub-frm-entry $PERIODIC_ID -c -l 8 $dev
	err=$? && fatal

	$ECHO "- add frame $PERIODIC_ID to scheduler $SCHD"
	$SUDO $LIN add unc-schd-slot $SCHD $PERIOD_MS $PERIODIC_ID $dev
	err=$? && fatal

	$ECHO "- starting scheduler $SCHD on $dev"
	$SUDO $LIN start schedule $SCHD $dev
	err=$? && fatal
fi

# 5 - Setup our ^C handler to properly disconnect from the LIN bus.
disconnect() {
	$ECHO "- disconnecting $dev from the LIN bus"
	$SUDO $LIN disconnect $dev
}
trap "disconnect" INT

# 6 - say we're ok:
$ECHO "- identifying $dev (LED should blink)"
$SUDO $LIN identify $dev
err=$? && fatal

# 7 - start reading frames (heXadecimal display; no scroll)
$ECHO "- reading LIN data from $dev (^C to stop)..."
$ECHO
$SUDO $LINREAD -V=3 -f=X --eol=0x0d $dev
err=$? && fatal

# 8 - Ok
exit 0
