#!/bin/bash
#
# Start Slave mode on given LIN device.
#
# USAGE:
#
# $ ./start_slave.sh /dev/plinX
#
BAUDRATE=9600

#HB_ID=0x1F
HB_ID=0x22

CHANGE_DATA_PERIOD=5

MYSELF=`basename $0`
LIN=lin
LINREAD=linread
LINWRITE=linwrite

CUT=cut
ECHO=/bin/echo
GREP=grep
#SUDO=sudo

fatal()
{
	if [ -z "$err" -o $err -ne 0 ]; then
		if [ -z "$*" ]; then
			$ECHO "Failed (err $err)"
		else
			$ECHO $*
		fi
		exit 1
	fi
	err=""
}

[ "${1}x" == "x" ] && fatal "Usage: $MYSELF /dev/plinX"

dev=$1

# 0 - reset device
$ECHO "- resetting $dev..."
$SUDO $LIN reset $dev
err=$? && fatal

# 1 - setup Slave mode
$ECHO "- starting $dev in SLAVE mode at $BAUDRATE bps"
$SUDO $LIN start slave $BAUDRATE $dev
err=$? && fatal "Failed to turn $dev in slave mode (err $err)"

# 2 - get the status of the LIN node
$ECHO "- checking if mode and baudrate are ok"
dev_baudrate=`$SUDO $LIN get status $dev | $GREP "^baudrate=" | $CUT -d= -f2`
[ "${dev_baudrate}x" != "${BAUDRATE}x" ] && fatal "Error: unable to setup $dev with baudrate=$BAUDRATE"

# 3 - in order to see LIN frames, set (for example) USB id
#     filter to "all open" (default is: "all closed")
$ECHO "- setting ID filter to 'receive all'"
#$SUDO $LIN set id-filter 0xff:0xff:0xff:0xff:0xff:0xff:0xff:0xff $dev
$SUDO $LIN set id-filter all-opened $dev
err=$? && fatal

if [ -n "$HB_ID" ]; then
#	$ECHO "- define response frame to $HB_ID (with enhanced cs)"
#	$SUDO $LIN set pub-frm-entry $HB_ID -r -e -d "1 2 3 4 5 6 7 8" $dev
	$ECHO "- define response frame to $HB_ID (with classic cs)"
	$SUDO $LIN set pub-frm-entry $HB_ID -r -c -d "1 2 3 4 5 6 7 8" $dev
	err=$? && fatal
fi

# 4 - Setup our ^C handler to properly disconnect from the LIN bus.
disconnect() {
	$ECHO "- disconnecting $dev from the LIN bus"
	$SUDO $LIN disconnect $dev
}
trap disconnect SIGINT

# 5 - say we're ok:
$ECHO "- identifying $dev (LED should blink)"
$SUDO $LIN identify $dev
err=$? && fatal

# Simulate a change in HB_ID data every 5 seconds
if [ -n "$HB_ID" -a -n "$CHANGE_DATA_PERIOD" ]; then
	( while :; do
		sleep $CHANGE_DATA_PERIOD
		ps $$ > /dev/null
		[ $? -ne 0 ] && break

		#$ECHO "- change data in response frame $HB_ID"
		if [ "${inverted}es" != "yes" ]; then
			$SUDO $LIN set frm-entry-data $HB_ID -d "8 7 6 5 4 3 2 1" $dev
			[ $? -eq 0 ] && inverted=y
		else
			$SUDO $LIN set frm-entry-data $HB_ID -d "1 2 3 4 5 6 7 8" $dev
			[ $? -eq 0 ] && inverted=n
		fi
	done ) &
fi

# 6 - start reading frames (heXadecimal display; no scroll)
$ECHO "- reading LIN data from $dev (^C to stop)..."
$ECHO
$SUDO $LINREAD -V=3 -f=X --eol=0x0d $dev
err=$? && fatal

# 7 - Ok
exit 0
