/*
Example for using Interrupt-driven UART Functions.

*/

#include "compiler.h"
#include "mb90495.h"
#include "vectors.h"
#include "can.h"
#include "hardware.h"
#include "timer.h"
#include "i2c.h"
#include "i2c_2.h"
#include "eeprom.h"
#include "adi.h"
#include "pwm.h"
#include "uart1.h"
#include "mcan.h"


#define MODULE 0

#if MODULE==0
 #define TX_ID	0x101
 #define RX_ID	0x100
#else
 #define TX_ID	0x100
 #define RX_ID	0x101
#endif

//*************************************************************
static void Eval_CANFrames(void);			//CAN receive function
//***********************************************************************
// CAN-Queues
#define QUEUE_RCV_SIZE	8	// reserve memory for CAN-Queue, count must be 2,4,8,16,...
#define QUEUE_XMT_SIZE	8	// reserve memory for CAN-Queue, count must be 2,4,8,16,...
//**************************************************************
//UART-Queues
#define TX_entries 60		//UART transmit buffer 59 Byte (max. size + 1)		
#define RX_entries 60		//UART receive buffer 59 Byte (max. size + 1)
//*************** Globale Variablen ****************************
CANI_qentrytyp rcv_queue[QUEUE_RCV_SIZE];	
CANI_qentrytyp xmt_queue[QUEUE_XMT_SIZE];	

u8_t Buf_XMTQueue[TX_entries];
u8_t Buf_RCVQueue[RX_entries];
//***********************************************************************

void Timer_2000Hz()
{
	static u8_t n;
	
	n++;
	if(n&1)	// runs only ever second call
	{
		LED_1000Hz();
	}
}

//***********************************************************************
static void Eval_CANFrames()
{	u8_t ret;
	ret = CAN_Read();
	if(ret == CAN_ERR_OK)	// if an new CAN-Message was received
	{
		switch(CAN_BUFF_ID)
		{
			// Firmwareupdate via CAN
			//convert x.mhx into x.bin via mot2bin.exe (see ...\Release\ABS) and upload with CAN2Flash.exe to MicroMod  
			case 0x7e7:
/* 
The Library-Funktion  CAN_Read() fills the variables CAN_BUFF_ID, CAN_BUFF_LEN, CAN_BUFF_DATA[8] with the last Queue-entry of the CAN-message.
When using own CAN-LowLevel drivers must befor call the function mCAN_EvalInitialCmd() the variables
CAN_BUFF_ID
CAN_BUFF_LEN
CAN_BUFF_DATA[8]
be filled with the CAN-data.

*/
				// all outputs off
				Set_DOUTByte(0);
								
				mCAN_EvalInitialCmd();			
			break;
			
			case RX_ID:
				UART1_Write((u8_t*)CAN_BUFF_DATA, CAN_BUFF_LEN);
			break;
			
	//process other CAN-ID 
			default:
			
			break;
		} // switch
		
	} // if(CAN_Read()=...
	else if(ret & CAN_ANY_ERROR)
	{
		switch(CAN_BUFF_ERROR)
		{
			case CAN_ERR_ACTIVE:
				UART1_Write((u8_t*)"Error Active\r",13);
			break;
			case CAN_ERR_WARNING:
				UART1_Write((u8_t*)"Error Warning\r",14);
			break;
			
			case CAN_ERR_PASSIVE:
				UART1_Write((u8_t*)"Error Passive\r",14);
			break;
			
			case CAN_ERR_BUSOFF:
				UART1_Write((u8_t*)"Busoff\r",7);
			break;

		}
	}
}


//***********************************************************************
void main()
{
	u8_t buf[10];
		
	InitIrqLevels();
    __set_il(7);		// allow all levels 

	Hardware_Init();
	Timer_Init();
	
	UART1_SetXMTQueue(Buf_XMTQueue, TX_entries);		//transmit-buffer
	UART1_SetRCVQueue(Buf_RCVQueue, RX_entries);		//receive-buffer
	UART1_Init(19200,BITLEN_8,NO_PARITY,STOPLEN_1);

	CAN_SetRCVQueue(rcv_queue, QUEUE_RCV_SIZE);	// Set Receive Queue to 8 entries, 
	CAN_SetXMTQueue(xmt_queue, QUEUE_XMT_SIZE);	// Set CAN Transmit Queue to 8 entries
	CAN_Init(CAN_BAUD_500K);
	CAN_RegisterMsg(0);
	
	CAN_Online();
	
	__EI();				// Enable Interrupts

	Timer_Set(1,2000);	
	UART1_Write((u8_t*)"Start\r",6);
	
	for(;;)							//Main_loop
	{
		Eval_CANFrames();		//call CAN receive function
		WATCHDOG;
		
		
		if(UART1_Read(buf,1)==1)	// read one byte from UART
		{
			if(buf[0]=='1')	// Set CAN-bus Online
			{
				CAN_Online();
				UART1_Write((u8_t*)"Online\r",7);
			}

			else if(buf[0]=='0')	// Set CAN-Bus Offline
			{
				CAN_Offline();
				UART1_Write((u8_t*)"Offline\r",8);
			}

			else if(buf[0]=='t')	// Transmit a CAN-Message
			{
				CAN_BUFF_ID=0x123;
				CAN_BUFF_LEN=1;
				CAN_BUFF_DATA[0]=0x55;
				CAN_BUFF_FORMAT=0;
				CAN_Write();
			}

			else if(buf[0]=='L')	// Set Listen Only Mode
			{	// do NOT try to send CAN-Messages while in Listen-Only Mode. This will lead to CAN-Error.
				CAN_SetListenOnly(1);
				UART1_Write((u8_t*)"ListenOnly=ON\r",14);
			}

			else if(buf[0]=='N')	// Set Normal Mode
			{
				CAN_SetListenOnly(0);
				UART1_Write((u8_t*)"ListenOnly=OFF\r",15);
			}
		}
	} //for_ever loop
}