﻿// Copyright (C) 2009 PEAK-System Technik GmbH.  All rights reserved.

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.Xml;
using System.Runtime.InteropServices;
using PLinView.Properties;

namespace PLinView
{
    public partial class MainWnd : Form
    {
        #region Attributes

        #region Misc
        /// <summary>
        /// Constant value that indicate the mask of the client filter (64bit)
        /// </summary>
        private const ulong FRAME_FILTER_MASK = 0xFFFFFFFFFFFFFFFF;
        /// <summary>
        /// Filename of loaded file.
        /// </summary>
        private String m_sFileName;
        /// <summary>
        /// Boolean to register the current edit mode of different controls.
        /// To protect the control's event for recursive calling.
        /// </summary>
        private Boolean m_fAutomatic;
        #endregion

        #region Frames
        /// <summary>
        /// Global Frame Table object to hold all frame definition.
        /// </summary>
        private CGlobalFrameTable m_pGFT; 
        /// <summary>
        /// Transmit list object to hold all user defined frames.
        /// </summary>
        private CXmtFrameList m_pXmtFrames;
        /// <summary>
        /// Receive list object to hold all received frames.
        /// </summary>
        private CRcvFrameList m_pRcvFrames;
        #endregion

        #region LIN
        /// <summary>
        /// Client handle
        /// </summary>
        private byte m_hClient;
        /// <summary>
        /// Hardware handle
        /// </summary>
        private ushort m_hHw;
        /// <summary>
        /// Hardware channel
        /// </summary>
        private ushort m_nHwChannel;
        /// <summary>
        /// LIN Hardware Modus (Master/Slave)
        /// </summary>
        private Peak.Lin.TLINHardwareMode m_HwMode;
        /// <summary>
        /// Client filter mask
        /// </summary>
        private ulong m_lMask;
        /// <summary>
        /// Baudrate Index of Hardware
        /// </summary>
        private ushort m_wBaudrate;
        /// <summary>
        /// Name of connected hardware
        /// </summary>
        private String m_sHardwareName;
        /// <summary>
        /// Baudrate Values of Hardware
        /// </summary>
        //private ushort[] m_BaudValues = new ushort[5] { 0, 2400, 9600, 10400, 19200 };
        /// <summary>
        /// Last LIN error
        /// </summary>
        private Peak.Lin.TLINError m_LastLINErr;
        /// <summary>
        /// Last LIN hardware mode
        /// </summary>
        private Peak.Lin.TLINHardwareMode m_LINMode;
        /// <summary>
        /// Last LIN hardware state
        /// </summary>
        private Peak.Lin.TLINHardwareState m_LINState;
        /// <summary>
        /// Current count of overruns in the receive queue of the LIN hardware.
        /// </summary>
        private ushort m_wOverruns;
        #endregion

        #endregion

        #region Methods

        /// <summary>
        /// Constructor of the MainWnd object to create the main window
        /// </summary>
        public MainWnd()
        {
            InitializeComponent();
            m_sFileName = "";
            m_fAutomatic = false;
            // Create the Global Frame Table
            m_pGFT = new CGlobalFrameTable(this);
            m_pGFT.OnPropertyChange += new PropertyChangeEventHandler(PropertyChange);
            // Create the Transmit List
            m_pXmtFrames = new CXmtFrameList(m_pGFT);
            // Create the Receive List
            m_pRcvFrames = new CRcvFrameList(this, m_pGFT);
            m_pRcvFrames.OnReceived += new FrameReceivedEventHandler(FrameReceived);
            // Initialize the LIN attributes
            m_hClient = 0;
            m_hHw = 0;
            m_nHwChannel = 0;
            m_lMask = 0xFFFFFFFFFFFFFFFF;
            m_HwMode = Peak.Lin.TLINHardwareMode.modSlave;
            m_wBaudrate = 19200;
            m_sHardwareName = "";
            m_LINMode = Peak.Lin.TLINHardwareMode.modNone;
            m_LINState = Peak.Lin.TLINHardwareState.hwsNotInitialized;
            m_wOverruns = 0;
            lvReceive.DoubleBufferedEx = true;
            lvTransmit.DoubleBufferedEx = true;
            lvGFT.DoubleBufferedEx = true;
            // Fill the ListView 'lvGFT' with the Frame Definition objects
            // from the Global Frame Table.
            FillListViews(false);

            // Initialize the SplitContainer Panels with the minimum size.
            SplitCtrClient.Panel1MinSize = 250;
            SplitCtrClient.Panel2MinSize = 200;
            SplitCtrClient.SplitterDistance = 500;
            SplitCtrGFT.Panel1MinSize = 150;
            SplitCtrGFT.Panel2MinSize = 150;
            SplitCtrGFT.SplitterDistance = 168;

            // Assign the online help topic for the main window.
            OnlineHelp.SetShowHelp(this, true);
            OnlineHelp.SetHelpNavigator(this, HelpNavigator.Topic);
            OnlineHelp.SetHelpKeyword(this, "UIReference\\MainWindow.htm");
            // Assign the online help topic for the receiving list.
            OnlineHelp.SetShowHelp(lvReceive, true);
            OnlineHelp.SetHelpNavigator(lvReceive, HelpNavigator.Topic);
            OnlineHelp.SetHelpKeyword(lvReceive, "UIReference\\RcvList.htm");
            // Assign the online help topic for the transmitting or publisher list.
            OnlineHelp.SetShowHelp(lvTransmit, true);
            OnlineHelp.SetHelpNavigator(lvTransmit, HelpNavigator.Topic);
            OnlineHelp.SetHelpKeyword(lvTransmit, "UIReference\\XmtPubList.htm");
            // Assign the online help topic for the Global Frame Table.
            OnlineHelp.SetShowHelp(lvGFT, true);
            OnlineHelp.SetHelpNavigator(lvGFT, HelpNavigator.Topic);
            OnlineHelp.SetHelpKeyword(lvGFT, "UIReference\\GFTable.htm");
            // Assign the online help topic for the properties dropdown list.
            OnlineHelp.SetShowHelp(cbGFT, true);
            OnlineHelp.SetHelpNavigator(cbGFT, HelpNavigator.Topic);
            OnlineHelp.SetHelpKeyword(cbGFT, "UIReference\\Properties.htm");
            // Assign the online help topic for the properties control.
            OnlineHelp.SetShowHelp(pgGFTDef, true);
            OnlineHelp.SetHelpNavigator(pgGFTDef, HelpNavigator.Topic);
            OnlineHelp.SetHelpKeyword(pgGFTDef, "UIReference\\Properties.htm");

            // Register this application with LIN as client.
            m_LastLINErr = Peak.Lin.PLinApi.RegisterClient(Resources.SPLinClientName, Handle, out m_hClient);
        }

        private void MainWnd_Shown(object sender, EventArgs e)
        {
            PM_Tools_SelfReceive.Checked = m_pRcvFrames.SelfReceive;
            // Show the Connection Dialog
            DoLinConnect();
            // Refresh the menu, toolbar and popup commands
            // depends on the state of LIN connection.
            RefreshCommands(m_hHw != 0, false);
            // Refresh the text of StatusBar
            RefreshStatusBar(m_hHw != 0);
            // Assign the client handle to the Receive List
            // so it can recieve LIN frames.
            m_pRcvFrames.ClientHandle = m_hClient;
            // Start the possibility to receive LIN frames.
            m_pRcvFrames.StartWatching();
            // Start the MainTimer to get periodical LIN status.
            MainTimer.Enabled = true;
        }

        /// <summary>
        /// Occurs before the form is closed.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">A FormClosingEventArgs that contains the event data.</param>
        private void MainWnd_FormClosing(object sender, FormClosingEventArgs e)
        {
            // Free the Receive List.
            m_pRcvFrames = null;
            // Free the Transmit List.
            m_pXmtFrames = null;
            // Free the Global Frame Table.
            m_pGFT = null;
            // Stop the MainTimer
            MainTimer.Enabled = false;
            if (m_hClient != 0)
            {
                DoLinDisconnect();
                m_hHw = 0;
                // Unregister the application
                Peak.Lin.PLinApi.RemoveClient(m_hClient);
                m_hClient = 0;
            }
        }

        #region Customer Methods

        /// <summary>
        /// Fills the Listview 'Global Frame Table' with the data from CGloabalFrameTable,
        /// the Transmit ListView with the empty entry and the Receive ListView the
        /// empty entry if the no entry located.
        /// </summary>
        /// <param name="OnlyGFT">Boolean that indicate if only the Gloabal Frame Table ListView should be filled.</param>
        private void FillListViews(bool OnlyGFT)
        {
            ListViewItem lvi;
            // Lock the ListView.
            lvGFT.BeginUpdate();
            // Clear the ListView and ComboBox that
            // will show the Frame Definition of the
            // Global Frame Table.
            lvGFT.Items.Clear();
            cbGFT.Items.Clear();
            // Add every object, Frame Definition, from the
            // Global Frame Table into the ListView and ComboBox.
            for (int i = 0; i < m_pGFT.Count; i++)
            {
                lvi = lvGFT.Items.Add(Convert.ToString(m_pGFT[i].ID));
                lvi.SubItems.Add(Convert.ToString(m_pGFT[i].ProtectedID));
                lvi.SubItems.Add(Convert.ToString(m_pGFT[i].Direction));
                lvi.SubItems.Add(Convert.ToString(m_pGFT[i].Length));
                lvi.SubItems.Add(Convert.ToString(m_pGFT[i].ChecksumType));
                cbGFT.Items.Add(String.Format(Resources.SFrameEntryCombo, m_pGFT[i].ID));
            }
            // Unlock and update the ListView now.
            lvGFT.EndUpdate();
            if (OnlyGFT)
                return;
            if (lvReceive.Items.Count == 0)
            {
                // If no frame was received in the Receive List
                // show an empty entry in the Receive ListView.
                lvi = lvReceive.Items.Add(Resources.SEmpty);
                lvi.ToolTipText = Resources.SToolTipRcvNoEntry;
                lvi.Tag = null;
            }
            // Show an empty entry in the Transmit ListView.
            lvi = lvTransmit.Items.Add(Resources.SEmpty);
            lvi.ToolTipText = Resources.SToolTipXmtNoEntry;
            lvi.Tag = null;
        }

        /// <summary>
        /// Refeshs the 'Enabled' property of some menu, toolbar
        /// and popup menu entries in conjunction with the state of
        /// hardware connection and the possiblity to edit.
        /// </summary>
        /// <param name="HwConnect">State of hardware connection. Set to true if a connection to a LIN-hardware exists. Otherwise this parameter must set to false.</param>
        /// <param name="TransmitEdit">State for add, edit or delete some elements from the transmit list. Set to true if elements can add, edit or delete. Otherwise this parameter must set to false.</param>
        private void RefreshCommands(bool HwConnect, bool TransmitEdit)
        {
            // Refresh the menu, toolbar and popup commands
            // with the parameter HwConnect and TransmitEdit.
            PM_Transmit_New.Enabled = HwConnect;
            PM_Transmit_Edit.Enabled = TransmitEdit;
            PM_Transmit_Delete.Enabled = TransmitEdit;
            PM_Transmit_ClearAll.Enabled = HwConnect;
            MM_Transmit_New.Enabled = HwConnect;
            MM_Transmit_Edit.Enabled = TransmitEdit;
            MM_Transmit_Delete.Enabled = TransmitEdit;
            MM_Transmit_ClearAll.Enabled = HwConnect;
            MM_Transmit_Send.Enabled = HwConnect && TransmitEdit;
            MM_LIN_Disconnect.Enabled = HwConnect;
            MM_LIN_Reset.Enabled = HwConnect;
            MM_LIN_GoToSleep.Enabled = HwConnect;
            MM_LIN_WakeUp.Enabled = HwConnect;
            TB_Transmit_AddNew.Enabled = HwConnect;
            TB_LIN_Reset.Enabled = HwConnect;
            MM_LIN_GoToSleep.Visible = m_HwMode == Peak.Lin.TLINHardwareMode.modMaster;
            MM_LIN_WakeUp.Visible = m_HwMode == Peak.Lin.TLINHardwareMode.modSlave;
        }

        /// <summary>
        /// Refreshs the StatusBar text in conjunction with the state
        /// of hardware connection.
        /// </summary>
        /// <param name="HwConnect">State of hardware connection. Set to true if a connection to a LIN-hardware exists. Otherwise this parameter must set to false.</param>
        private void RefreshStatusBar(bool HwConnect)
        {
            if (HwConnect)
            {
                String sBaudText;
                String sHwChannel;

                // Get the baudrate of the connected hardware.
                sBaudText = Convert.ToString(m_wBaudrate);
                // Get the channel of the connected hardware.
                sHwChannel = Convert.ToString(m_nHwChannel);
                // Hardware connection is valid
                // so refresh the text in the StatusBar
                // with the connection information.
                SB_LIN_Connect.ForeColor = SystemColors.ControlText;
                SB_LIN_Channel.ForeColor = SystemColors.ControlText;
                SB_LIN_Connect.Text = String.Format(Resources.SLINConnectTo, m_sHardwareName, sBaudText);
                SB_LIN_Channel.Text = String.Format(Resources.SLINConnectToCh, sHwChannel);
                SB_LIN_Overruns.Text = String.Format(Resources.SLINStatusOverruns, m_wOverruns);
            }
            else
            {
                // The hardware handle is invalid
                // so refresh the StatusBar text.
                SB_LIN_Connect.ForeColor = Color.Red;
                SB_LIN_Connect.Text = Resources.SLINNotConnected;
                SB_LIN_Channel.Text = "";
                SB_LIN_Mode.Text = "";
                SB_LIN_Status.Text = "";
                SB_LIN_Overruns.Text = "";
            }
        }

        /// <summary>
        /// Reads all values from the frame table of the hardware
        /// and assign it to the CGlobalFrameTable. Also refresh
        /// the Global Frame Table ListView with that values.
        /// </summary>
        private void ReadFrameTableFromHw()
        {
            int i, lnCnt;
            Peak.Lin.TLINFrameEntry lFrameEntry;
            Peak.Lin.TLINError lErr;
            ulong llMask;

            // Create a LIN frame entry object
            lFrameEntry = new Peak.Lin.TLINFrameEntry();
            // Get the count of Frame Definition from the
            // Global Frame Table.
            lnCnt = m_pGFT.Count;
            // Initialize the member attribut for the
            // client mask with 0.
            m_lMask = 0;
            // Each Frame Definition
            for (i = 0; i < lnCnt; i++)
            {
                // Before a frame entry can be read from the
                // hardware it have to set the Frame-ID of the
                // entry who want to get.
                lFrameEntry.FrameId = m_pGFT[i].IdAsByte;
                // Read the information of frame entry from the hardware.
                lErr = Peak.Lin.PLinApi.GetFrameEntry(m_hHw, ref lFrameEntry);
                // Check the result value of the LinApi function call.
                if (lErr == Peak.Lin.TLINError.errOK)
                {
                    // LinApi function call successfull.
                    // Copy the frame entry information to the Frame Definition.
                    m_pGFT[i].m_nLength = Convert.ToInt32(lFrameEntry.Length);
                    m_pGFT[i].m_bDirection = (byte)lFrameEntry.Direction;
                    m_pGFT[i].m_nChecksumType = (int)lFrameEntry.ChecksumType;
                    if (m_pGFT[i].Direction != EDirection.Disabled)
                    {
                        // If the direction not disabled than set the
                        // bit in the client filter mask.
                        llMask = ((ulong)1 << i) & FRAME_FILTER_MASK;
                        m_lMask = m_lMask | llMask;
                    }
                }
            }
            // If the Client and Hardware handles are valid.
            if ((m_hClient != 0) && (m_hHw != 0))
                // Set the client filter.
                Peak.Lin.PLinApi.SetClientFilter(m_hClient, m_hHw, m_lMask);
            // Refresh the Gloabl Frame Table in the ListView.
            FillListViews(true);
        }

        /// <summary>
        /// Get all values from the CGlobalFrameTable and write it
        /// to the frame table of the hardware.
        /// </summary>
        private void WriteFrameTableToHw()
        {
            int i, lnCnt;
            Peak.Lin.TLINFrameEntry lFrameEntry;

            // If the Client and Hardware handles
            // are invalid than return.
            if ((m_hClient == 0) || (m_hHw == 0))
                return;

            // Create a LIN frame entry object
            lFrameEntry = new Peak.Lin.TLINFrameEntry();
            // Get the count of Frame Definition from the
            // Global Frame Table.
            lnCnt = m_pGFT.Count;
            // Each Frame Definition
            for (i = 0; i < lnCnt; i++)
            {
                // Copy the Frame Definition information to the frame entry.
                lFrameEntry.FrameId = m_pGFT[i].IdAsByte;
                lFrameEntry.Length = (byte)m_pGFT[i].Length;
                lFrameEntry.ChecksumType = (Peak.Lin.TLINChecksumType)m_pGFT[i].ChecksumType;
                lFrameEntry.Direction = (Peak.Lin.TLINDirection)m_pGFT[i].Direction;
                lFrameEntry.Flags = Peak.Lin.PLinApi.FRAME_FLAG_RESPONSE_ENABLE;
                // Write the information of frame entry to the hardware.
                Peak.Lin.PLinApi.SetFrameEntry(m_hClient, m_hHw, ref lFrameEntry);
            }
        }

        /// <summary>
        /// The function is used to refresh all values of the CFrameDefintion
        /// objects in the CGlobalFrameTable to default values (LIN 1.2).
        /// </summary>
        private void DoResetGFT()
        {
            // Check the connection
            if ((m_hClient != 0) && (m_hHw != 0))
            {
                // If the client is connected to a hardware.
                Peak.Lin.TLINError lErr;
                
                // Reset the configuration on the hardware.
                lErr = Peak.Lin.PLinApi.ResetHardwareConfig(m_hClient, m_hHw);
                if (lErr == Peak.Lin.TLINError.errOK)
                {
                    // If the reset successfull than
                    // initialize hardware and read
                    // the frame entries from the hardware
                    // and refresh the Global Frame Table in
                    // the ListView.
                    Peak.Lin.PLinApi.InitializeHardware(m_hClient, m_hHw, m_HwMode, m_wBaudrate);
                    ReadFrameTableFromHw();
                }
            }
            else
            {
                // Set all Frame Definition object values inside the
                // Global Frame Table to the default values. (LIN 1.2)
                m_pGFT.Reset();

                // Refresh the Gloabl Frame Table in the ListView.
                FillListViews(true);
            }
        }

        /// <summary>
        /// Clear the Receive Frames in the CRcvFrameList and clear all elements
        /// in the ListView 'lvReceive'.
        /// </summary>
        private void DoResetRcvFrames()
        {
            ListViewItem lvi;

            // Clear the received LIN frames
            // from the Receive List.
            m_pRcvFrames.Clear();
            // Clear the Receive ListView.
            lvReceive.BeginUpdate();
            lvReceive.Items.Clear();
            // Add an empty entry in the Receive ListView.
            lvi = lvReceive.Items.Add(Resources.SEmpty);
            lvi.ToolTipText = Resources.SToolTipRcvNoEntry;
            lvi.Tag = null;
            lvReceive.EndUpdate();
        }

        /// <summary>
        /// Clear the Transmit Frames in the CXmtFrameList and clear all elements
        /// in the ListView 'lvTransmit'.
        /// </summary>
        private void DoResetXmtFrames()
        {
            ListViewItem lvi;

            // Clear the transmit LIN frames
            // from the Transmit List
            m_pXmtFrames.Clear();
            // Clear the Transmit ListView
            lvTransmit.BeginUpdate();
            lvTransmit.Items.Clear();
            // Add an ampty entry in the Transmit ListView.
            lvi = lvTransmit.Items.Add(Resources.SEmpty);
            lvi.ToolTipText = Resources.SToolTipXmtNoEntry;
            lvi.Tag = null;
            lvTransmit.EndUpdate();
        }

        /// <summary>
        /// Clear the Transmit Frames Counter in the CXmtFrameList and update all elements
        /// in the ListView 'lvTransmit'.
        /// </summary>
        private void DoResetXmtFramesCounter()
        {
            //ListViewItem lvi;

            // Refresh the all transmit LIN frames
            // Counter from the Transmit List
            m_pXmtFrames.ClearCounter();
            // Clear the Transmit ListView
            lvTransmit.BeginUpdate();
            // Refresh the text of the counter
            // in the ListView.
            foreach (ListViewItem lvi in lvTransmit.Items)
            {
                if (lvi.Tag != null)
                    lvi.SubItems[3].Text = "0";
            }
            lvTransmit.EndUpdate();
        }

        /// <summary>
        /// Initialize the CGlobalFrameTable with default values and clear the
        /// CRcvFrameList and CXmtFrameList.
        /// </summary>
        private void DoFileNew()
        {
            // Reset the Global Frame Table
            // and the assigned Window Controls.
            DoResetGFT();
            // Reset the Received Frames
            // and the assigned Window Controls.
            DoResetRcvFrames();
            // Reset the Transmit Frames
            // and the assigned Window Controls.
            DoResetXmtFrames();
            m_sFileName = "";
            Text = Resources.SAppCaption;
        }

        /// <summary>
        /// Shows an open dialog to select a XML file with data to read into the
        /// CGlobalFrameTable and CXmtFrameList.
        /// </summary>
        private void DoOpenFile()
        {
            ListViewItem lvi;
            String S;

            // Initialize the Open Dialog
            OpenDialog1.Title = Resources.SOpenCaption1;
            S = Environment.GetFolderPath(Environment.SpecialFolder.MyDocuments);
            OpenDialog1.InitialDirectory = S;
            OpenDialog1.Filter = Resources.SOpenFilter1;
            OpenDialog1.FileName = "";
            // Show the Open Dialog
            if (OpenDialog1.ShowDialog(this) == DialogResult.OK)
            {
                // If the user want to open a file.
                S = OpenDialog1.FileName;
                // Read the selected file. 
                if (ReadFile(S) == true)
                {
                    // If the reading file returned successfull
                    // fill the readed values into the Window Controls.
                    FillListViews(m_pXmtFrames.Count != 0);

                    lvTransmit.BeginUpdate();
                    lvTransmit.Items.Clear();
                    for (int i = 0; i < m_pXmtFrames.Count; i++)
                    {
                        lvi = new ListViewItem(m_pXmtFrames.Item[i].IdAsString);
                        lvi.SubItems.Add(m_pXmtFrames.Item[i].Length.ToString());
                        lvi.SubItems.Add(m_pXmtFrames.Item[i].DataAsString);
                        lvi.SubItems.Add(m_pXmtFrames.Item[i].CountAsString);
                        lvi.SubItems.Add(m_pXmtFrames.Item[i].DirectionAsString);
                        lvi.SubItems.Add(m_pXmtFrames.Item[i].ChecksumType.ToString());
                        lvi.Tag = m_pXmtFrames.Item[i];
                        lvTransmit.Items.Add(lvi);
                    }
                    lvTransmit.EndUpdate();
                    m_sFileName = S;
                    // Set the application title including the filename.
                    Text = String.Format(Resources.SAppCaptionFileName, S);
                }
            }
        }

        /// <summary>
        /// Save the data from CGlobalFrameTable and CXmtFrameList into a XML file.
        /// </summary>
        private void DoSaveFile()
        {
            // If the filename was set before
            // means that the user had opened
            // a file previos time and the
            // file can be write directly.
            // Otherwise it is a new file and
            // the command SaveAs have to call.
            if (m_sFileName == "")
                DoSaveAsFile();
            else
                WriteFile(m_sFileName);
        }

        /// <summary>
        /// Shows an save dialog to select a XML file in which the data from the
        /// CGlobalFrameTable and CXmtFrameList should be written and write the
        /// data into the selected file.
        /// </summary>
        private void DoSaveAsFile()
        {
            String S;

            // Initialize the Save Dialog.
            S = Environment.GetFolderPath(Environment.SpecialFolder.MyDocuments);
            SaveDialog1.InitialDirectory = S;
            SaveDialog1.FileName = m_sFileName;
            // Show the Save Dialog.
            if (SaveDialog1.ShowDialog(this) == DialogResult.OK)
            {
                // If a file was selected to
                // save the data from the
                // Global Frame Table and Transmit List.
                S = SaveDialog1.FileName;
                if (WriteFile(S) == true)
                {
                    // If the writing progress was successfull
                    // refresh the title bar of the application
                    // with the new filename.
                    m_sFileName = S;
                    Text = String.Format(Resources.SAppCaptionFileName, S);
                }
            }
        }

        /// <summary>
        /// Reads data from a XML file and initialize the CGlobalFrameTable
        /// and CXmtFrameList with the readed data.
        /// </summary>
        /// <param name="FileName">XML file to read for initialization.</param>
        /// <returns>
        /// Returns true if the XML file readed successfull.
        /// Otherwise false.
        /// </returns>
        private bool ReadFile(String FileName)
        {
            XmlReader doc;
            XmlReaderSettings setting = new XmlReaderSettings();
            String S;
            Version lThisVer, lFileVer;
            Peak.Lin.TLINHardwareMode lMode;

            // Initialize the settings to read a XML file.
            setting.ConformanceLevel = ConformanceLevel.Fragment;
            setting.IgnoreWhitespace = true;
            setting.IgnoreComments = true;

            // Create a XML file reader object.
            doc = XmlReader.Create(FileName, setting);
            // Starts the reading of data from a XML file.
            if (doc.Read() == true)
            {
                // Get the version of this application.
                lThisVer = System.Reflection.Assembly.GetExecutingAssembly().GetName().Version;
                if (doc.HasAttributes)
                {
                    // If the XML file has an attibut 'Version'
                    // read this information.
                    S = doc.GetAttribute("Version");
                    lFileVer = new Version(S);
                    // If the XML file has an attibut 'HardwareMode'
                    // read this information.
                    S = doc.GetAttribute("HardwareMode");
                    if (S != null)
                        lMode = (Peak.Lin.TLINHardwareMode)Convert.ToInt32(S);
                    else
                        lMode = Peak.Lin.TLINHardwareMode.modNone;
                }
                else
                {
                    // Assign the current version
                    lFileVer = lThisVer;
                    lMode = Peak.Lin.TLINHardwareMode.modNone;
                }

                // Compare the version of the XML file with
                // the version of this application
                if (lThisVer.CompareTo(lFileVer) < 0)
                {
                    // The Version of the file is newer than the application.
                    return false;
                }
                else
                {
                    // The version of the XML file is older or
                    // has the same version that this application.
                    
                    // If the any hardware mode was load from file
                    // and any connection exists already.
                    if ((lMode != Peak.Lin.TLINHardwareMode.modNone)
                    && (m_hHw != 0))
                    {
                        // Create the message box text.
                        S = String.Format(Resources.SDifferentHwModes, System.IO.Path.GetFileName(FileName));
                        // Until the user set the hardware mode to the
                        // hardware mode which was loading.
                        while (lMode != m_HwMode)
                        {
                            DialogResult dRes;
                            dRes = MessageBox.Show(this, S, Resources.SAppCaptionQuestion, MessageBoxButtons.YesNoCancel, MessageBoxIcon.Question);
                            // Check the confirmation
                            if (dRes == DialogResult.Cancel)
                                // Exit this function and do not loading file.
                                return false;
                            else if (dRes == DialogResult.Yes)
                            {
                                // Try to change hardware mode before loading the file.
                                DoLinConnect();
                                // Refresh the text of StatusBar
                                RefreshStatusBar(m_hHw != 0);
                            }
                            else if (dRes == DialogResult.No)
                                // Irgnore the message and continue the loading.
                                break;
                        }
                    }

                    // Read data into Global Frame Table
                    m_pGFT.ReadData(doc);
                    // Refresh the frame table of the hardware
                    // with the table that readed from file.
                    WriteFrameTableToHw();
                    // Clear the transmit frames from the
                    // Transmit List before read the data
                    // from the XML file.
                    m_pXmtFrames.Clear();
                    // Read data into Transmit List
                    m_pXmtFrames.ReadData(doc);
                }
                // Reading finished successfull.
                return true;
            }
            // Reading NOT finished successfull.
            return false;
        }

        /// <summary>
        /// Writes data from the CGlobalFrameTable and CXmtFrameList into
        /// the XML file.
        /// </summary>
        /// <param name="FileName">XML file to write with data.</param>
        /// <returns>
        /// Returns true if the XML file wrote successfull.
        /// Otherwise false.
        /// </returns>
        private bool WriteFile(String FileName)
        {
            XmlWriter doc;
            XmlWriterSettings setting = new XmlWriterSettings();
            Version lVer;

            // Initialize the settings to write into XML file.
            setting.Indent = true;
            setting.OmitXmlDeclaration = true;
            setting.NewLineOnAttributes = true;

            // Get the version of this application.
            lVer = System.Reflection.Assembly.GetExecutingAssembly().GetName().Version;
            try
            {
                // Create a XML file reader object.
                doc = XmlWriter.Create(FileName, setting);
                // Starts of write data
                doc.WriteStartElement("Project");
                // Write the application version as attribut.
                doc.WriteAttributeString("Version", lVer.ToString());
                // Write the hardware mode as attribut.
                doc.WriteAttributeString("HardwareMode", ((int)m_HwMode).ToString());

                // Write data from Global Frame Table
                m_pGFT.WriteData(ref doc);
                // Write data from Transmit List
                m_pXmtFrames.WriteData(ref doc);

                // End of writing data
                doc.WriteEndElement();
                doc.Flush();
                // Writing data finished successfull.
                return true;
            }
            catch
            {
                // If an error occurs
                // writing data NOT finished successfull.
                return false;
            }
        }

        /// <summary>
        /// Shows the connection dialog to select a hardware, baudrate and
        /// client filter and connect to the hardware with the setting values
        /// from the connection dialog.
        /// </summary>
        private bool DoLinConnect()
        {
            byte i;
            int nCurIdx = -1;
            ushort lwHw, lwBaud;
            ushort[] lwHwHandles;
            ushort lwBuffSize, lwCount;
            int lnHwType, lnDevNo, lnChannel, lnMode, lnCurrBaud;
            ulong llMask;
            string sErrText;
            bool fRet;
            ConnectDlg lConnectDlg;
            Peak.Lin.TLINError lLINErr;
            Peak.Lin.TLINHardwareMode lHwMode;
            ListViewItem lvi;

            fRet = false;
            // Create the Connection Dialog
            // Define and initialize local variables and objects.
            lConnectDlg = new ConnectDlg();
            sErrText = new string('\0', 255);

            // Get the buffer length needed...
            lwCount = 0;
            lLINErr = Peak.Lin.PLinApi.GetAvailableHardware(new ushort[0], 0, out lwCount);
            if (lwCount == 0)   // use default value if either no hw is connected or an unexpected error occured
                lwCount = 16;
            lwHwHandles = new ushort[lwCount];
            lwBuffSize = Convert.ToUInt16(lwCount * sizeof(ushort));

            // Get all available LIN hardware.
            lLINErr = Peak.Lin.PLinApi.GetAvailableHardware(lwHwHandles, lwBuffSize, out lwCount);
            if (lLINErr == Peak.Lin.TLINError.errOK)
            {
                // If no error occurs
                if (lwCount == 0)
                {
                    // No LIN hardware was found.
                    // Show an empty entry
                    lwHw = 0;
                    lvi = new ListViewItem(Resources.SHardwareNothingFound);
                    lvi.Tag = lwHw;
                    lConnectDlg.lvHardware.Items.Add(lvi);
                }
                // For each founded LIN hardware
                for (i = 0; i < lwCount; i++)
                {
                    // Get the handle of the hardware.
                    lwHw = lwHwHandles[i];
                    // Read the type of the hardware with the handle lwHw.
                    Peak.Lin.PLinApi.GetHardwareParam(lwHw, Peak.Lin.TLINHardwareParam.hwpType, out lnHwType, 0);
                    // Read the device number of the hardware with the handle lwHw.
                    Peak.Lin.PLinApi.GetHardwareParam(lwHw, Peak.Lin.TLINHardwareParam.hwpDeviceNumber, out lnDevNo, 0);
                    // Read the channel number of the hardware with the handle lwHw.
                    Peak.Lin.PLinApi.GetHardwareParam(lwHw, Peak.Lin.TLINHardwareParam.hwpChannelNumber, out lnChannel, 0);
                    // Read the mode of the hardware with the handle lwHw (Master, Slave or None).
                    Peak.Lin.PLinApi.GetHardwareParam(lwHw, Peak.Lin.TLINHardwareParam.hwpMode, out lnMode, 0);

                    // Create a ListView entry
                    // If the hardware type is a knowing hardware
                    // show the name of that in the label of the entry.
                    if (lnHwType == Peak.Lin.PLinApi.LIN_HW_TYPE_USB_PRO)
                        lvi = new ListViewItem(Resources.SHardwareTypeLinUsbPro);
                    else if (lnHwType == Peak.Lin.PLinApi.LIN_HW_TYPE_USB_PRO_FD)
                        lvi = new ListViewItem(Resources.SHardwareTypeLinUsbProFd);
                    else if (lnHwType == Peak.Lin.PLinApi.LIN_HW_TYPE_PLIN_USB)
                        lvi = new ListViewItem(Resources.SHardwareTypePLIN);
                    else
                        // Show as unknown hardware
                        lvi = new ListViewItem(Resources.SHardwareTypeUnkown);

                    // Assign the handle of the LIN hardware with a
                    // property of the ListView entry for using later.
                    lvi.Tag = lwHw;
                    // Add the device number to the ListView entry.
                    lvi.SubItems.Add(Convert.ToString(lnDevNo));
                    // Add the channel number to the ListView entry.
                    lvi.SubItems.Add(Convert.ToString(lnChannel));
                    // Add the mode of the hardware to the ListeView entry.
                    if ((Peak.Lin.TLINHardwareMode)lnMode == Peak.Lin.TLINHardwareMode.modSlave)
                        lvi.SubItems.Add(Resources.SHardwareModeSlave);
                    else if ((Peak.Lin.TLINHardwareMode)lnMode == Peak.Lin.TLINHardwareMode.modMaster)
                        lvi.SubItems.Add(Resources.SHardwareModeMaster);
                    else
                        lvi.SubItems.Add(Resources.SHardwareModeNone);

                    // Add the ListView entry to the ListView.
                    lConnectDlg.lvHardware.Items.Add(lvi);
                    if (lwHw == m_hHw)
                        nCurIdx = i;
                }
                if (nCurIdx >= 0)
                    lConnectDlg.lvHardware.Items[nCurIdx].Selected = true;
                // Assign the Hardware mode to the control.
                if (m_HwMode == Peak.Lin.TLINHardwareMode.modMaster)
                    lConnectDlg.cbMode.SelectedIndex = 1;
                else
                    lConnectDlg.cbMode.SelectedIndex = 0;
                // Assign the Baudrate to the Control.
                lConnectDlg.cbBaudrate.Text = m_wBaudrate.ToString();
            }
            // Assign the online help topic for the connect window.
            OnlineHelp.SetShowHelp(lConnectDlg, true);
            OnlineHelp.SetHelpNavigator(lConnectDlg, HelpNavigator.Topic);
            OnlineHelp.SetHelpKeyword(lConnectDlg, "UIReference\\DlgConnect.htm");
            // Show the Connection Dialog
            if (lConnectDlg.ShowDialog(this) == DialogResult.OK)
            {
                if (m_hHw != 0)
                {
                    // If a connection to hardware already exits
                    // disconnect this connection at first.
                    if (DoLinDisconnect() == false)  
                        return fRet;
                }
                // Check if any hardware was selected on the dialog.
                if (lConnectDlg.lvHardware.SelectedItems.Count > 0)
                {
                    // Get the selected entry from the ListView.
                    lvi = lConnectDlg.lvHardware.SelectedItems[0];
                    // Read the assigned hardware handle from the entry.
                    lwHw = (ushort)(lvi.Tag);
                    // Check handle
                    if (lwHw != 0)
                    {
                        // The local hardware handle is valid.
                        // Get the current mode of the hardware
                        Peak.Lin.PLinApi.GetHardwareParam(lwHw, Peak.Lin.TLINHardwareParam.hwpMode, out lnMode, 0);
                        // Get the current baudrate of the hardware
                        Peak.Lin.PLinApi.GetHardwareParam(lwHw, Peak.Lin.TLINHardwareParam.hwpBaudrate, out lnCurrBaud, 0);
                        // Try to connect the application client to the
                        // hardware with the local handle.
                        m_LastLINErr = Peak.Lin.PLinApi.ConnectClient(m_hClient, lwHw);
                        if (m_LastLINErr == Peak.Lin.TLINError.errOK)
                        {
                            // If the connection successfull
                            // assign the local handle to the
                            // member handle.
                            m_hHw = lwHw;
                            // Get the selected hardware mode.
                            if (lConnectDlg.cbMode.SelectedIndex == 1)
                                lHwMode = Peak.Lin.TLINHardwareMode.modMaster;
                            else
                                lHwMode = Peak.Lin.TLINHardwareMode.modSlave;
                            // Get the selected baudrate
                            lwBaud = Convert.ToUInt16(lConnectDlg.cbBaudrate.Text);
                            // Get the selected hardware channel
                            m_nHwChannel = Convert.ToUInt16(lvi.SubItems[2].Text);
                            if (((Peak.Lin.TLINHardwareMode)lnMode == Peak.Lin.TLINHardwareMode.modNone)
                                || (Convert.ToUInt16(lnCurrBaud) != lwBaud))
                            {
                                // Only if the current hardware is not initialize
                                // try to Intialize the hardware with mode and baudrate
                                m_LastLINErr = Peak.Lin.PLinApi.InitializeHardware(m_hClient,
                                    m_hHw, lHwMode, lwBaud);
                            }
                            if (m_LastLINErr == Peak.Lin.TLINError.errOK)
                            {
                                // Assign the Hardware Mode to member attribut
                                m_HwMode = lHwMode;
                                // Assign the baudrate index to member attribut
                                m_wBaudrate = lwBaud;
                                // Get the the client filter mask.
                                llMask = m_lMask;
                                // Set the client filter with the mask.
                                m_LastLINErr = Peak.Lin.PLinApi.SetClientFilter(m_hClient, m_hHw, llMask);
                                if (m_LastLINErr == Peak.Lin.TLINError.errOK)
                                    // Assign the filter mask to attibut.
                                    m_lMask = llMask;
                                // Set the name of connected hardware.
                                m_sHardwareName = lvi.Text;
                                // Read the frame table from the connected hardware.
                                ReadFrameTableFromHw();
                                // Reset the LIN mode and LIN state to default. 
                                m_LINMode = Peak.Lin.TLINHardwareMode.modNone;
                                m_LINState = Peak.Lin.TLINHardwareState.hwsNotInitialized;
                                // Reset the last LIN error code to default.
                                m_LastLINErr = Peak.Lin.TLINError.errOK;
                                fRet = true;
                            }
                            else
                            {
                                // An error occurs while initialize hardware.
                                // Set the member variable to default.
                                m_hHw = 0;
                                m_nHwChannel = 0;
                                m_sHardwareName = "";
                                fRet = false;
                            }
                        }
                        else
                        {
                            // The local hardware handle is invalid
                            // and/or an error occurs while connecting
                            // hardware with client.
                            // Set the member variable to default.
                            m_hHw = 0;
                            m_nHwChannel = 0;
                            m_sHardwareName = "";
                            fRet = false;
                        }
                        
                        // Check if any LIN error code was received.
                        if (m_LastLINErr != Peak.Lin.TLINError.errOK)
                        {
                            StringBuilder strTemp = new StringBuilder(255);
                            // If any error are occured
                            // display the error text in a message box.
                            // 0x00 = Neutral
                            // 0x07 = Language German
                            // 0x09 = Language English
                            Peak.Lin.PLinApi.GetErrorText(m_LastLINErr, 0x09, strTemp, 255);
                            MessageBox.Show(strTemp.ToString(), Resources.SAppCaptionError);
                            fRet = false;
                        }
                    }
                    else // Should never occur
                        m_hHw = 0; // But if it occurs, set handle to default
                }
                else // Should never occur
                    m_hHw = 0; // But if it occurs, set handle to default

            }
            // Check the mode of the Hardware (Master or Slave)
            // and refresh the picture on the left side of the application.
            if (m_HwMode == Peak.Lin.TLINHardwareMode.modMaster)
            {
                pbTransmit.Image = Resources.XmtText;
                PM_Tools_SelfReceive.Enabled = true;
                PM_Tools_SelfReceive.Visible = true;
                m_pRcvFrames.SelfReceive = true;
                MM_Transmit.Text = Resources.SMMTransmit;
            }
            else if (m_HwMode == Peak.Lin.TLINHardwareMode.modSlave)
            {
                pbTransmit.Image = Resources.PubText;
                PM_Tools_SelfReceive.Enabled = false;
                PM_Tools_SelfReceive.Visible = false;
                m_pRcvFrames.SelfReceive = true;
                MM_Transmit.Text = Resources.SMMPublish;
            }
            return fRet;
        }

        /// <summary>
        /// Disconnects an existing connection to a LIN hardware and returns
        /// true if disconnect finished succesfull or no connection exists.
        /// Returns false if the current connection can not be disconnected.
        /// </summary>
        /// <returns>
        /// Returns true if the function finished successfull. Otherwise
        /// returns false.
        /// </returns>
        private bool DoLinDisconnect()
        {
            // If the application was registered with LIN as client.
            if (m_hHw != 0)
            {
                // The client was connected to a LIN hardware.
                // Before disconnect from the hardware check
                // the connected clients and determine if the
                // hardware configuration have to reset or not.

                // Initialize the locale variables.
                bool lfOtherClient = false;
                bool lfOwnClient = false;
                byte[] lhClients = new byte[255];

                // Get the connected clients from the LIN hardware.
                m_LastLINErr = Peak.Lin.PLinApi.GetHardwareParam(m_hHw, Peak.Lin.TLINHardwareParam.hwpConnectedClients, lhClients, 255);
                if (m_LastLINErr == Peak.Lin.TLINError.errOK)
                {
                    // No errors !
                    // Check all client handles.
                    for (int i = 0; i < lhClients.Length; i++)
                    {
                        // If client handle is invalid
                        if (lhClients[i] == 0)
                            continue;
                        // Set the boolean to true if the handle isn't the
                        // handle of this application.
                        // Even the boolean is set to true it can never
                        // set to false.
                        lfOtherClient = lfOtherClient || (lhClients[i] != m_hClient);
                        // Set the boolean to true if the handle is the
                        // handle of this application.
                        // Even the boolean is set to true it can never
                        // set to false.
                        lfOwnClient = lfOwnClient || (lhClients[i] == m_hClient);
                    }
                }
                // If another application is also connected to
                // the LIN hardware do not reset the configuration.
                if (lfOtherClient == false)
                {
                    // No other application connected !
                    // Reset the configuration of the LIN hardware.
                    Peak.Lin.PLinApi.ResetHardwareConfig(m_hClient, m_hHw);
                }
                // If this application is connected to the hardware
                // than disconnect the client. Otherwise not.
                if (lfOwnClient == true)
                {
                    // Disconnect if the application was connected to a LIN hardware.
                    m_LastLINErr = Peak.Lin.PLinApi.DisconnectClient(m_hClient, m_hHw);
                    if (m_LastLINErr == Peak.Lin.TLINError.errOK)
                    {
                        m_hHw = 0;
                        // Refresh the StatusBar text.
                        RefreshStatusBar(false);
                        return true;
                    }
                    else
                    {
                        StringBuilder sErrText = new StringBuilder('\0', 255);
                        // Error while disconnecting from hardware.
                        Peak.Lin.PLinApi.GetErrorText(m_LastLINErr, 0x09, sErrText, 255);
                        MessageBox.Show(sErrText.ToString(), Resources.SAppCaptionError);
                        return false;
                    }
                }
                else
                    return true;
            }
            else
                return true;
        }

        /// <summary>
        /// Resets the Client, Hardware, Receive List and the Receive ListView.
        /// </summary>
        private void DoLinReset()
        {
            // Reset all Receive Objects and Controls.
            DoResetRcvFrames();
            // Reset all Counter of the all Transmit Frames.
            DoResetXmtFramesCounter();
            // If the client is registered.
            if (m_hClient > 0)
                m_LastLINErr = Peak.Lin.PLinApi.ResetClient(m_hClient);
            // If the client is registered and
            // is connected to a hardware.
            if ((m_hClient > 0) && (m_hHw > 0))
                m_LastLINErr = Peak.Lin.PLinApi.ResetHardware(m_hClient, m_hHw);
        }

        /// <summary>
        /// Deletes the selected entry from the Receive ListView and
        /// it's assigned LIN frame from the Receive List.
        /// </summary>
        private void DoReceiveDelete()
        {
            int nIdx;
            ListViewItem lvi;

            // If one entry is selected in the Receive List.
            if (lvReceive.SelectedItems.Count > 0)
            {
                // Get the selected ListView entry.
                lvi = lvReceive.SelectedItems[0];
                // Check the assigned LIN frame.
                if (lvi.Tag == null)
                    // Return if the assigned value is invalid.
                    return;

                // Get the index of the selection.
                nIdx = lvReceive.SelectedIndices[0];
                if (nIdx >= 0)
                {
                    // The index is valid.
                    // Lock the ListView for update.
                    lvReceive.BeginUpdate();
                    // Remove the LIN frame that is assigned
                    // with the selected ListView entry from
                    // the Receive List.
                    m_pRcvFrames.RemoveFrame(nIdx);
                    // Remove the selected entry from
                    // the ListView.
                    lvReceive.Items.RemoveAt(nIdx);
                    // Unlock and update the ListView now.
                    lvReceive.EndUpdate();
                }
            }
        }

        /// <summary>
        /// Opens the Frame Dialog to add a new LIN frame to the Transmit List
        /// creates an entry for the Transmit ListView and assign the
        /// added LIN frame with the entry in the ListView.
        /// </summary>
        private void DoTransmitNew()
        {
            FrameDlg lFrameDlg;
            ListViewItem lvi;
            CCustomFrame lcFrame;
            int nIdx = -1;
            bool lfMustCreate = true;

            // Create the Frame Dialog to add a new LIN frame
            // to the Transmit List.
            lFrameDlg = new FrameDlg(m_pGFT, m_pXmtFrames, m_HwMode);
            lFrameDlg.Text = Resources.SFramesDlgNewCaption;
            // Set the index to -1 because a new LIN frame
            // should be create.
            nIdx = -1;
            // Assign the online help topic for the frame window.
            OnlineHelp.SetShowHelp(lFrameDlg, true);
            OnlineHelp.SetHelpNavigator(lFrameDlg, HelpNavigator.Topic);
            OnlineHelp.SetHelpKeyword(lFrameDlg, "UIReference\\DlgEditFrame.htm");
            // Show the Frame Dialog
            if (lFrameDlg.ShowDialogEx(this, ref nIdx, true) == DialogResult.OK)
            {
                // The returned index is the position of
                // the created LIN frame in the Transmit List.
                if (nIdx >= 0)
                {
                    // The index is valid.
                    // Lock the Transmit ListView for update.
                    lvTransmit.BeginUpdate();
                    // Get the created LIN frame
                    lcFrame = m_pXmtFrames.Item[nIdx];
                    if (lvTransmit.Items.Count > 0)
                    {
                        // If there are at least one entry
                        // in the ListView first check if
                        // it is the empty entry or not.
                        // If it is the empty entry with
                        // caption '<empty>' than replace
                        // the values of this entry.
                        // Otherwise a new entry have to create.
                        lvi = lvTransmit.Items[0];
                        // Check the assigned LIN frame.
                        // No assigned frame (NULL) means empty entry.
                        // Assigned frame (!= NULL) means valid entry.
                        lfMustCreate = lvi.Tag != null;
                    }
                    else
                        lvi = null;

                    // A new entry have to create
                    if (lfMustCreate)
                    {
                        // Create a new ListView entry
                        // and assign the values.
                        lvi = new ListViewItem(lcFrame.IdAsString);
                    }
                    else if (lvi != null)
                    {
                        // The entry exits already but the value
                        // can replace by the values from the LIN frame.
                        // It was an emty entry.
                        // Replace the caption text of entry
                        lvi.Text = lcFrame.IdAsString;
                    }
                    
                    // In both cases it have to add all other
                    // values as sub items.
                    lvi.SubItems.Add(lcFrame.Length.ToString());
                    lvi.SubItems.Add(lcFrame.DataAsString);
                    lvi.SubItems.Add(lcFrame.CountAsString);
                    lvi.SubItems.Add(lcFrame.DirectionAsString);
                    lvi.SubItems.Add(lcFrame.ChecksumType.ToString());
                    // Assign the created LIN frame for using later.
                    lvi.Tag = lcFrame;

                    // Only if a new entry have to create
                    if (lfMustCreate)
                        // Add the created entry to the ListView.
                        lvTransmit.Items.Add(lvi);

                    // Unlock and Update the ListView now.
                    lvTransmit.EndUpdate();
                    // After add a new entry update the Transmit or Publish commands.
                    RefreshCommands(m_hHw != 0, lvTransmit.SelectedIndices.Count > 0);
                }
            }
        }

        /// <summary>
        /// Opens the Frame Dialog to edit the LIN frame from the Transmit List
        /// that is assigned with the selected entry from the Transmit ListView.
        /// </summary>
        private void DoTransmitEdit()
        {
            FrameDlg lFrameDlg;
            ListViewItem lvi;
            CCustomFrame lcFrame;
            int nIdx;

            // If one entry is selected in the Transmit List
            if (lvTransmit.SelectedItems.Count > 0)
            {
                // Get the selected ListView entry.
                lvi = lvTransmit.SelectedItems[0];
                // Check the assigned LIN frame.
                if (lvi.Tag == null)
                    // Return if the assigned value is invalid.
                    return;
                
                // Get the index of the selection.
                nIdx = lvTransmit.SelectedIndices[0];
                if (nIdx >= 0)
                {
                    // The index is valid.
                    // Create the Frame Dialog to edit an
                    // existing LIN frame.
                    lFrameDlg = new FrameDlg(m_pGFT, m_pXmtFrames, m_HwMode);
                    lFrameDlg.Text = Resources.SFramesDlgEditCaption;
                    // Assign the online help topic for the frame window.
                    OnlineHelp.SetShowHelp(lFrameDlg, true);
                    OnlineHelp.SetHelpNavigator(lFrameDlg, HelpNavigator.Topic);
                    OnlineHelp.SetHelpKeyword(lFrameDlg, "UIReference\\DlgEditFrame.htm");
                    // Show the Frame Dialog with the index.
                    if (lFrameDlg.ShowDialogEx(this, ref nIdx, false) == DialogResult.OK)
                    {
                        // If the user changed some values in the LIN frame
                        // get the LIN frame from the Transmit List
                        lcFrame = m_pXmtFrames.Item[nIdx];
                        // Refresh the Transmit ListView with the values.
                        lvTransmit.BeginUpdate();
                        lvi = lvTransmit.Items[nIdx];
                        lvi.Text = lcFrame.IdAsString;
                        lvi.SubItems[1].Text = lcFrame.Length.ToString();
                        lvi.SubItems[2].Text = lcFrame.DataAsString;
                        lvi.SubItems[3].Text = lcFrame.CountAsString;
                        lvi.SubItems[4].Text = lcFrame.DirectionAsString;
                        lvi.SubItems[5].Text = lcFrame.ChecksumType.ToString();
                        lvTransmit.EndUpdate();
                    }
                }
            }
        }

        /// <summary>
        /// Deletes the selected entry from the Transmit ListView and
        /// it's assigned LIN frame from the Transmit List.
        /// </summary>
        private void DoTransmitDelete()
        {
            int nIdx;
            ListViewItem lvi;

            // If one entry is selected in the Transmit List.
            if (lvTransmit.SelectedItems.Count > 0)
            {
                // Get the selected ListView entry.
                lvi = lvTransmit.SelectedItems[0];
                // Check the assigned LIN frame.
                if (lvi.Tag == null)
                    // Return if the assigned value is invalid.
                    return;

                // Get the index of the selection.
                nIdx = lvTransmit.SelectedIndices[0];
                if (nIdx >= 0)
                {
                    // The index is valid.
                    // Lock the ListView for update.
                    lvTransmit.BeginUpdate();
                    // Remove the LIN frame that is assigned
                    // with the selected ListView entry from
                    // the Transmit List.
                    m_pXmtFrames.RemoveFrame(nIdx);
                    // Remove the selected entry from
                    // the ListView.
                    lvTransmit.Items.RemoveAt(nIdx);
                    // If no entry is defined in the ListView
                    // after deleting the selected entry.
                    if (lvTransmit.Items.Count == 0)
                    {
                        // Show an empty entry in the Transmit ListView.
                        lvi = lvTransmit.Items.Add(Resources.SEmpty);
                        lvi.ToolTipText = Resources.SToolTipXmtNoEntry;
                        lvi.Tag = null;
                    }
                    // Unlock and update the ListView now.
                    lvTransmit.EndUpdate();
                }
            }
        }

        /// <summary>
        /// Deletes all entries from the Transmit ListView
        /// and removes the assigned LIN frames from the
        /// Transmit List.
        /// </summary>
        private void DoTransmitClear()
        {
            ListViewItem lvi;

            // Lock the the ListView for update.
            lvTransmit.BeginUpdate();
            // While the count of the Transmit ListView
            // does not zero remove the last entry
            // from the ListView.
            while (lvTransmit.Items.Count > 0)
            {
                // Remove the assigned LIN frame
                // from the Transmit List
                m_pXmtFrames.RemoveFrame(lvTransmit.Items.Count - 1);
                // Remove the last entry from the
                // Transmit ListView.
                lvTransmit.Items.RemoveAt(lvTransmit.Items.Count - 1);
            }
            // If no entry is defined in the ListView
            // so show an empty entry in the Transmit ListView.
            lvi = lvTransmit.Items.Add(Resources.SEmpty);
            lvi.ToolTipText = Resources.SToolTipXmtNoEntry;
            lvi.Tag = null;
            // Unlock and update the ListView now.
            lvTransmit.EndUpdate();
        }

        /// <summary>
        /// Sends or updates the LIN frame that is assigned with the
        /// selected entry from the Transmit ListView.
        /// </summary>
        private void DoTransmitSend()
        {
            int nIdx;
            Peak.Lin.TLINMsg lMsg;
            CCustomFrame lCF;
            //String sErrText;
            ListViewItem lvi;

            // If one entry is selected in the Transmit List.
            if (lvTransmit.SelectedItems.Count > 0)
            {
                // Get the selected ListView entry.
                lvi = lvTransmit.SelectedItems[0];
                // Check the assigned LIN frame.
                if (lvi.Tag == null)
                    // Return if the assigned value is invalid.
                    return;
                // Get the index of the selection.
                nIdx = lvTransmit.SelectedIndices[0];
                // Get the Transmit Frame on the selected position.
                lCF = m_pXmtFrames.Item[nIdx];
                if (lCF == null)
                    return;
                // Check if the hardware is initialize as master
                if(m_HwMode == Peak.Lin.TLINHardwareMode.modMaster)
                {
                    // Only if it is a master first create
                    // a LIN frame message and copy the data.
                    lMsg = new Peak.Lin.TLINMsg();
                    lMsg.FrameId = (byte)lCF.ProtectedIdAsInt;
                    lMsg.Data = lCF.Data;
                    lMsg.ChecksumType = (Peak.Lin.TLINChecksumType)lCF.ChecksumType;
                    lMsg.Direction = (Peak.Lin.TLINDirection)lCF.Direction;
                    lMsg.Length = Convert.ToByte(lCF.Length);
                    // Calculate the checksum contained with the
                    // checksum type that set some line before.
                    Peak.Lin.PLinApi.CalculateChecksum(ref lMsg);
                    // Try to send the LIN frame message with LIN_Write.
                    m_LastLINErr = Peak.Lin.PLinApi.Write(m_hClient, m_hHw, ref lMsg);
                }
                else
                {
                    // If the hardare is initialize as slave
                    // only update the data in the LIN frame.
                    m_LastLINErr = Peak.Lin.PLinApi.UpdateByteArray(
                                                m_hClient,
                                                m_hHw,
                                                (byte)lCF.IdAsInt,
                                                (byte)0,
                                                (byte)lCF.Length,
                                                lCF.Data);
                }
                // Increment the counter of the Transmit Frame object.
                lCF.Count++;
                // Refresh the counter in the Transmit ListView.
                lvTransmit.SelectedItems[0].SubItems[3].Text = lCF.CountAsString;
            }

        }

        /// <summary>
        /// Checks the last LIN error code which was returned from
        /// a LIN function call and returns if an error code exists.
        /// </summary>
        /// <returns>Returns true if any error code exists. Otherwise returns false.</returns>
        private bool DoCheckLastError()
        {
            // If the last LIN error code was not O.K.
            if (m_LastLINErr != Peak.Lin.TLINError.errOK)
            {
                // Check if the hardware handle is valid.
                if (m_LastLINErr == Peak.Lin.TLINError.errIllegalHardware)
                {
                    // The hardware handle is invalid.
                    // Set the class member handle to invalid handle
                    // so that other function does not use this handle
                    // again with LIN functions.
                    m_hHw = 0;
                    // Refresh the menu, toolbar and popup commands
                    // depends on the state of LIN connection.
                    RefreshCommands(false, false);
                    // Refresh the StatusBar text.
                    RefreshStatusBar(false);
                }
                return true;
            }
            else
                return false;
        }

        /// <summary>
        /// Checks the states of the connected LIN hardware and displays
        /// changes of the states in the StatusBar.
        /// </summary>
        private void DoLINCheckStates()
        {
            Peak.Lin.TLINHardwareStatus lStatus;

            if (m_hHw != 0)
            {
                // The Timer is used to get Status from the LinApi and
                // the LIN-Bus that is connected with the application.
                // When at least one Status (State or Mode) of the LIN-Bus is
                // changed the new state will be display in the StatusBar
                // of the application.

                if (m_fAutomatic == true)
                    return;
                m_fAutomatic = true;

                // Get the current Status from the LIN-Bus.
                m_LastLINErr = Peak.Lin.PLinApi.GetStatus(m_hHw, out lStatus);

                // Check the last Error.
                if (DoCheckLastError() == true)
                {
                    // With the last error code 
                    // this function have to exit.
                    m_fAutomatic = false;
                    return;
                }

                // If the State has been changed.
                if (m_LINState != lStatus.Status)
                {
                    // Inspect the new State and
                    // display the text in the
                    // StatusBar panel 'SB_LIN_Status'.
                    switch (lStatus.Status)
                    {
                        case Peak.Lin.TLINHardwareState.hwsNotInitialized:
                            SB_LIN_Status.ForeColor = Color.Red;
                            SB_LIN_Status.Text = Resources.SLINStatusNotInit;
                            break;
                        case Peak.Lin.TLINHardwareState.hwsAutobaudrate:
                            SB_LIN_Status.ForeColor = SystemColors.ControlText;
                            SB_LIN_Status.Text = Resources.SLINStatusBaudDetect;
                            break;
                        case Peak.Lin.TLINHardwareState.hwsActive:
                            SB_LIN_Status.ForeColor = Color.Green;
                            SB_LIN_Status.Text = Resources.SLINStatusActive;
                            break;
                        case Peak.Lin.TLINHardwareState.hwsSleep:
                            SB_LIN_Status.ForeColor = SystemColors.ControlText;
                            SB_LIN_Status.Text = Resources.SLINStatusSleep;
                            break;
                        case Peak.Lin.TLINHardwareState.hwsShortGround:
                            SB_LIN_Status.ForeColor = Color.Red;
                            SB_LIN_Status.Text = Resources.SLINStatusShortGround;
                            break;
                        case Peak.Lin.TLINHardwareState.hwsVBatMissing:
                            SB_LIN_Status.ForeColor = SystemColors.ControlText;
                            SB_LIN_Status.Text = Resources.SLINStatusVBatMissing;
                            break;
                    }
                    // Keep the new state.
                    m_LINState = lStatus.Status;
                }
                // If the Mode has been changed.
                if (m_LINMode != lStatus.Mode)
                {
                    // Inspect the new Mode and
                    // dislay the text in the
                    // StatusBar panel 'SB_LIN_Mode'.
                    switch (lStatus.Mode)
                    {
                        case Peak.Lin.TLINHardwareMode.modNone:
                            SB_LIN_Mode.ForeColor = Color.Red;
                            SB_LIN_Mode.Text = Resources.SLINModeNotInit;
                            break;
                        case Peak.Lin.TLINHardwareMode.modMaster:
                            SB_LIN_Mode.ForeColor = SystemColors.ControlText;
                            SB_LIN_Mode.Text = Resources.SLINModeMaster;
                            break;
                        case Peak.Lin.TLINHardwareMode.modSlave:
                            SB_LIN_Mode.ForeColor = SystemColors.ControlText;
                            SB_LIN_Mode.Text = Resources.SLINModeSlave;
                            break;
                    }
                    // Keep the new Mode.
                    m_LINMode = lStatus.Mode;
                }
                if (m_wOverruns != lStatus.ReceiveBufferOverrun)
                {
                    m_wOverruns = lStatus.ReceiveBufferOverrun;
                    if (m_wOverruns > 0)
                        SB_LIN_Overruns.ForeColor = Color.Red;
                    else
                        SB_LIN_Overruns.ForeColor = SystemColors.ControlText;
                    SB_LIN_Overruns.Text = String.Format(Resources.SLINStatusOverruns, m_wOverruns);
                }
                m_fAutomatic = false;
            }
        }

        #endregion

        #region User Events

        #region ListView 'lvTransmit'

        /// <summary>
        /// Occurs when the SelectedIndices collection changes.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">An EventArgs that contains no event data.</param>
        private void lvTransmit_SelectedIndexChanged(object sender, EventArgs e)
        {
            int nCnt;
            bool lfEditLine;
            ListViewItem lvi;
            
            // Get the count of selected elements in
            // the ListView 'lvTransmit'.
            nCnt = lvTransmit.SelectedItems.Count;
            if (nCnt > 0)
            {
                // If at least one element in the
                // ListView is selected it is allow
                // to edit this element if it is
                // assigned with a CCustomFrame object
                // from CXmtFrameList.
                lvi = lvTransmit.SelectedItems[0];
                lfEditLine = lvi.Tag != null;
            }
            else
                // Else do NOT allow to edit.
                lfEditLine = false;

            // Refesh all commands.
            RefreshCommands(m_hHw > 0, lfEditLine);
        }

        /// <summary>
        /// Occurs when the control is double clicked by the mouse.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">A MouseEventArgs that contains the event data.</param>
        private void lvTransmit_MouseDoubleClick(object sender, MouseEventArgs e)
        {
            // If a double click on a element in
            // the ListView 'lvTransmit' occurs
            // than start to edit the selected
            // LIN-Frame with open the Frame Dialog.
            DoTransmitEdit();
        }

        /// <summary>
        /// Occurs when a key is pressed while the control has focus.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">A KeyPressEventArgs that contains the event data.</param>
        private void lvTransmit_KeyPress(object sender, KeyPressEventArgs e)
        {
            // If the space key was pressed
            if (e.KeyChar == (char)Keys.Space)
                // Send the selected LIN-Frame.
                DoTransmitSend();
            // If the enter key was pressed
            if (e.KeyChar == (char)Keys.Enter)
                // Edit the selected LIN-Frame
                // with the Frame Dialog.
                DoTransmitEdit();
        }

        #endregion

        #region ListView 'lbGFT'
        
        /// <summary>
        /// Occurs when the SelectedIndices collection changes. The selection
        /// will be assigned to the Property-Control 'pgGFTDef'.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">An EventArgs that contains no event data.</param>
        private void lvGFT_SelectedIndexChanged(object sender, EventArgs e)
        {
            int nCnt, nIdx, i;

            if (!m_fAutomatic)
            {
                m_fAutomatic = true;
                // Get the count of selected elements
                // in the ListView 'lvGFT'.
                // Each element is assigned with an
                // element in the CGlobalFrameTable.
                nCnt = lvGFT.SelectedItems.Count;
                if (nCnt == 1)
                {
                    // If only one element is selected
                    // it can be assigned directly to the
                    // Property-Control.
                    nIdx = lvGFT.SelectedItems[0].Index;
                    cbGFT.SelectedIndex = nIdx;
                    // Assign the CFrameDefinition element
                    // from the CGlobalFrameTable.
                    pgGFTDef.SelectedObject = m_pGFT[nIdx];
                }
                else if (nCnt > 1)
                {
                    // If more than one element are selected
                    // in the ListView means that the user
                    // want to change at least one property of
                    // all elements to a new value with
                    // the Property-Control.
                    // So get first the selected elements and the
                    // assigned objects from the CGlobalFrameTable
                    // into a local Array of elements.
                    CFrameDefinition[] lDefFrames = new CFrameDefinition[nCnt];
                    for (i = 0; i < nCnt; i++)
                    {
                        nIdx = lvGFT.SelectedItems[i].Index;
                        lDefFrames[i] = m_pGFT[nIdx];
                    }
                    cbGFT.SelectedIndex = -1;
                    cbGFT.Text = String.Format(Resources.SFrameEntriesCombo, nCnt);
                    // Assign the local array of elements
                    // from the CGlobalFrameTable.
                    pgGFTDef.SelectedObjects = lDefFrames;
                }
                m_fAutomatic = false;
            }
        }

        #endregion

        #region cbGFT

        /// <summary>
        /// Occurs when the Selection of the CombBox 'cbGFT' is changed.
        /// The selection will be assigned to the Property-Control 'pgGFTDef'.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void cbGFT_Change(object sender, EventArgs e)
        {
            int nIdx;

            // Check the current edit mode.
            if (!m_fAutomatic)
            {
                // No other control event is the caller
                // of this event.
                // Set the edit mode.
                m_fAutomatic = true;
                // Get the selected index from the ComboBox.
                nIdx = cbGFT.SelectedIndex;
                if (nIdx >= 0)
                {
                    // If the index is valid
                    // get the Frame Definition.
                    pgGFTDef.SelectedObject = m_pGFT[nIdx];
                    // Clear the selection in the
                    // Global Frame Table ListView.
                    lvGFT.SelectedItems.Clear();
                    // Select the entry of the selected
                    // Frame Definition in the
                    // Global Frame Table ListView.
                    lvGFT.Items[nIdx].Selected = true;
                }
                // Set the edit mode to false
                // to allow other control event handler
                // to start editing.
                m_fAutomatic = false;
            }
        }


        /// <summary>
        /// Occurs when the user try to enter text with the keyboard.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void cbGFT_KeyDown(object sender, KeyEventArgs e)
        {
            e.Handled = true;
            // Do not allow to enter text with the keyboard.
            e.SuppressKeyPress = true;
        }

        #endregion

        #region PropertyGrid 'pgGFTDef'

        /// <summary>
        /// Occurs before a property value of at least one CFrameDefinition object changes.
        /// </summary>
        /// <param name="s">The source of the event</param>
        /// <param name="e">A PropertyValueChangeEventArg that contains the event data.</param>
        private void PropertyChange(object s, PropertyChangeEventArg e)
        {
            CFrameDefinition lFD;
            Peak.Lin.TLINFrameEntry lFrameEntry;
            Peak.Lin.TLINError lErr;
            ulong lMask;
            
            // Try to get the sender as CFrameDefinition
            lFD = (CFrameDefinition)s;
            // The sender of this event is the CFrameDefinition that
            // property should be change by User.
            if (lFD != null)
            {
                // Check if the definition is used already from
                // a CCustomFrame object in the Transmit List.
                if (m_pXmtFrames.AnyUsedBy(lFD))
                {
                    // If the property 'Direction' of the
                    // CFrameDefinition will be change in
                    // the Slave Mode.
                    if((e.Type == EProperty.Direction)&&(m_HwMode == Peak.Lin.TLINHardwareMode.modSlave))
                        // If the new value for the property 'Direction'
                        // should NOT be 'Publisher' than DO NOT allowed.
                        e.Allowed = ((EDirection)e.Value == EDirection.Publisher);
                    // If the property 'Direction' of the
                    // CFrameDefinition will be change in
                    // the Master Mode.
                    if ((e.Type == EProperty.Direction) && (m_HwMode == Peak.Lin.TLINHardwareMode.modMaster))
                        // If the new value for the property 'Direction'
                        // should be 'Disbaled' than DO NOT allowed.
                        e.Allowed = ((EDirection)e.Value != EDirection.Disabled);
                    // If DO NOT allow than return.
                    if (!e.Allowed)
                        return;
                }

                // If data length will be set check the value.
                if (e.Type == EProperty.Length)
                    // Only a value between 0 and 8 are valid.
                    e.Allowed = (e.Value >= 0) && (e.Value <= 8);
                // If DO NOT allow than return.
                if (!e.Allowed)
                    return;

                // Only if the hardware was initialize as Slave
                // set the direction of the LIN-Frame.
                // By the Master the direction will be used with
                // the LIN_Write and do not have to set here.
                if (m_HwMode == Peak.Lin.TLINHardwareMode.modSlave)
                {
                    // Temporary set Allowed to false is to easy
                    // return only later.
                    e.Allowed = false;

                    // Create a Frame Entry object to get
                    // and set the direction.
                    lFrameEntry = new Peak.Lin.TLINFrameEntry();
                    // Set the Frame-ID of the Frame to get and set.
                    // The ID have to set before get the entry.
                    lFrameEntry.FrameId = lFD.IdAsByte;
                    // Get the Frame Entry with the Frame-ID from
                    // the Hardware via the LinApi.
                    lErr = Peak.Lin.PLinApi.GetFrameEntry(m_hHw, ref lFrameEntry);
                    // If an error occurs do not allow to change
                    // the property and return.
                    // The Allowed parameter was set some lines before.
                    if (lErr != Peak.Lin.TLINError.errOK)
                        return;
                    // Switch between the different kind of property types.
                    switch (e.Type)
                    {
                        // Direction property should be set
                        case EProperty.Direction:
                            lFrameEntry.Direction = (Peak.Lin.TLINDirection)e.Value;
                            break;
                        // Length property should be set
                        case EProperty.Length:
                            lFrameEntry.Length = Convert.ToByte(e.Value);
                            break;
                        // ChecksumType property should be set
                        case EProperty.ChecksumType:
                            lFrameEntry.ChecksumType = (Peak.Lin.TLINChecksumType)e.Value;
                            break;
                    }
                    lFrameEntry.Flags = Peak.Lin.PLinApi.FRAME_FLAG_RESPONSE_ENABLE;
                    lErr = Peak.Lin.PLinApi.SetFrameEntry(m_hClient, m_hHw, ref lFrameEntry);
                    // If an error occurs do not allow to change
                    // the property and return.
                    // The Allowed parameter was set some lines before.
                    if (lErr != Peak.Lin.TLINError.errOK)
                        return;

                    // Temporary set Allowed to true for next check.
                    // The action was successfull on this line.
                    e.Allowed = true;
                }

                // If the property 'Direction' of one
                // CFrameDefinition will be change
                // here we need a special request to set
                // the client filter.
                if (e.Type == EProperty.Direction)
                {
                    // If the new value for the property 'Direction'
                    // should NOT be 'Disabled' check first if
                    // the CFrameDefinition is defined already with some
                    // other value then 'Disabled'.
                    if ((EDirection)e.Value != EDirection.Disabled)
                    {
                        if (lFD.Direction == EDirection.Disabled)
                        {
                            // If the old property of CFrameDefintion
                            // was set to 'Disabled' the new value
                            // means that the Frame-ID have to add to
                            // the client filter by the LinApi.
                            // Set the client filter.
                            // The Filter is a bit mask.
                            lMask = (ulong)1 << lFD.IdAsInt;
                            m_lMask = m_lMask | lMask;
                            lErr = Peak.Lin.PLinApi.SetClientFilter(m_hClient, m_hHw, m_lMask);
                            // Only allow to change the property if the Frame-ID
                            // was added successfull to the Filter.
                            e.Allowed = lErr == Peak.Lin.TLINError.errOK;
                        }
                    }
                    else
                    {
                        // If the value of direction should set on 'disable'
                        // Remove the Frame-ID from the client filter.
                        lMask = (ulong)1 << lFD.IdAsInt;
                        m_lMask = m_lMask & ~lMask;
                        lErr = Peak.Lin.PLinApi.SetClientFilter(m_hClient, m_hHw, m_lMask);
                        // Only allow to change the property if the Frame-ID
                        // was removed successfull from the Filter.
                        e.Allowed = lErr == Peak.Lin.TLINError.errOK;
                    }
                }
            }
        }

        /// <summary>
        /// Occurs when a property value of at least one CFrameDefinition object
        /// was changed by the Property-Control.
        /// </summary>
        /// <param name="s">The source of the event.</param>
        /// <param name="e">A PropertyValueChangedEventArgs that contains the event data.</param>
        private void pgGFTDef_PropertyValueChanged(object s, PropertyValueChangedEventArgs e)
        {
            int i, j, nCnt, nIdx;
            CFrameDefinition lFD;
            CCustomFrame lCF;
            ListViewItem lvi;
            int[] lnUpdated;
            int nUpdCnt;

            lFD = null;
            // Lock the ListView for update elements.
            lvGFT.BeginUpdate();
            // Get the selected count of CFrameDefinition.
            nCnt = pgGFTDef.SelectedObjects.GetLength(0);
            // For each selected CFrameDefinition in
            // the ListView there have to update they
            // property values.
            for (i = 0; i < nCnt; i++)
            {
                // Get the selected CFrameDefinition X from the
                // Property-Control that has changed the
                // property values.
                lFD = (CFrameDefinition)pgGFTDef.SelectedObjects[i];
                if (lFD != null)
                {
                    // Refresh the property values of all
                    // CCustomeFrame elements in the CXmtFrameList 
                    // that defined based on this CFrameDefinition
                    nUpdCnt = m_pXmtFrames.UpdateFrame(lFD, out lnUpdated);
                    // Refresh the Transmit ListView that contained
                    // all defined CCustomFrame objects from the
                    // CXmtFrameList.
                    lvTransmit.BeginUpdate();
                    for (j = 0; j < nUpdCnt; j++)
                    {
                        lvi = lvTransmit.Items[lnUpdated[j]];
                        lCF = (CCustomFrame)lvi.Tag;
                        lvi.SubItems[1].Text = lCF.Length.ToString();
                        lvi.SubItems[2].Text = lCF.DataAsString;
                        lvi.SubItems[4].Text = lCF.DirectionAsString;
                        lvi.SubItems[5].Text = lCF.ChecksumType.ToString();
                    }
                    lvTransmit.EndUpdate();
                    // Now get the position in the ListView of
                    // the CFrameDefinition to refresh the
                    // property value.
                    nIdx = m_pGFT.IndexOf(lFD);
                    if ((nIdx >= 0) && (nIdx < lvGFT.Items.Count))
                    {
                        // Get the item that is assign with the
                        // CFrameDefinition from ListView 'lvGFT'
                        // and refesh it.
                        lvi = lvGFT.Items[nIdx];
                        lvi.SubItems[1].Text = Convert.ToString(lFD.ProtectedID);
                        lvi.SubItems[2].Text = Convert.ToString(lFD.Direction);
                        lvi.SubItems[3].Text = Convert.ToString(lFD.Length);
                        lvi.SubItems[4].Text = Convert.ToString(lFD.ChecksumType);
                    }

                }
            }
            // Free the lock of ListView and
            // update all elements now.
            lvGFT.EndUpdate();
        }

        #endregion

        #endregion

        #region Menu Commands

        #region File

        /// <summary>
        /// Occurs when the ToolStripItem "File->New" is clicked.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">An EventArgs that contains no event data.</param>
        private void File_New(object sender, EventArgs e)
        {
            // Call the function to create a new file
            // for saving data from Transmit List and
            // Global Frame Table.
            DoFileNew();
        }

        /// <summary>
        /// Occurs when the ToolStripItem "File->Open" is clicked.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">An EventArgs that contains no event data.</param>
        private void File_Open(object sender, EventArgs e)
        {
            // Call the function to open a existing file
            // and read data for the Transmit List and
            // Global Frame Table from the file.
            DoOpenFile();
        }

        /// <summary>
        /// Occurs when the ToolStripItem "File->Save" is clicked.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">An EventArgs that contains no event data.</param>
        private void File_Save(object sender, EventArgs e)
        {
            // Call the function to save the Transmit List
            // and Global Frame Table into a file.
            DoSaveFile();
        }

        /// <summary>
        /// Occurs when the ToolStripItem "File->SaveAs" is clicked.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">An EventArgs that contains no event data.</param>
        private void File_SaveAs(object sender, EventArgs e)
        {
            // Call the function to open the Save Dialog
            // to locate the file to save the current data.
            DoSaveAsFile();
        }

        /// <summary>
        /// Occurs when the ToolStripItem "File->Exit" is clicked.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">An EventArgs that contains no event data.</param>
        private void File_Exit(object sender, EventArgs e)
        {
            // Exit the application.
            Close();
        }

        #endregion

        #region LIN

        /// <summary>
        /// Occurs when the ToolStripItem "LIN->Connect" is clicked.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">An EventArgs that contains no event data.</param>
        private void LIN_Connect(object sender, EventArgs e)
        {
            Peak.Lin.TLINHardwareMode LastMode;
            ushort LastHw;

            // Get the last connected hardware mode and
            // hardware handle for checking after connect.
            LastMode = m_HwMode;
            LastHw = m_hHw;
            // Call the function to show the Connection Dialog.
            DoLinConnect();
            // Refresh the menu, toolbar and popup commands.
            RefreshCommands(m_hHw != 0, false);
            // Refresh the text of StatusBar
            RefreshStatusBar(m_hHw != 0);
            // If the hardware mode was changed
            // reset the transmit list.
            if ((LastMode != Peak.Lin.TLINHardwareMode.modNone)
            && (m_HwMode != Peak.Lin.TLINHardwareMode.modNone)
            && (LastMode != m_HwMode))
                DoResetXmtFrames();
            // If the hardware handle was changed
            // reset the transmit list.
            if(LastHw != m_hHw)
                DoResetXmtFrames();
            DoResetRcvFrames();
        }

        /// <summary>
        /// Occurs when the ToolStripItem "LIN->Disconnect" is clicked.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">An EventArgs that contains no event data.</param>
        private void LIN_Disconnect(object sender, EventArgs e)
        {
            // Call the function to show the Connection Dialog.
            DoLinDisconnect();
            // Refresh the menu, toolbar and popup commands.
            RefreshCommands(m_hHw != 0, false);
            // Refresh the text of StatusBar
            RefreshStatusBar(m_hHw != 0);
        }

        /// <summary>
        /// Occurs when the ToolStripItem "LIN->Reset" is clicked.
        /// Resets the client and the hardware if there is a connection.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">An EventArgs that contains no event data.</param>
        private void LIN_Reset(object sender, EventArgs e)
        {
            // Call the function to reset the LIN connection.
            DoLinReset();
        }

        /// <summary>
        /// Occurs when the ToolStripItem "LIN->GoToSleep" is clicked.
        /// Send the Got to Sleep Command on the LIN bus.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">An EventArgs that contains no event data.</param>
        private void LIN_GoToSleep(object sender, EventArgs e)
        {
            // Check the handles.
            if ((m_hClient > 0) && (m_hHw > 0))
            {
                // Define a locale Message Structure.
                Peak.Lin.TLINMsg lMsg;
                // Initialize the locale Structure
                // with the value for the sleep command.
                lMsg = new Peak.Lin.TLINMsg();
                // Master Request
                lMsg.FrameId = 0x3C;
                // Get Protected ID
                Peak.Lin.PLinApi.GetPID(ref lMsg.FrameId);
                lMsg.Length = 8;
                lMsg.Direction = Peak.Lin.TLINDirection.dirPublisher;
                lMsg.ChecksumType = Peak.Lin.TLINChecksumType.cstClassic;
                lMsg.Data = new byte[8];
                lMsg.Data[0] = 0; // Must be zero for sleep command.
                lMsg.Data[1] = 0xFF;
                lMsg.Data[2] = 0xFF;
                lMsg.Data[3] = 0xFF;
                lMsg.Data[4] = 0xFF;
                lMsg.Data[5] = 0xFF;
                lMsg.Data[6] = 0xFF;
                lMsg.Data[7] = 0xFF;
                // Calculate the checksum in the
                // Message Structure.
                Peak.Lin.PLinApi.CalculateChecksum(ref lMsg);
                // Send the command on the LIN bus.
                Peak.Lin.PLinApi.Write(m_hClient, m_hHw, ref lMsg);
            }
        }

        /// <summary>
        /// Occurs when the ToolStripItem "LIN->WakeUp" is clicked.
        /// Sends a wake-up message if there is a connection to hardware.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">An EventArgs that contains no event data.</param>
        private void LIN_WakeUp(object sender, EventArgs e)
        {
            // Check the handles.
            if ((m_hClient > 0) && (m_hHw > 0))
            {
                // If the application is registered as LIN client
                // and connected to a LIN hardware.
                // Try to wake up the LIN Bus system.
                m_LastLINErr = Peak.Lin.PLinApi.XmtWakeUp(m_hClient, m_hHw);
                if (m_LastLINErr != Peak.Lin.TLINError.errOK)
                {
                    // If the wake up failed
                    // show the error message.
                    StringBuilder sErrText = new StringBuilder('\0', 255);
                    Peak.Lin.PLinApi.GetErrorText(m_LastLINErr, 0x09, sErrText, 255);
                    MessageBox.Show(sErrText.ToString(), Resources.SAppCaptionError);
                }
                else
                    DoLINCheckStates();
            }
        }

        #endregion

        #region Receive

        /// <summary>
        /// Occurs when the ToolStripItem "PM_Receive_Delete" is clicked.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">An EventArgs that contains no event data.</param>
        private void Receive_Delete(object sender, EventArgs e)
        {
            // Call the function to remove an existing Receive Frame.
            DoReceiveDelete();
        }

        #endregion

        #region Transmit

        /// <summary>
        /// Occurs when the ToolStripItem "Transmit->New" is clicked.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">An EventArgs that contains no event data.</param>
        private void Transmit_AddNew(object sender, EventArgs e)
        {
            // Call the function to create a new Transmit Frame.
            DoTransmitNew();
        }

        /// <summary>
        /// Occurs when the ToolStripItem "Transmit->Edit" is clicked.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">An EventArgs that contains no event data.</param>
        private void Transmit_Edit(object sender, EventArgs e)
        {
            // Call the function to edit an existing Transmit Frame.
            DoTransmitEdit();
        }

        /// <summary>
        /// Occurs when the ToolStripItem "Transmit->Delete" is clicked.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">An EventArgs that contains no event data.</param>
        private void Transmit_Delete(object sender, EventArgs e)
        {
            // Call the function to remove an existing Transmit Frame.
            DoTransmitDelete();
        }

        /// <summary>
        /// Occurs when the ToolStripItem "Transmit->ClearAll" is clicked.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">An EventArgs that contains no event data.</param>
        private void Transmit_ClearAll(object sender, EventArgs e)
        {
            // Call the function to remove all existing Transmit Frames.
            DoTransmitClear();
        }

        /// <summary>
        /// Occurs when the ToolStripItem "Transmit->Send" is clicked.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">An EventArgs that contains no event data.</param>
        private void Transmit_Send(object sender, EventArgs e)
        {
            // Call the function to send or update an existing Transmit Frame.
            DoTransmitSend();
        }

        #endregion

        #region Tools

        /// <summary>
        /// Occurs when the ToolStripItem "Tools->ResetGFT" is clicked.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">An EventArgs that contains no event data.</param>
        private void Tools_ResetGFT(object sender, EventArgs e)
        {
            // Call the function to set the Global Frame Table
            // to default values.
            DoResetGFT();
        }

        /// <summary>
        /// Orrurs when the ToolStripItem "Tools->SelfReceive" is clicked.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">An EventArgs that contains no event data.</param>
        private void Tools_SelfReceive(object sender, EventArgs e)
        {
            // Set the flag for self receive in the Receive List.
            // This only effects if the hardware is initialize as Master.
            // If the hardware is initialize as slave this flag has
            // no effect.
            m_pRcvFrames.SelfReceive = PM_Tools_SelfReceive.Checked;
        }

        #endregion

        #region Help

        /// <summary>
        /// Occurs when the ToolStripItem "Help->About" is clicked.
        /// Shows the About Dialog with the assembly information.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">An EventArgs that contains no event data.</param>
        private void Help_About(object sender, EventArgs e)
        {
            AboutDlg lAboutDlg;
            // Create the About Dialog.
            lAboutDlg = new AboutDlg();
            // Show the About Dialog.
            lAboutDlg.ShowDialog(this);
        }
        /// <summary>
        /// Occurs when the ToolStripItem "Help->PLIN-View Help" is clicked.
        /// Shows the Online Help within the contents help.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">An EventArgs that contains no event data.</param>
        private void Help_Help(object sender, EventArgs e)
        {
            // Open the Online Help which is assigned with the HelpProvider.
            Help.ShowHelp(this, OnlineHelp.HelpNamespace);
        }

        #endregion

        #endregion

        #region Non-User Events

        /// <summary>
        /// Occurs when the MainTimer interval has elapsed and it is enabled.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">An EventArgs that contains no event data.</param>
        private void OnMainTimerTick(object sender, EventArgs e)
        {
            DoLINCheckStates();
        }

        /// <summary>
        /// Occurs when a LIN-Frame was received bei the CRcvFrameList.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">An FrameReceivedEventArgs that contains the event data.</param>
        private void FrameReceived(object sender, FrameReceivedEventArgs e)
        {
            ListViewItem lvi;
            Boolean lfMustCreate = true;
            Int32 nIdx, i, j, nCnt;
            String S;

            nCnt = e.ReceivedCount;
            for (i = 0; i < nCnt; i++)
            {
                // This event handle routine is defined to notice
                // when a LIN-Frame was received by the CRcvFrameList.
                if (e.Frames[i].IsNewOne == true)
                {
                    // If the received LIN-Frame is a new one
                    // a new element must be create in the ListView
                    // to show the data of the LIN-Frame.
                    if (lvReceive.Items.Count > 0)
                    {
                        // Check for the empty entry '<empty>'
                        lvi = lvReceive.Items[0];
                        // Check the assigned LIN frame.
                        // No assigned frame (NULL) means empty entry.
                        // Assigned frame (!= NULL) means valid entry.
                        lfMustCreate = lvi.Tag != null;
                    }
                    else
                        lvi = null;

                    // A new entry have to create
                    if (lfMustCreate)
                        // Create a new ListView entry
                        // and assign the values.
                        lvi = new ListViewItem(e.Frames[i].RcvFrame.ID);
                    else if (lvi != null)
                        // The entry exits already but the value
                        // can replace by the values from the LIN frame.
                        // It was an emty entry.
                        lvi.Text = e.Frames[i].RcvFrame.ID;

                    lvi.SubItems.Add(e.Frames[i].RcvFrame.Length.ToString());
                    // Check if the Error "SlaveNotResponding" is received
                    if ((e.Frames[i].ErrorFlags & Peak.Lin.TLINMsgErrors.SlaveNotResponding) == 0)
                        S = e.Frames[i].RcvFrame.DataAsString;
                    else
                    {
                        // If the Error "SlaveNotResponding" received
                        // than show only "--" as data text.
                        S = "";
                        for (j = 0; j < e.Frames[i].RcvFrame.Length; j++)
                            S = S + "-- ";
                    }
                    lvi.SubItems.Add(S);
                    lvi.SubItems.Add(e.Frames[i].RcvFrame.PeriodAsString);
                    lvi.SubItems.Add(e.Frames[i].RcvFrame.CountAsString);
                    lvi.SubItems.Add(e.Frames[i].RcvFrame.DirectionAsString);
                    if ((int)e.Frames[i].ErrorFlags == 0)
                        lvi.SubItems.Add(Resources.SOk);
                    else
                        lvi.SubItems.Add(e.Frames[i].ErrorFlags.ToString());
                    lvi.SubItems.Add(e.Frames[i].RcvFrame.ChecksumType.ToString());
                    lvi.SubItems.Add(e.Frames[i].RcvFrame.ChecksumAsString);
                    lvi.Tag = e.Frames[i].RcvFrame;

                    // A new entry have to create
                    if (lfMustCreate)
                        lvReceive.Items.Add(lvi);
                }
                else
                {
                    // If the received LIN-Frame already exits in
                    // the ListView search for the element in the list.
                    nIdx = m_pRcvFrames.IndexOf(e.Frames[i].RcvFrame);
                    if (nIdx >= 0)
                    {
                        // Update the data of the element in
                        // the ListView with the received data.
                        lvi = lvReceive.Items[nIdx];
                        S = e.Frames[i].RcvFrame.Length.ToString();
                        if (lvi.SubItems[1].Text != S)
                            lvi.SubItems[1].Text = S;
                        if ((e.Frames[i].ErrorFlags & Peak.Lin.TLINMsgErrors.SlaveNotResponding) == 0)
                            S = e.Frames[i].RcvFrame.DataAsString;
                        else
                        {
                            S = "";
                            for (j = 0; j < e.Frames[i].RcvFrame.Length; j++)
                                S = S + "-- ";
                        }
                        if (lvi.SubItems[2].Text != S)
                            lvi.SubItems[2].Text = S;
                        S = e.Frames[i].RcvFrame.PeriodAsString;
                        if (lvi.SubItems[3].Text != S)
                            lvi.SubItems[3].Text = S;
                        S = e.Frames[i].RcvFrame.CountAsString;
                        if (lvi.SubItems[4].Text != S)
                            lvi.SubItems[4].Text = S;
                        S = e.Frames[i].RcvFrame.DirectionAsString;
                        if (lvi.SubItems[5].Text != S)
                            lvi.SubItems[5].Text = S;
                        if ((int)e.Frames[i].ErrorFlags == 0)
                            lvi.SubItems[6].Text = Resources.SOk;
                        else
                            lvi.SubItems[6].Text = e.Frames[i].ErrorFlags.ToString();
                        S = e.Frames[i].RcvFrame.ChecksumType.ToString();
                        if (lvi.SubItems[7].Text != S)
                            lvi.SubItems[7].Text = S;
                        S = e.Frames[i].RcvFrame.ChecksumAsString;
                        if (lvi.SubItems[8].Text != S)
                            lvi.SubItems[8].Text = S;
                    }
                }
            }
        }

        #endregion

        #endregion
    }
    public class peakListView : ListView
    {
        public bool DoubleBufferedEx
        {
            get
            {
                return base.DoubleBuffered;
            }
            set
            {
                base.DoubleBuffered = value;
            }
        }
    }
}
